; Copyright (c) 1998-2012 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	XAS_XRF_EXTRACT
;
;LAST CHANGED: ----------------------------------- 20-Feb-12
;
;PURPOSE:
;	This function extracts absorption stacks from a 4-d array (*.ssb)
; written by XAS_XRF_STACK and writes them as regular 3D stacks (*.ncb)
;
;CATEGORY:
;	STAND ALONE: utility
;
;CALLING SEQUENCE:
;	Result = XAS_XRF_EXTRACT( [,ch_low=ch_low, ch_hi=ch_hi. file=file, silent=silent])
;
;CALLED FROM AXIS:
;	Read->Images->AXIS
;
;INPUTS:
;	TMP - AXIS 4d array (x,y,E-XRF,E-XAS)
;
;KEYWORDS:
;	CH_LOW	lower limit of channels / energies to integrate
;	CH_HI	upper limit of channels / energies to integrate
;	FILE - filename to read
;	SILENT  flag to control entry to axis log
;
;COMMON BLOCKS:
; AXIS_COM	standard set of common blocks
; STACK_PROCESS_COM
; BSIF_COM
; volume_data, image_stack
; volume_data2, xas_xrf_stack
;
;;MODIFICATION HISTORY:
; (21-Apr-10 aph) first written
; (20-feb-12 aph) extend to extracting XRF spectra; save mean XRF spectrum

function xas_xrf_extract, file=file, ch_low=ch_low, ch_hi=ch_hi, silent=silent, $
             verbose=verbose, group=group, extra = extra
@axis_com
@stack_process_com
COMMON volume_data, image_stack
COMMON volume_data2, xas_xrf_stack
@bsif_com
on_error,2

; print, 'entry'
; help, xas_xrf_stack

;  ---------- read  *.ssb data
; if n_elements(xas_xrf_stack) EQ 0 then  begin
	xas_xrf_file =''
	xas_xrf_file = pickfile2(/READ, FILTER='*.ssb', /LPATH, DEFPATH=defpath)
	if strlen(xas_xrf_file) eq 0 then return, 0.	; bail if canceled
	ev = stack2_rb(file=xas_xrf_file)
	t=size(xas_xrf_stack)
	nx = t(1)
	ny = t(2)
	nch = t(3)
	nev = t(4)
; endif

; -------- generate sum of all XRF spectra if requested
xrf_e = findgen(nch)
xrf_mean = fltarr(nch)
for i=0,nch-1 do xrf_mean(i)=mean(xas_xrf_stack(*,*,i,*))
t=ax_name(xas_xrf_file)
dl = 'mean XRF spectrum ' + t(1)
s = {t:'1d', x:xrf_e, d:xrf_mean, dn:xrf_mean, xl:'XRF channel', dl: dl}
file =  pickfile2(/READ, prompt='name for Mean XRF spectrum', FILTER='*.txt', /LPATH, DEFPATH=defpath)
if strlen(file) ne 0 then t = spc_save(s, file=file)

;--------- select extraction of XRF or XAS signals
test2 = dialog_message(/question, 'extract XAS (Yes) or XRF (No) signals')
if Test2 EQ 'Yes' then begin

	; --------- ask user for lower and upper channel limits of XRF spectra
	text=' '
	ch_low= get_num(Val=ch_low,PROMPT='lower limit of XRF spectrum (Channel #; min = 0)', group=axis_ID)
	text = 'upper limit of XRF spectrum: Channel, max = ' + strtrim(string(nch)-1,2)
	ch_hi = get_num(Val=ch_hi, PROMPT=text, group=axis_ID)

	; -------- define size of 3d stack to extract
	im_size = size(xas_xrf_stack)
;	nx = im_size(1)
;	ny = im_size(2)
	npts = im_size(3)
	n_stacks = im_size(4)
	image_stack = fltarr(nx, ny, nev)

	; ------ extract from  xas_xrf_stack
	for i = ch_low, ch_hi do begin
		image_stack = image_stack + xas_xrf_stack(*,*,i,*)
	endfor

endif else begin

	; --------- ask user for lower and upper channel limits of AS spectra
	text=' '
	ch_low= get_num(Val=ch_low,PROMPT='lower limit of XRF spectrum (Channel #; min = 0)', group=axis_ID)
	text = 'upper limit of XRF spectrum: Channel, max = ' + strtrim(string(nev)-1,2)
	ch_hi = get_num(Val=ch_hi, PROMPT=text, group=axis_ID)

	; -------- define size of 3d stack to extract
	im_size = size(xas_xrf_stack)
	npts = im_size(3)
	n_stacks = im_size(4)
	image_stack = fltarr(nx, ny, nch)

	; ------ extract XRF spectra from  xas_xrf_stack over a set of incident photon energies
	ev = findgen(nch)
	for i = ch_low, ch_hi do begin
		image_stack = image_stack + xas_xrf_stack(*,*,*,i)
	endfor

endelse

; ------ write as binary stack
file = pickfile2(/write, TITLE = 'name of stack file', FILTER='*.ncb', /LPATH, DEFPATH=defpath)
stack_wb, file

s=1
return, s

end
