; Copyright (c) 1998-2011 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		WRITE_SPECTRUM
;
;LAST CHANGED: ----------------------------------- 	31-Jul-11
;
; PURPOSE:
;	This procedure writes a spectral netCDF file (format for old NSLS1 X1A STXM data)
;
; CATEGORY:
;	Image display.
;
; CALLING SEQUENCE:
;  WRITE_SPECTRUM,  nc_filename, ev_nm_angstroms, signal, $
;                    comments=comments, ev=ev, angstroms=angstroms, nm=nm

; INPUTS:
;	NC_FILENAME	flename
;	EV_NM_ANGSTROMS	parameter defining x-axis unit
; 	SIGNAL	y-axis data
;
; KEYWORDS:
;	COMMENTS - optional header comments
;	EV		- energy axis
;	ANGSTROMS - wavelength axis (in A)
;	NM - wavelength axis (in A)
;
; OUTPUTS: file is written
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;
; SIDE EFFECTS:
;	A window is created/destroyed.
;
; MODIFICATION HISTORY:
; (01--Jun-93 cjj) from Mark Rivers stuff;
; (31-Jul-11 aph) AXIS standard header added
;-


PRO write_spectrum, nc_filename, ev_nm_angstroms, signal, $
                    comments=comments, ev=ev, angstroms=angstroms, nm=nm

@bsif_com
on_error, 2

IF (n_params() eq 0) THEN BEGIN
    print,'write_spectrum,filename, ev_nm_angstroms, signal'
    print,'  options: [comments=,/ev,/nm,/angstroms]
    print,'  Input is assumed to be in angstroms unless /ev or /nm are set'
    print,'  Spectrum is converted to angstroms before writing'
    print,'    as a STXM NetCDF file.'
    return
ENDIF

 ; First save existing BSIF parameters
old_n_rows = n_rows
old_n_cols = n_cols
old_n_data = n_data
old_x_normal = x_normal
old_y_normal = y_normal
old_rotated = rotated
old_x_start = x_start
old_x_stop = x_stop
old_y_start = y_start
old_y_stop = y_stop
old_x_dist = x_dist
old_y_dist = y_dist
old_image_title = image_title
old_x_title = x_title
old_y_title = y_title
old_data_title = data_title
old_image_data = image_data

IF (NOT keyword_set(comments)) THEN comments=''

IF keyword_set(ev) THEN BEGIN
    ws_angstroms = 12398.52/float(ev_nm_angstroms)
ENDIF ELSE IF keyword_set(nm) THEN BEGIN
    ws_angstroms = 10.*ev_nm_angstroms
ENDIF ELSE BEGIN
    ws_angstroms = ev_nm_angstroms
ENDELSE

indices = sort(ws_angstroms)
ws_angstroms = ws_angstroms(indices)
ws_signal = signal(indices)
n_cols = n_elements(ws_angstroms)
abs_max_signal = max(abs(ws_signal))
n_rows = 1
n_data = 2
x_normal = 1
y_normal = 1
rotated = 0
x_dist = ws_angstroms
y_dist = ws_signal
x_start = ws_angstroms(0)
x_stop = ws_angstroms(n_cols-1)
y_start = min(ws_signal)
y_stop = max(ws_signal)

; Because
;   khz=sd.clock_frequency*image_data(*,*,0)/float(image_data(*,*,1))
; we want
;   image_data(*,*,0)=ws_signal*image_data(*,*,1)/sd.clock_frequency
; but we also want to make sure this is bounded by 32767 when we
; use 10000 for image_data(*,*,1)
my_clock_counts = 1.e4
my_clock_frequency = 1
data_multiplier = float(my_clock_frequency)/float(my_clock_counts)
WHILE (((abs_max_signal/data_multiplier) GE 32767.) AND $
       (float(my_clock_frequency) LT (0.1*2.^30))) DO BEGIN
    my_clock_frequency = my_clock_frequency*long(10)
    data_multiplier = float(my_clock_frequency)/float(my_clock_counts)
ENDWHILE
my_clock_counts = fix(my_clock_counts)
my_clock_frequency = long(my_clock_frequency)

image_data = intarr(n_cols,n_rows,n_data)
image_data(0:(n_cols-1),0,1) = fix(my_clock_counts)
image_data(0:(n_cols-1),0,0) = fix(0.5+ws_signal/data_multiplier)

IF (n_elements(data_title) EQ 0) THEN BEGIN
    data_title = 'Data title'
ENDIF
IF (n_elements(x_title) EQ 0) THEN BEGIN
    x_title = 'X title'
ENDIF
IF (n_elements(y_title) EQ 0) THEN BEGIN
    y_title = 'Y title'
ENDIF
IF (n_elements(image_title) EQ 0) THEN BEGIN
    image_title = 'Image title'
ENDIF

init_sd,sd
sd.clock_frequency = long(my_clock_frequency)
sd.file_name = byte(nc_filename)
sd.date_time = byte(systime())
sd.operator = byte('STACK')
sd.comment1 = byte(comments)
sd.scan_width = n_cols
sd.scan_height = n_rows
sd.x_step_size = float(x_stop-x_start)/float(n_cols-1)
sd.y_step_size = 1.
sd.i0_mode = 3
sd.scan_driver_mode = 4 ; for spectrum
sd.x_start = x_start
sd.x_stop = x_stop
sd.y_start = y_start
sd.y_stop = y_stop
sd.initial_lambda = ws_angstroms(0)
sd.final_lambda = ws_angstroms(n_cols-1)

write_stxm, nc_filename, sd

; Restore the preexisting BSIF parameters
n_rows = old_n_rows
n_cols = old_n_cols
n_data = old_n_data
x_normal = old_x_normal
y_normal = old_y_normal
rotated = old_rotated
x_start = old_x_start
x_stop = old_x_stop
y_start = old_y_start
y_stop = old_y_stop
x_dist = old_x_dist
y_dist = old_y_dist
image_title = old_image_title
x_title = old_x_title
y_title = old_y_title
data_title = old_data_title
image_data = old_image_data

return
END

