; Copyright (c) 2001, Adam Hitchcock  All rights reserved.
;	Unauthorized reproduction prohibited.
;+
;
; NAME:
;	READ_STXM4
;
; LAST CHANGED:                   30-Sep-01
;
; PURPOSE:
;	This function reads a STXMIV image (spectrum) file into
;   the standard 2d (1d) structure used internally in AXIS (ALS format)
;	Spectra are distinguished from images by existence of only one row
;
; CATEGORY:
;	Data analysis / conversion:
;
; CALLING SEQUENCE:
;
;	Result = read_stxm4()
;
; INPUTS:
;
; KEYWORD PARAMETERS:
;	FILE:  provides filename and thus pickfile dialog not launched (/silent)
; SILENT:  if set there is no logging of the conversion (for multifile processing)
;
; OUTPUTS:
;	RESULT is EITHER:
;  a 2d image in AXIS 2d data structure
;   s = {t:'2d', x:x, y:y, e:energy; d:d, xl:x_label, yl: y_label, dl: data_label}
;  OR a 1d spectrum in AXIS 1d data structure
;   s = {t:'1d', x:x, d:d, dn:d, xl:x_label, yl: y_label, dl: data_label}
;
; COMMON BLOCKS:
;	@axis_com
;	@bsif_com
;
; MODIFICATION HISTORY:
; (29-jun-01 aph) develop to read new STXM IV ; modified from read_cryo
; (18-jul-01 aph) add storage of parameters in sd structure of bsif_com to properly transfer info
; (30-sep-01 aph) adapt to read NSLS STRXM-IV spectra; images in kHz
;-

Function read_stxm4, File = File, silent = silent

on_error,2
@axis_com
@bsif_com

null=''

if NOT keyword_set(file) then begin
	sxm_file = pickfile2(/read,FILTER='*.sm', /LPATH, DEFPATH=defpath)
	if sxm_file EQ '' then return, null
endif else sxm_file = file

; ------------- read from sm format
energy = 1.
read_sm, sxm_file, sm_par, khz, col_um, row_um

;  ------- generate AXIS format structure
; x = findgen(n_cols)*sm_par.col_pixel_um  ; +  sm_par.col_start_um
; y = findgen(n_rows)*sm_par.row_pixel_um  ; +  sm_par.col_start_um
t = ax_name(image_title) &  dl = t(1)
t = size(row_um)
if t(1) GT 1 then begin
; -------- generate 2d structure
	energy = sm_par.start_ev
	dwell = sm_par.dwell_msec
	xl = string(FORMAT='("x (um)     E = ",f8.3," eV     dwell = ",f5.2," ms")', $
      energy, dwell)
	 x= col_um
	 y = row_um
	 s = {t: '2d', x: x, y: y, e: energy, $
	     d : khz, xl: xl, yl: ' nsls stxmIV    y (um)', dl: dl}

	; ---------- store parameters in BSIF_COM and SD structure for possible use elsewhere
	sd.wavelength = 12398.0/energy
	sd.dwell_time = dwell
	sd.x_start = x(0) & sd.x_stop = x(n_cols-1) & sd.x_step_size = x(1) - x(0)
	sd.y_start = y(0) & sd.y_stop = y(n_rows-1) & sd.y_step_size = y(1) - y(0)
	x_start = x(0) & x_stop = x(n_cols-1)
	y_start = y(0) & y_stop = y(n_rows-1)
	image_data(*,*,0) = khz
	image_data(*,*,1) = 1.
	; --------  print information about the image
	 t = size(image_data)
	 if NOT keyword_set(silent) then $ 	; do not print out info string if silent keyword set
	        print, sxm_file, ' read in as NSLS X1A stxmIV image. ', strtrim(string(n_cols),2),' x ', $
	        strtrim(string(n_rows),2),' pixels;  Energy = ', energy, ' eV. dwell = ', dwell, ' msec'
endif else begin

; -------- generate 1d structure (splot,pectrum)
	x = 1.2398/col_um
	dwell = sm_par.dwell_msec
	xl = string(FORMAT='("E (eV).   dwell = ",f5.2," ms")', dwell)
    s = {t: '1d', x: x, d : khz, xl: xl, dn: khz, yl: ' nsls stxmIV counts per sec (kHz)', dl: dl}

	; --------  print information about the spectrum
	 if NOT keyword_set(silent) then $ 	; do not print out info string if silent keyword set
	        print, sxm_file, ' read in as NSLS X1A stxmIV spectrum. '
endelse

return, s

END
