; Copyright (c) 1998-2024 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_NANOMAX_STACK.PRO
;;
;LAST CHANGED: ----------------------------------- 24 July 2024 (aph)

; PURPOSE:
;	This function reads a set of images using the read_nanomax_image routine to read NanoMAX h5 files
;
; CATEGORY:
;	    Runs stand alone or inside aXis2000.
;
; CALLING SEQUENCE:
;	RESULT = READ_NANOMAX_STACK(path=path, type = type, silent = silent, verbose=verbose, group = group)
;
; INPUTS: none
;
; KEYWORDS:
;	PARTH
;	VERBOSE
;	GROUP
;
; OUTPUTS:  stack as (*.dat, *.ncb)
;
;
; COMMON BLOCKS:
; AXIS_COM    standard set of common blocks
; stack_process_com
; COMMON volume_data, image_stack
; bsif_com


; MODIFICATION HISTORY:
; (26-Mar-24 aph) first version
; (05-Apr-24 aph) evolve from sets of files to single h5 file with switching among stxm, ptycho ~(amp,pha)
; (18-Apr-24 aph)  make it work with energy_series_7.h5 (in pass test area)
; (08-Jul-24 aph) for notebooks\...\stxm  undo, ln()
; (24-Jul-24 aph) change default directory; put stack file name in axis log

;-

FUNCTION  read_nanomax_stack, path=path, type = type, silent=silent, verbose=verbose, group = group
@axis_com
@stack_process_com
COMMON volume_data, image_stack
@bsif_com
on_error,2

; check to see if run stand-alone (axis_on = 0) or from aXis2000 (axis_on = 1)
if n_elements(group) EQ 0 then begin
	if widget_info(/active) then axis_on=1 else axis_on=0
endif else axis_on = 1


null = 0

; ------------ select NanoMAX stack
;print, "old default path is ", DefPath
if n_elements(last_stack) EQ 0 then last_stack = ' '
file = dialog_pickfile(title='Select NanoMAX stack  *.h5 files',filter=['*.h5'], file=last_stack, path=DefPath, get_path=folder_path)
last_stack = file
DefPath=folder_path
t = ax_name(file)
fileshort = t(1)
cd, DefPath
;print, "new default path is ", DefPath

; ------------ read NanoMAX  *.h5 file(s)
a = h5_parse(file, /read_data)

; ---------- get photon energies
energies = a.info.energies._data
ev = energies
n_e = n_elements(energies)

; ---------- get sequential scan numbers
scan_nums = a.info.scans._data
if NOT keyword_set(silent) then print, scan_nums

; ask user what type of data to read

; ---- have user choose  from  ptycho_<A>mplitude,  ptycho_<P>hase, <S>TXM
if n_elements(last_type) EQ 0 then last_type ='A'
p_text = "ptycho_<A>mplitude, ptycho_<P>hase, <R>aw STXM&Io, <S>TXM_Eiger "
if axis_on EQ 1 then begin
	type = get_text(prompt = p_text, val = last_type, group=group)
endif else begin
	type = get_text(prompt = p_text, val = last_type)
endelse
type = strtrim(strupcase(type),2)
if type NE 'A' AND  type NE 'P' AND type NE 'R' AND type NE 'S'  then begin
	axis_log, 'Type must be  A, P, R or S'
	return, null
endif
last_type = type


CASE type OF
; ----- reading STXM stack
;
; ----- reading ptycho_opticaldensity  stack, and remove ln()
'A': begin
		axis_log, ' reading ptycho amplitude stack from '+ fileshort
		d =  a.maps.ptycho_opticaldensity.data._data
		d = d - (min(d) + 1.e-2*(max(d)-min(d)))	; remove offset
		image_stack = exp(d)						; undo ln()
		pix_size_array = a.maps.ptycho_opticaldensity._psizes._data
		pix_size=mean(pix_size_array*1e9)
	end

; ----- reading ptycho phase stack
'P': begin
		axis_log, ' reading ptycho phase stack from '+ fileshort
		image_stack =  a.maps.ptycho_phase.data._data
		pix_size_array = a.maps.ptycho_phase._psizes._data
		pix_size=mean(pix_size_array*1e9)
	end

; ----- reading raw STXM and Io stack

'R': begin
		axis_log, ' reading raw_STXM (integrated Eiger DIs) and Io stacks from ' + fileshort
		image_stack =  a.maps.raw_stxm.data_it_Eiger._data/1e8
		image_stack_io=  a.maps.raw_stxm.data_I0._data/1e-8
		image_stack = image_stack / image_stack_io
		pix_size_array = a.maps.raw_stxm._psizes._data
		pix_size=mean(pix_size_array*1e9)
	 end

; ----- reading partially processed STXM: ln(raw_stxm.data_it_Eiger._data/raw_stxm.data_I0)

'S': begin
		print, ' reading partially processed_STXM stack [ln(raw_stxm.data_it_Eiger._data/raw_stxm.data_I0)]'
		axis_log, ' reading  from Eiger STXM stack ' + fileshort
		d =  a.maps.stxm.data._data
		d = d - (min(d) + 1.e-2*(max(d)-min(d)))	; remove offset
		image_stack = exp(d)						; undo ln()
		pix_size_array = a.maps.raw_stxm._psizes._data
		pix_size=mean(pix_size_array*1e9)
	 end

ENDCASE

; get sizes to make x,y axes
tmp = size(image_stack)
nx = tmp(1)
ny = tmp(2)
if NOT keyword_set(silent) then print, 'stack size: nx   ny   ', nx, ny

; ------- generate x, y axes  &  store in bsif_com parameters
if NOT keyword_set(silent) then print, 'pixel size (nm) ', pix_size
x = findgen(nx)
x = pix_size*x
y = findgen(ny)
y = pix_size*y

x_start = x(0)/1000
x_stop  = x(n_elements(x)-1)/1000.
x_step = (x_stop - x_start) /(nx-1)

y_start = y(0)/1000
y_stop  = y(n_elements(y)-1)/1000
y_step = (y_stop - y_start) /(ny-1)

n_rows = nx
n_cols = ny

; generate tag lines for each image of the stack
filename_ev_msec_list = strarr(n_e)

; ---------- loop through to build tag line of stack and flip thhe images (Y ==> -Y)
for i = 0, n_e-1 do begin
	d = image_stack(*,*,i)
	d = rotate(d, 7)
	image_stack(*,*,i) = d
	filename_ev_msec_list(i) = 'scan ' + strtrim(string(scan_nums(i)),2) + '  ' + string(energies(i))
;	axis_log, filename_ev_msec_list(i)
endfor

; ---- Ptycho reconstructions (NB ion chamber was incorporated when prepare sets of DI's for reconstruction)
; pixel size at each energy, NB the ptycho data is already sampled to have same (nx, ny) but they cover different areas

; pixel_sizes =  a.maps.ptycho_opticaldensity._psizes._data/1e8

; The images should be resampled using the pixel size at the highest photon energy

; at  Soleil the probleme was the (nx,ny) values differed but the pixel size was the same. This was fised by
		; tmp = ax_interp(tmp, mesh=[nx,ny])
; at NannoMAX (nx,ny)is the same for each image but the size of the pixel increases (slightly) with increasing E
;
; For each image of stack to cover the same physical area, we need to
; (i) define x and y axes in real space using pixel_size for each image
; (ii) start from highest E image and truncate to the SAME physical area
; (iii) re-mesh the truncated images to the same (nx, ny)


; -------- write aXis2000 format stack
if n_elements(stack_file) LE 0 then stack_file = ''
stack_file = get_text(prompt = 'name of stack ', val = stack_file, group = group)
print, '|',stack_file,'|'
stack_file = DefPath + stack_file
stack_wb, stack_file  ;, /real

return, stack_file

end


;  FIRST VERSION WHERE EACH SCAN WAS SAVED IN A SEPERATE FILE

; ---------- loop through to get all images of the stack
;for i = 0, n_e-1 do begin
;	tmp = read_nanomax_ptycho(file=filelist(i), /silent)
;	image_stack(*,*,i) = exp(tmp.d)						; exponentiate since Maik Khn logged prior to normaization to alda2.1 ion chamber signal
;	t = ax_name(filelist(i))
;	ev(i) = tmp.e
;	filename_ev_msec_list(i) = t(1) + '  '+ string(ev(i))
;	axis_log, filename_ev_msec_list(i)
;endfor
;
;if n_elements(stack_file) LT 0 then stack_file = ' '
;stack_file = get_text(prompt = 'name of stack ', val = stack_file)
;stack_file = DefPath + stack_file
;stack_wb, stack_file, /real
