; Copyright (c) 2025-2025 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_ALS_STXM_5322
;
;LAST CHANGED: ----------------------------------- 29 Nov 1015 (from 05 Jul 2025) (aph)
;
;PURPOSE:
;	This function converts unformatted (binary) *.stxm files
;   to axis2000 files (*.axb for single images; (*.dat, *.ncb) for stacks
;It can process:
;   pre-July2025 images 
;   stacks measured using rectilinear scanning 
;   Jun-2025 spiral scan stacks 
;   (& perhaps spiral scan single images, but not tested as of 4-Jul-2025)
;   
;CATEGORY:
;	STAND ALONE: utility & may be called from aXis2000
;
;CALLING SEQUENCE:
;	Result = READ_ALS_STXM_5322(in_file=in_file, filter=filter, hdf5_only=hdf5_only)
;
;CALLED FROM AXIS:  Read~STXM(Nexus)~ALS *.stxm532
;
;INPUTS:
;  binary *.stxm file (written by the 5322 stxm)  ==>  WHAT ABOUT 11.0.2 ??
;
;KEYWORDS
;	IN_FILE - name of *.stxm	file
;	FILTER  -  extension (*.stxm for ALS cosmic;5322)
; hdf5_only - return the structure generated by the IDL 9.1 command h5_parse(filename,/read_data)

;COMMON BLOCKS:
;	@AXIS_COM	standard set of common blocks
;	@bsif_com
; @stack_process_com
; @nexus_common
;
;MODIFICATION HISTORY:
; (18-Jun-25 aph) First version using IDL hdf5 binary file read-in techniques
; 		h5_parse(filename,/read_data)
;    This does not work in ILD6.3, or 8.6, but does work in IDL9.1
;    (NB in IDL6.3 h5_parse works with pixelator and CLS py_stxm hdf5 formatted files) 
; (29-Jun-25 aph) get image (rectilinear) and stack (rectilinear, spiral) read-in working in IDL9.1
; (04-Jul-25 aph) for rectilinear image scans use only linear portion of scan line
; (05-Jul-25 aph)  clean-up code
; (29-Nov-25 aph) adapted from read_als_stxm()  to enable reaad=in of stack 5322_251117129.stxm

FUNCTION READ_ALS_STXM_5322, in_file=in_file, filter=filter, hdf5_only=hdf5_only

@AXIS_COM
@bsif_com
@stack_process_com
@nexus_com

on_error, 2

COMMON volume_data, image_stack

; ----------------------
if not keyword_set(filter) then filter="*.stxm"
if n_elements(def_path) EQ 0 then def_path = 'c:\aXis2000'  ; in case start without prior call to aXis2000
if n_elements(in_file) EQ 0 then begin
	in_file=pickfile2(LPATH=lp, filter=filter, TITLE="Read ALS 5.3.2.2 *.stxm file",  $
         LFILE=lf, group = group, _extra=e)
endif
t = ax_name(in_file)
fileshort=t(1)
s=0

print, ' Reading ALS 5.3.2.2 *.stxm file (post Nov-2025)'

;  use h5_parse in IDL9.1 to extract an hdf5 (nexus) file into an IDL structure
;  this only works in IDL versions > 9.0  (does not work in IDL8.6)
a = h5_parse(in_file, /READ_DATA)

if keyword_set(hdf5_only) then begin
  return, a
endif

;  NB  the test file (5322_251117129.stxm) says it is an image, but the data is a stack

a_data =  a.entry0.default.data._data
ev = a.entry0.default.energy._data
n_e = n_elements(ev)
if n_e EQ 1 then a_type = 'image'
a_dim = size(a_data)
if a_dim(0) EQ 3 then a_type = 'stack'
 
IF n_e EQ 1 then print, 'Single image at E(eV) = ', ev(0) 
IF n_e GT 1 then print, ' STACK: ', strtrim(string(n_e),2), ' energies from ', ev(0), '  to  ', ev(n_e-1)
if n_elements(ev) LE 0 then begin
  print, 'no energies found'
  return, s 
endif

;  so if energy = 1, there is only 1 image.
;  ; but I expect there will be single images recorded with spiral scan
;  ; TEST data file: Y:\aXis-Pass-Test-data\00-READ\02-STXM-NEXUS\ALS_.stxm\NS_231029005.stxm 
;  
; FOR rectilinear scanned images, the x-dimension is oversampled
;  a.entry0.requested_values.sample_x.,_data  have 100 points 
;  the measured X-values and x-data dimension is  362  =>  OVERSAMPLED  in x

; need to interpolate the image data from the number of x-measured values to the number requested
; IDEALLy shoud use the MEASURED X-values but for now, I will linearize to the requested values
; 
; read x,y-co-ordinates
; if type EQ 'Spiral Image' then begin ; spiral scans are converted to a rectilinear map with x,y supplied 
;   xr = a.ENTRY0.binned_values.sample_x._data 
;   yr = a.ENTRY0.binned_values.sample_y._data
;   ddd  = a.ENTRY0.binned_values.data._data
;
; endif else begin    
; ;  for data measured using a rectilinear scan (type = 'Image'even if >1 image)
  xr= a.entry0.default.sample_X._data
  yr= a.entry0.default.sample_Y._data

;  print, 'Image array dinmensions'
;  help, ddd

  ; extract linear part of the scan & re-mesh to the requested x-values
;   print, 'isolate parts of images along the linear part of the scan line' 
;   xm = a.ENTRY0.counter0.sample_x._data
;;   print, 'min x-requested ',min(xr),'   min x-measured(line 0) ',  min(xm(*,0))
;   t = where(xm(*,0) LE min(xr), count)
;   if count LE 0 then begin
;     axis_log, 'minimum x-measured is greater than minimum x-requested. Stop reading '+ fileshort
;     return, s
;   endif
;   t_xmin = t(count-1) 
;   t = where(xm(*,0) GE max(xr), count)
;   if count LE 0 then begin
;     axis_log, 'maximum x-measured is smaller than maximum x-requested. Stop reading '+ fileshort
;     return, s
;   endif
;   t_xmax = t(0)
;   tmp = xm(*,0)          ; use first scan line as x-axis for all lines
;   print, 'using points ',strtrim(string(t_xmin),2),'  to  ', strtrim(string(t_xmax),2), $
;                '  of ',  strtrim(string(n_elements(tmp)),2),' points' 
;   xm0 = tmp(t_xmin:t_xmax)
;   nxm0 = n_elements(xm0)
   ; ideally should do this for each y-value and average, or fit all or at least more y-lines
   ; a quick examination shows little variation in the linear range
   ; best would be to treat each line of the image seperately and deal with possibly different 
   ; # of points along the line
;   xr = congrid(xm0,n_elements(xr))
   
;  now extract the measured intensities (in DDD) from t_xmin to t_xmax 
;  ddd = ddd(t_xmin:t_xmax,*,*)
  
; endelse
 
;  help, ddd
;  help, xr, yr

  nx = n_elements(xr)
  ny = n_elements(yr)
  xmin = min(xr)    & xmax = max(xr)
  xstep = (xmax - xmin)/(nx-1)
  ymin = min(yr)    & ymax = max(yr)
  ystep = (ymax - ymin)/(ny-1)
 
  ;  interpolate the data to match the number of requested samples
  ; NB initially I used image_data as a variable but that is in bsif_com where it can be changed
  ; also I suspect 'image' is a reserved word, so change to Rimage
;  if n_e eq 1 then begin         ; interpolate single image
;    Rimage = congrid(ddd, nx, ny)
;    
;  endif else begin               ; interpolate stack if recorded with rectilinear scan
;      help, ddd                  ; should not chage spiral scan data
;       image_stack = fltarr(nx, ny, n_e)
;       for i = 0, n_e-1 do begin 
;          t = ddd(*,*,i)
;          tmp = congrid(t, nx, ny)
;          image_stack(*,*,i) = tmp
;       endfor
;       if type EQ 'Image' then $
;          print, 'stack interpolated to (nx,ny) ', strtrim(string(nx),2),' x ',strtrim(string(ny),2)
;       help, image_stack
;  endelse
;

image_stack = a_data
 
 dwell = mean(a.entry0.default.count_time._data)
 ; ----- populate parameters in bsif_com
  x_start = xmin
  x_stop  = xmax
  y_start = ymin
  y_stop  = ymax
  n_rows = nx
  n_cols = ny
;  sampled_dwell = a.ENTRY0.COUNTER0.count_time._data
;  help,  sampled_dwell 
;  if type EQ 'Spiral Image' then nxm0 = nx
;  dwell =  sampled_dwell(0)*(float(nxm0)/float(nx))
;  print, 'Sampled dwell = ',strtrim(string(sampled_dwell(0)),2),'  Effective dwell (ms) = ', dwell(0)
; 
;; preferred X-label and stack label formats
;; filename_ev_msec_list(i) = filenm  + ' : '+$
;  ;     strtrim(string(ev(i),format='(f10.2)'),2)+' eV '+$
;  ;     strtrim(string(dwell,format='(f7.2)'),2)+' msec'
;
;  xl = '  ' +  strtrim(string(ev(i),format='(f10.2)'),2) + ' eV.   dwell = ' $
;             + strtrim(string(dwell,format='(f7.2)'),2)  + ' ms'
;  print, 'X-label = ', xl
;  yl = 'ALS COSMIC *.stxm' + type
;  print, 'y-label = ', yl

; ==================================
;  so now should have x, y, ddd, and ev values matching
if n_e GT 1 then begin
  t = size(image_stack) 
  print, 'STACK:  nx ',nx,'  ny ', ny, ' image  size ', t(1),' x ', t(2), '  # energies ', n_e 
endif else begin
  print, 'Single image (nx,ny) ', nx, ny, '  energy (eV) = ', ev
endelse

; ==================================  SINGLE IMAGE ====================
if n_elements(ev) EQ 1 then begin
    ; ========  save as a single image =======================
    
    filename = a.entry0.default._file
    t = ax_name(filename)
    dl = t(1)
    x = xr    ; as read from *.stxm, all variables are doubles
    y = yr
    ; preferred X-label and stack label formats
    ; filename_ev_msec_list(i) = filenm  + ' : '+$
    ;     strtrim(string(ev(i),format='(f10.2)'),2)+' eV '+$
    ;     strtrim(string(dwell,format='(f7.2)'),2)+' msec'

    xl = '  ' +  strtrim(string(ev(0),format='(f10.2)'),2) + ' eV.   dwell = ' $
      + strtrim(string(dwell,format='(f7.2)'),2)  + ' ms'
    print, 'X-label = ', xl
    yl = 'ALS 5322 *.stxm' + type
    print, 'y-label = ', yl
    tmp = {t:'2d', x:xr, y:yr, xl:xl, yl:yl, e: ev(0), d: Rimage, dl:dl}
 ;   help, tmp    
    axis_log, 'read STXM5322 image ' + dl
    return, tmp

endif else begin
;  ==================================  STACK ====================
; ========  save as a stack. Stacks were acquired with both rectilinear scans and spiral scans
; TEST DATA
; 
;   STXM_SCAN_TYPE = Spiral Image
; Y:\aXis-Pass-Test-data\00-READ\02-STXM-NEXUS\ALS_.stxm\NS_250525081.stxm
; 
;   STXM_SCAN_TYPE = Image  
; Y:\aXis-Pass-Test-data\00-READ\02-STXM-NEXUS\ALS_.stxm\NS_231029069.stxm
  
 ; Identify stack scan type from STXM_SCAN_TYPE
;  if a.entry0.counter0.STXM_SCAN_TYPE._data EQ 'Spiral Image' then begin 
;    
; NB  The DATA and X, Y co-ordinates for spiral scans are saved as
;     ENTRY0.binned_values.sample_x._data
;     different from the rectilinear scans

    print, 'Recorded stack (nx,ny,nE) = ', $
       strtrim(string(nx),2), '   ', strtrim(string(ny),2),'   ',  strtrim(string(n_e),2)

  ; generate filename_ev_msec_list
    t=ax_name(in_file)
    fileshort = t(1)
    if not keyword_Set(out_file) then out_file = t(2)
    out_file = pickfile2(filter='*.ncb', file=fileshort, title='binary stack file ', WRITE=write)
    IF strlen(out_file) EQ 0 THEN BEGIN
      axis_log, 'no output file name.'
      return, s
    ENDIF
    t=ax_name(out_file)
    fileshort_out = t(1)
    
    print, ' Output file: ',  fileshort_out, '.ncb' 

    filename_ev_msec_list = strarr(n_e)  
    ; (24-Apr-16 aph)  syntax of filename_ev_msec_list not consistent with stack_process
    ;       filename_ev_msec_list(i) = filenm  + ' : '+$
    ;     strtrim(string(ev(i),format='(f10.2)'),2)+' eV '+$
    ;     strtrim(string(dwell,format='(f7.2)'),2)+' msec'

    FOR i = 0, n_e-1 DO BEGIN
      filename_ev_msec_list(i) = fileshort_out + ' ' + strtrim(string(i),2)+' ' + string(ev(i), format = '(F10.2)') + $
        ' eV ' +  string(dwell(0), format='(F7.2)') + ' msec'
 ;     print,  strtrim(string(i),2), '    ', filename_ev_msec_list(i)
    ENDFOR
    
; convert doubles to single floats
  image_stack = ax_dts(image_stack) 

;  write stack to disk
    stack_wb, out_file
    
    t = ax_name(out_file)
    Data_File_Name = t(1)
    log_text = string(format='(i3," x",i3," stack with ",i3," energies read from ",A,".")',nx, ny, n_e, Data_File_Name)
    axis_log, log_text
    
    return, s
endelse
end
