; Copyright (c) 1998-2100 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	AX_XRF_CONVERT
;
;LAST CHANGED: ----------------------------------- 14-Dec-09
;
; PURPOSE:
;	This function converts  a series of XRF fluorescence spectra written by XGLabs software
; into 1-channel fields with truncation to a user specified upper energy and optional binning
; it also generates the sum of all the XRF spectra and generates a stack of the data
;
; CATEGORY:
;	utiltiy
;
; CALLING SEQUENCE:
;	AX_XRF_CONVERT
;
; INPUTS:
;
; KEYWORDS:
;	GROUP		group leader (Axis_ID if called from aXis2000 and axis)
;	MULTI_DET 	multiple detector signals in one file
;	SILENT  	if set, no feedback
;	EXTRA 		passed on parameters
;
; OUTPUTS:
; set of 1-column ascii files suitable to read into PYMCA
; sum of all all XRF spectra
;
; COMMON BLOCKS:
;   @AXIS_COM - set of common blocks for AXIS
;	Volume_data - stack
;   @BSIF_COM - common block for netCDF
;
; SIDE EFFECTS:
;
; RESTRICTIONS:
; assumes all XGLabs file names involve cycling the last 3 spaces before '.' as the sequence number
;
; PROCEDURE:
;
; RESTRICTIONS:
;
; MODIFICATION HISTORY:
; (14-dec-09 aph) dirsat written as tool for Twinmic staff & users
;               to get around problems that XGLabs software optionally writes only non-zero channels
;-

FUNCTION AX_XRF_CONVERT, group=group, verbose=verbose, multi_det=multi_det, silent=silent, _extra=e
@axis_com
@stack_process_com
COMMON volume_data, image_stack
@bsif_com
on_error,2

s= 0

; if keyword_set(multi_det) then filter = '*_All_*.dta' else filter = '*_Sum_*.dta'
filter = '*.dta'
file1uc=PICKFILE2(/Read, title='first file of sequence', FILTER=filter, /LPATH, DEFPATH=defpath)
if strlen(file1uc) le 0 THEN return, s
; -- determine which type (SUM, ALL), and force fitler to only those types
file1 = strlowcase(file1uc)
spos = strpos(file1, '_all_')		; NB ax_name forces all strings to LOWER CASE !!!
; print, file1
if spos GT 1 then begin
	filter = '*_All_*.dta'
	check_str = 'all_'
	all = 1
endif else begin
	filter = '*_Sum_*.dta'
	check_str = 'sum_'
	all = 0
endelse
spos1=strpos(file1,'.dta')
spos2=strpos(file1,check_str)
; print, 'first ',  spos1, spos2
num_dig1 = spos1-spos2-4
first = strmid(file1, spos2+4,num_dig1)
f1 = strmid(file1uc, 0, spos2+3)
T=AX_NAME(f1)
file_path=t(0)
fileshort=t(1)
fileroot=file_path + fileshort
print, ' '
print, 'AX_XRF_CONVERT - converting XGLabs  XRF files to 1-column counts'
if check_str EQ 'all_' then print, 'Reading in a multi-detector formatted file' else $
 	print, 'Reading in the sum of detectors file '
; print, 'FIRST file: ', fileshort, '    FIRST number = ', first, '    # of digits = ', num_dig1

file2uc=PICKFILE2(/Read, TITLE='last file of sequence', FILTER=filter, /LPATH, DEFPATH=defpath)
if strlen(file2uc) le 0 THEN RETURN, s
file2 = strlowcase(file2uc)
spos1=strpos(strlowcase(file2),'.dta')
spos2=strpos(file2,check_str)
; print, 'last ', spos1, spos2
num_dig2 = spos1-spos2-4
last = strmid(file2, spos2+4,num_dig2)
f2 = strmid(file2uc, 0, spos2+3)
T=AX_NAME(f2)
IF t(1) NE fileshort then begin
	 axis_log, 'Root names of first & last filenames do not match'
	 return, s
ENDIF

out_dir = fileshort + '_cts'
if keyword_Set(group) then  $
     out_dir=get_text(prompt ='name of folder for output', val = out_dir, group=group) $
     else out_dir=get_text(prompt ='name of folder for output', val = out_dir)
file_path_out= file_path + out_dir
file_mkdir, file_path_out
if not keyword_set(silent)then begin
;	print, 'LAST file: ', fileshort, '    LAST number = ', last, '    # of digits = ', num_dig2
;	print, 'file_path= ', file_path
;	print, 'fileshort= ', fileshort
;	print, 'fileroot= ', fileroot
;	print, ''
	print, 'Output files (*.cts) stored in ', file_path_out
endif

; -------- define dimensions of map so can check if there are enough files
if keyword_Set(group) then begin
     ncols = get_num(prompt='map: # columns (X)', val=ncols, group=group)
     nrows = get_num(prompt='map: # row (Y)', val=nrows, group=group)
endif else begin
     ncols = get_num(prompt='map: # columns (X)', val=ncols)
     nrows = get_num(prompt='map: # row (Y)', val=nrows)
endelse
npixels=ncols*nrows
nfiles = fix(last)-fix(first)+1
if npixels NE nfiles  then begin
	axis_log, strtrim(string(nfiles),2) + ' files provided. Map requires ' + strtrim(string(fix(npixels)),2)
	return, s
endif

; -------  IF a multi-detector file,  ask user which detector number to read
if all EQ 1 then begin
	num_det_max = 8
	num_det_prompt = 'detector (1-' + strtrim(string(fix(num_det_max)),2) +')'
	if keyword_Set(group) then detector = get_num(prompt=num_det_prompt, val=1, group=group) $
	                      else detector = get_num(prompt=num_det_prompt, val=1)
	if detector LT 0 OR detector GT num_det_max-1 then detector = 1
endif else detector=0

vary_dig = 0
if num_dig2 - num_dig1 NE 0 then begin
	vary_dig = 1
	num_dig1 = 4
	print, 'variable nuumber of digits in file number'
endif

maxE = read_xrf_spectra(file=file1,all=all, detector=detector, old=old, /silent, /maxE)
if not keyword_set(silent)then begin
	if old EQ 1 then  print, 'Data from old software (original detectors)'
endif else begin
 	print, 'Data from new software (post Dec-09 detectors)'
endelse

if old EQ 1 then tp = 'Upper channel' else tp = 'Upper cut-off energy (eV)'
if keyword_Set(group) then begin
     upperE = get_num(prompt=tp, val=maxE, group=group)
     bin = get_num(prompt='bin', val=1, group=group)
endif else begin
	upperE = get_num(prompt=tp, val=maxE)
	bin = get_num(prompt='bin', val=1)
endelse

tt = systime(1)
if num_dig1 EQ 1 then num_ref=10
if num_dig1 EQ 2 then num_ref=100
if num_dig1 EQ 3 then num_ref=1000
if num_dig1 EQ 4 then num_ref=10000
if num_dig1 EQ 5 then num_ref=100000


if vary_dig EQ 0 then begin
	fn = num_ref + fix(first)
	filer = fileshort + '_' + strmid(strtrim(string(fn),2),1,num_dig1)
 endif else 	filer = fileshort  +  '_' + strtrim(first,2)
file = file_path + filer + '.dta'
; print, 'Reading file ', file
tmp = read_xrf_spectra(file=file, all=all, detector=detector, /silent, old=old, upperE=upperE, /counts)
xrf_sum = lonarr(n_elements(tmp))
nfiles = fix(last)-fix(first)+1
tnfiles = string(nfiles,format='(I5)')
npts = n_elements(tmp)
c = fltarr(npts/bin, nfiles)
;help, tmp
;help, c

for i = fix(first), fix(last) do begin
	if vary_dig EQ 0 then begin
		fn = num_ref + i
		filer = fileshort + '_' + strmid(strtrim(string(fn),2),1,num_dig1)
	endif else 	filer = fileshort +  '_' +  strtrim(string(i),2)
	file = file_path + filer + '.dta'
	tmp = read_xrf_spectra(file=file, /silent,all=all, detector=detector, old=old, upperE=upperE, /counts)
	if bin GT 1 then begin
		tmpo = tmp
		tmpt = tmp
		for j = 0,(n_elements(tmp)/bin-1) do tmpt(j)=fix(total(tmpo(bin*j:bin*j+bin-1)))
		tmp=tmpt(0:(n_elements(tmp)/bin-1))
	endif
	filenum = i-fix(first)
	c(*,filenum)=tmp
	if vary_dig EQ 1 then begin
		fn = num_ref + i
		filer = fileshort + '_' + strmid(strtrim(string(fn),2),1,num_dig1)
	endif
	fileout = file_path_out + ax_sep()  +  filer  + '.cts'
;	print, ' counts to be saved in ', fileout
;	help, tmp
	openw, lun, fileout, /get_lun
	printf,lun, tmp, format='(I6)'
	close, lun & free_lun, lun
	xrf_sum = xrf_sum + tmp
	if not keyword_set(silent) then print, filer, n_elements(tmp)
	axis_log, string(filenum + 1) + ' of ' +  tnfiles, /clear
	if keyword_Set(group) then begin
		WIDGET_CONTROL, /Hourglass
	endif
endfor
;help, c
; axis_log, tnfiles + ' of ' +  tnfiles
axis_log, 'Conversion time' + string(systime(1)-tt,format='(I5)') + ' sec'


; ----- write sum of XRF in *.cts, and *.txt format
if detector GT 0 then begin
   fileout = file_path_out + ax_sep() + fileshort + 'all_det-' + strtrim(string(fix(detector)),2) + '.cts'
endif else begin
	fileout = file_path_out + ax_sep() + fileshort + 'sum_all.cts'
endelse
; -- write out sum as a *.cts file
tmp = xrf_sum
;if max(xrf_sum) GT 65000 then begin
;	scale = fix( max(xrf_sum)/65000.)
;	tmp = xrf_sum/float(scale)
;axis_log, 'Max counts is ' + string(max(xrf_sum))
;axis_log, 'Data in sum is divided by ' + string(scale, format='(I2)')
;endif
;tmp = fix(tmp)
; print, 'file to output sum as 1 col ', fileout
; help, tmp
openw, lun, fileout, /get_lun
printf,lun, tmp, format='(F10.0)'
close, lun & free_lun, lun

t =ax_name(fileout)
   axis_log, 'Sum of all XRF spectra written to file ' +  fileout ;  t(1) +  '.cts'

;  ------------ write out stack and sum of all spectra
;print, 'file_path= ', file_path
;print, 'file_path_out = ', file_path_out
;print, 'fileshort= ', fileshort
;print, 'fileroot= ', fileroot
;print, 'filer= ', filer

if vary_dig EQ 0 then begin
	fn = num_ref + fix(first)
	filer = fileshort + '_' + strmid(strtrim(string(fn),2),1,num_dig1)
 endif else 	filer = fileshort  +  '_' + strtrim(first,2)
file = file_path + filer + '.dta'
s = read_xrf_spectra(file= file,  all=all, detector=detector, /silent, upperE=upperE)	; get last datafile in aXis format as a template

; ------------- prepare & write aXsi2000 stack file
c = transpose(c)
; -------- Create aXis2000 stack
npts = npts/bin
image_stack=reform(c, ncols, nrows, npts)
ev_max= s.x(n_elements(s.x)-1) & ev_min = s.x(0)
slope = (ev_max - ev_min)/npts
ev = ev_min + slope*findgen(npts)

filename_ev_msec_list = strarr(npts)
for i = 0, npts-1 do filename_ev_msec_list(i) = string(ev(i))
x_start = 0  & x_stop = ncols  & x_step = 1.
y_start = 0  & y_stop = nrows  & y_step = 1.

; --------- ask user what file name they want - default name is same as files without counter
fileout = file_path_out + ax_sep() + fileshort + '.ncb'
sname=pickfile2(/write, file= fileout, filter='*.ncb', title = 'name of binary stack file')
stack_wb, sname

; ------- convert sum of all XRF spectra to axis 1d structure, with binning as needed
if old EQ 1 then xl = 'Channel #' else xl = 'X-ray energy (eV) '
yl = 'sum of ' +  string(nfiles) + ' XRF spectra '
if old EQ 1 then yl = yl + 'old detectors' else  yl = yl + 'new detectors'
x = s.x
d = float(xrf_sum)			; should have large intensity !!
if bin GT 1 then begin
	npts = n_elements(xrf_sum)
	x = findgen(npts)
	slope = (upperE - s.x(0))/npts
	x = s.x(0) + slope*x
endif
if all EQ 1 then begin
	dl = fileshort + '   Detector# ' + strtrim(string(fix(detector)),2)
	fileout = file_path_out + ax_sep() + fileshort
	fileout = fileout + '_det_' + strtrim(string(fix(detector)),2) +'.txt'
endif else begin
	dl = fileshort
	fileout = file_path_out + ax_sep() + fileshort + '.txt'
endelse
s = {t:'1d', x:x, d:d, dn:d, xl:xl, yl:yl, dl:dl}
q=spc_save(s, file=fileout, /noLbl)
t=ax_name(q)
if not keyword_set(silent)then  print, 'sum of all spectra wrote to ', t(0)
return, s

end
