; Copyright (c) 1998-2000 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	AX_STXM4NC
;
;LAST CHANGED: ----------------------------------- 18-Jul-01
;
; PURPOSE:
;	This procedure converts one or a series of NSLS STXM 4 image files
; into netCDF format and, if multiple files, writes a stack_list (*.sl) file
; with names of converted files
;
; CATEGORY:
;	AXIS: image analysis ; runs standalone
;
; CALLING SEQUENCE:
;	AX_STXM4NC, file
;
; INPUTS:
;	FILE - an NSLS STXM 4 image file (*.sm) to be converted to netCDF format (*.nc)
;
; KEYWORD PARAMETERS:
;	LIST - name of file with list of image files (*.lst)
;   OUTLIST - name of a file to write list of netCDF files (*.sl)
;	ONE - convert only a single file
;
; OUTPUTS:
;  FILE.NC - image written to a netCDF format binary file
;
; COMMON BLOCKS:
;   @AXIS_COM - set of common blocks for AXIS
;   @BSIF_COM - common block for netCDF
;
; SIDE EFFECTS:
;	many !
;
; RESTRICTIONS:
;	caveat emptor
;
; PROCEDURE:
;	AX_STXM4NC assumes that ax_rd_im has put the image and associated information
; into BSIF_COMMON variables:
;	image_data (the array which holds the image - possibly several
;	planes deep for I, IO, CLOCK)
;
; MODIFICATION HISTORY:
; (18-Jul-01 aph) derived from  als2ncdf using readSTXM4
;-

pro ax_stxm4nc, file , list=list, outlist = outlist, one = one, delE=delE

@axis_com
@bsif_com
on_error,2

if keyword_set(list) then begin
   osname=strupcase(strmid(!version.os,0,3))
   stack_readlist, list, filename_list
endif else begin
	if n_elements(file) eq 0 then begin  ;popup file dialog box
	   if keyword_set(one) then begin
		   filename_list = PICKFILE2(/Read, FILTER='*.sm', title = 'Select STXM IV file', /LPATH, DEFPATH=defpath)
		endif else begin
	 		list=PICKFILE2(/Read, FILTER='*.lst', title = 'Select list file', /LPATH, DEFPATH=defpath)
	 		stack_readlist, list, filename_list
		endelse
	endif else filename_list = file
endelse
n_files = n_elements(filename_list)
if keyword_set(delE) then del_E = float(delE) else del_E=0.

; -------- use GET_PATH to establish correct file names
; -------- check if file exists - this handles subdirectory changes
file1 = filename_list(0)
New_Path = get_path(file1, list = list)
if New_Path EQ '' then return	; let user cancel
for i = 0, n_files-1 do begin
	t = ax_name(filename_list(i))
	filename_list(i) = New_Path + t(1) + '.' + t(2)
endfor

; ------ open *.sl file for stack_list -----------------
if keyword_set(outlist) then begin
	if  keyword_set(list) then t = ax_name(list) $   ; make stacklist file name same as list file
	    else t = ax_name(outlist)
	 	outlist = t(0) + t(1) + '.sl' ; force outlist to have .sl extension
	openw,unitw, outlist,/get_lun
	printf, unitw, strcompress(New_Path)		; write path at top of file
endif

; -------------- check back for user to see if filename list is valid---------
	t = ax_name(filename_list(0))
	print, 'First file to process is ', t(0) + t(1) + '.' + t(2)

FOR i_file = 0,n_files-1 DO BEGIN
	WIDGET_CONTROL, /hourglass
	file = filename_list(i_file)
	IF strlen(file) GT 0 THEN BEGIN
;	print,' Procesing file ', fix(i_file), '. . . ', file
		t = ax_name(file)
		ext = t(2)
		fileshort = t(1)
		Data_path = t(0)
		file = t(0) + t(1) + '.' + ext
; READ IN DATA
		tmp = read_stxm4(file=file, /silent)		; gets data in AXIS format !! 2-d structure
		energy = 12398.0/sd.wavelength + del_E(0)
		sd.wavelength = 12398.0/energy
;		print, Format = '(A, "  E=",F7.3, "(eV).  Dwell=", f4.2 ,"(ms)")', file, energy, sd.dwell_time
		data_title = byte(fileshort)
		x_title = 'X'
		y_title = 'Y'
; -- print out the values for BSIF_COM variables
;		print, 'BSIF common stuff'
;		print, 'n_rows, n_cols, n_data', n_rows, n_cols , n_data
;		print, 'x_start, x_stop', x_start, x_stop
;		print, 'y_start, y_stop', y_start, y_stop

		image_title = byte(fileshort)
		sd.clock_frequency = 1.
; WRITE-OUT DATA
		file = t(0) + t(1) + '.nc'    ; setup correct name for output in nc format
		wrstx_ax,file, /real
		if keyword_set(outlist) then printf,unitw, strcompress(t(1) + '.nc')
		print,'wrote nsls NetCDF format file: ', file
		WIDGET_CONTROL,UPrompt, Bad_ID = badID, SET_VALUE='STXM4_to_netCDF '+ fileshort $
		   + '       E(eV)= '+strcompress(string(energy))
	ENDIF
ENDFOR

if keyword_set(outlist) then begin
	close, unitw
	free_lun, unitw
	print, 'wrote stack_list file: ', outlist
	WIDGET_CONTROL, Uprompt, Bad_ID = badID, $
	    SET_VALUE='stack_list saved in ' + strcompress(outlist)
endif
close,/all		; useful to ensure logical units are available !

if NOT keyword_set(list) OR keyword_Set(one) then begin	; plot if only 1 file converted
	t = Widget_info(/active)	; BUT only if AXIS is running
	if t NE 0 then begin
		tmp = read_bnl(file)
		Label(CurBuf) = tmp.dl
		HANDLE_VALUE, Data(CurBuf), tmp, /set
		PlotBuf, CurBuf
	endif
endif
END
