; Copyright (c) 1998-2011 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		READ_MAPPER
;
;LAST CHANGED: ----------------------------------- 	27-Jul-11
;
; PURPOSE:
;	This procedure extracts the data from Mapper files (files written by CJJJ's stack alignmnent)
;  and stores it in float arrays x and y
;
; CATEGORY:
;	read in
;
; CALLING SEQUENCE:
;	READ_MAPPER, filename, x, y, help=help, quiet=quiet
;
; INPUTS:
; 	FILENAME	name of file
;
; KEYWORDS:
;	HELP	list parameters
;	QUIET	trun off print comments
;
; FUNCTIONS
;	Read_mapper_firstnum	strips out all stuff before a comma
;
; MAPPER file format is  (this is output oz Zimba auto-align)
;
; ! X-Y Pixel shifts after alignment
; ! Full images used
; ! Aligned to preceding images
; ! Correlation maximum determined by 3-pt quadratic fit
; ! No edge enhancement
; ! Maximum allowed shift of 10 pixels
; ! Shift threshold of 0.01 pixels
; ! Edgegauss smoothing of 3 pixels
; ALIGN(-1,-1,0,0,10,0.01,3,0,0,0,0,-1
; PLOTIT(4
; 532_110326020_a000.xim  278.00   0.80,0.0000,0.0000
; 532_110326020_a001.xim  285.30   0.80,-0.1701,-1.0884
; 532_110326020_a002.xim  292.40   0.80,-0.0824,-1.3861
; 532_110326020_a003.xim  300.20   0.80,0.0361,-1.5492
;
; OUTPUTS:
;		X  x-values
;		Y  y-values (can be multi-dimensional)
;
; COMMON BLOCKS: none
;
; MODIFICATION HISTORY:
; (20-Nov-77 cjj)  rewritten from Rivers stuff
; (26-Jul-11 aph) AXIS standard header added
;-

FUNCTION Read_mapper_firstnum, input_string, firstnum_string
;
; This routine strips out the first stuff before a comma

IF (strlen(input_string) EQ 0) THEN BEGIN
    firstnum_string = ''
    return,0
ENDIF

comma_pos = strpos(input_string,',')
IF (comma_pos EQ (-1)) THEN BEGIN
    firstnum_string = input_string
    input_string = ''
ENDIF ELSE BEGIN
    firstnum_string = strmid(input_string,0,comma_pos)
    length = strlen(input_string)-comma_pos-1
    input_string = strmid(input_string,comma_pos+1,length)
ENDELSE

return,1
END

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO Read_mapper, filename, x, y, help=help, quiet=quiet

IF (keyword_set(help) OR (n_elements(filename) EQ 0) OR $
    (n_params() EQ 0)) THEN BEGIN
    print, 'read_mapper,filename,x,y [/quiet]'
    print, '  If y has several points, you will get arrays like'
    print, '    x(n) and y(m,n)'
    return
ENDIF

filelist = findfile(filename)
svec = size(filelist)
IF (svec(0) EQ 0) THEN BEGIN
    print,'Could not open file "'+filename+'"'
    return
ENDIF

on_ioerror,bailout
get_lun,lun
openr, lun, filename

;  Data for plotting in Mapper files is usually proceded by the PLOTIT
;  commmand.  To find the data, search for this command.
plotit_string = "PLOTIT("
input_string = ""
n_points = 0
n_y_data = 0
got_plotit = 0
this_num = 0.
WHILE (NOT EOF(lun)) DO BEGIN
    readf, lun, input_string

    IF (got_plotit EQ 0) THEN BEGIN
        IF (strpos(strupcase(input_string), 'PLOTIT') NE (-1)) THEN BEGIN
            got_plotit = 1
        ENDIF
    ENDIF ELSE BEGIN
        ; We've had a PLOTIT command, so now we're reading data.
        IF (n_points EQ 0) THEN BEGIN
            IF (read_mapper_firstnum(input_string,numstr) EQ 0) THEN BEGIN
                goto,bailout
            ENDIF
            reads,numstr,this_num
            numbers = [this_num]
            WHILE (read_mapper_firstnum(input_string,numstr) NE 0) DO BEGIN
                reads,numstr,this_num
                numbers = [numbers,this_num]
            ENDWHILE
            x = [numbers(0)]
            n_y_data = n_elements(numbers)-1
            temp_y = [numbers(1:n_y_data)]
            n_points = 1
        ENDIF ELSE BEGIN
            IF (read_mapper_firstnum(input_string,numstr) EQ 0) THEN BEGIN
                goto,bailout
            ENDIF
            reads,numstr,this_num
            numbers = [this_num]
            WHILE (read_mapper_firstnum(input_string,numstr) NE 0) DO BEGIN
                reads,numstr,this_num
                numbers = [numbers,this_num]
            ENDWHILE
            x = [x,numbers(0)]
            IF (n_y_data NE (n_elements(numbers)-1)) THEN BEGIN
                print,'Inconsistent number of Y points'
                goto,bailout
            ENDIF
            temp_y = [temp_y, numbers(1:n_y_data)]
            n_points = n_points+1
        ENDELSE
    ENDELSE
ENDWHILE

; Now rearrange the data
IF (n_y_data EQ 1) THEN BEGIN
    y = temp_y
ENDIF ELSE BEGIN
    y = fltarr(n_y_data,n_points)
    lindgen_list = lindgen(n_points)
    FOR i_data = 0, (n_y_data-1) DO BEGIN
        index_list = long(i_data) + long(n_y_data) * lindgen_list
        y(i_data,0:(n_points-1)) = temp_y(index_list)
    ENDFOR
ENDELSE

bailout:
close,lun
free_lun,lun

return
END


