; Copyright (c) 1998-2025 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		READ_BESSY_TXM.PRO
;25
;LAST CHANGED: -----------------------------------  10-Apr-2025 (from 21-Mar-14)
;
; PURPOSE:
;	This function reads a single image from Bessy TXM; applies
; positional and size information to the displayed image; optional binning and storage
; Sometime between 2014 and 2025 the positions.txt meta data file was changed
;
; CATEGORY:
;	data read. can execute outside of aXis2000
;
; CALLING SEQUENCE:
;	Result = READ_BESSY_TXM(file=file, filter=filter, bin-bin. normalize=normalize, store=store, silent=silent)
;
; INPUTS:
;
; KEYWORDS:
;	FILE:  name of file (*.pgm)
;	BIN	:   if set, integer factor by which to bin the image
;	FILTER: extension (default is '*.pgm')
;	NORMALIZE: if set, modify intensities to standard ring current (value of normalize parameter
;	STORE: if set, write *.axb (aXis2000 format) binary file of image
;	SILENT: if set, do not print tracking comments
;
; OUTPUTS:
;	No explicit outputs.
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;
; SIDE EFFECTS: none
;
; RESTRICTIONS: none
;
; MODIFICATION HISTORY:
; (20-Sep-11 aph) first version adapted from read_bessy_stxm
; (29-Oct-11 aph) pick up the baton again; fix up error in binning by adding new bin routine
; (21-mar-14 aph) add princeton_header_define to set up structure
; (10-Apr-25 aph) adapt to both old & new (>2025) version of positions file for Bessy TXM
;-

function read_bessy_txm, file=file, filter=filter, bin=bin, normalize=normalize, store=store, silent=silent
@axis_com
on_error,2

if NOT keyword_set(silent) then print, 'SILENCE PLEASE !

if n_elements(file) eq 0 then begin  ;popup file dialog box
   if not keyword_set(filter) then filter='*.spe'
   file=PICKFILE2(/Read, FILTER=filter, /LPATH, DEFPATH=defpath)
endif

if strlen(file) GT 0 THEN BEGIN
	name_bits = ax_name(file)	; sort the name into its parts
	path = name_bits(0)
	fileshort = name_bits(1)
    ext = name_bits(2)
 endif else begin
	axis_log, file + ' is not defined, or does not exist; read aborted'
	return, 0
endelse

if not keyword_set(silent) then print, 'Reading Bessy TXM file ', file

;  - Check that the text file with size information exists
file_txt = name_bits(0) +  name_bits(1) + '-positions.txt'
test=file_test(file_txt)
if test EQ 0 then begin
	axis_log, file_txt + ' is not defined, or does not exist; read aborted'
	return, 0
endif


; -------- Reading the "positions" file  to determined which format
openr, unit, file_txt, /get_lun

; check if the positions.txt file  is post-xxx (2025 for now) or pre-2014 version
; by searching for the first entry, expected to be 'MDIZ3T5G:current

a=''
WHILE(~ EOF(unit)) DO BEGIN
	READF,unit,a
	b = strsplit(a,' ',/extract)
	if b(0) EQ 'MDIZ3T5G:current' then position_type = 'New'
	if b(0) EQ 'sr:current' then position_type = 'Old'
ENDWHILE
close,unit & free_lun, unit
; point_lun, unit, 1

; -------- Reading the "positions" file for later (>2025 ??)  data files:  GET CHANGE DATE from Peter Guttmann
IF position_type Eq 'New'then begin
	openr, unit, file_txt, /get_lun
	a=''
	WHILE(~ EOF(unit)) DO BEGIN			; code sent by Hasharaj  08-Apr-2025
		READF,unit,a
		b = strsplit(a,' ',/extract)
	;	if NOT  keyword_set(silent) then PRINT,b(0), ' = ', b(1)
		if b(0) EQ 'MDIZ3T5G:current' then current = b(1)
		if b(0) EQ 'TXM1CL:micos:X' then x_centre = b(1)
		if b(0) EQ 'TXM1CL:micos:Y' then y_centre = b(1)
		if b(0) EQ 'u411pgm1:monoGetEnergy' then energy = b(1)
		if b(0) EQ 'TXM1CL:ccd:p_img' then pix_size = b(1)/1000.
		if b(0) EQ 'xm:sophia:cam1:AcquireTime_RBV' then dwell = 1000.*b(1)

	ENDWHILE
	close,unit & free_lun, unit
ENDIF

; -------- Reading the "positions" file for earlier (pre 2014 ?) data files:

IF position_type Eq 'Old'then begin
	openr, unit, file_txt, /get_lun
	a=''
	WHILE(~ EOF(unit)) DO BEGIN
		READF,unit,a
		b = strsplit(a,' ',/extract)
;	if not keyword_set(silent) then PRINT,b(0), ' = ', b(1)
		if b(0) EQ 'sr:current' then current = b(1)
		if b(0) EQ 'xm:sample:x' then x_centre = b(1)
		if b(0) EQ 'xm:sample:y' then y_centre = b(1)
		if b(0) EQ 'xm:mono:energy' then energy = b(1)
		if b(0) EQ 'xm:imagepixelsize' then pix_size = b(1)/1000.
		if b(0) EQ 'xm:ccd:exp_time' then dwell = 1000.*b(1)
	ENDWHILE
	close,unit & free_lun, unit
ENDIF

; ------ Reading the SPE file -------------

princeton_header_define		; set up structure needed to load parameters

if NOT keyword_set(silent) then read_princeton, file, img $
    else read_princeton, file, img, /silent
; --------------------
; delete first 8 rows which are the header (I think !)
t = size(img)
img = img(*,7:t(2)-1)
img=float(rotate(img, 7))
t = size(img)
n_cols = t(1)
n_rows = t(2)

; ------------  generate an axis2000 format structure from header information & image
x_step = pix_size    & y_step = pix_size
x = findgen(n_cols)  & y = findgen(n_rows)
x = x_centre/2.+ x*x_step
y = y_centre/2 + y*y_step
xl = string(FORMAT='("x (um)     E = ",f8.3," eV     dwell = ",f7.0," ms")', $
      energy, dwell)

; -------------- normalize to standard ring current, if selected
if keyword_Set(normalize) AND current NE 0 then begin
	imge = img*normalize/current
	yl = 'Bessy TXM   N-' + string(normalize, format='(F5.1)') + ' y (um)'
endif else begin
    yl = 'Bessy TXM   i-ring= '+ string(current, format='(F5.1)')+ ' y (um)'
endelse

s={t:'2d', d:img, x:x, y:y, e: energy, xl:xl, yl: yl, dl:fileshort}

; ---- bin the image
	if not keyword_set(bin) then bin = 1
	if bin GT 1 then begin
		s = ax_binf(s, bin = bin, /silent)
	endif

if not keyword_set(silent) then axis_log, 'read Bessy TXM file ' + fileshort

if keyword_set(store) then begin
	t=axb_save(s)
endif
return, s
end