; Copyright (c) 1998-2023 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	AX_9PAD,PRO
;
;
;LAST CHANGED: ----------------------------------- 17-May-2023

; PURPOSE:
;	This procedure evaluates signal in each pad of a 9-pad pattern.
; The image or map of a 9-pad damage should be matched to the appropriate
; template_a-xq-mask.axb  wherer q is 1, 2 or 3
;
; CATEGORY:
;	    Runs inside aXis2000.
;
; CALLING SEQUENCE:
; for procedures:
;	AX_9PAD, parameters
;
; INPUTS: none
;
; KEYWORDS: none
;
; OUTPUTS:
;	*.(dat,ncb) written
;
; COMMON BLOCKS:
; AXIS_COM    standard set of common blocks


; MODIFICATION HISTORY:
; (28-Jan-19 aph) first version
; (17-May-23 aph) format NAME for html Help file
;-

PRO ax_9pad
@axis_com
on_error,2

; read in the mask for the undamged region (%%-pad0.axb)
; to identify path to folder with the 9 pad files and the type of template

	file=PICKFILE2(/Read, FILTER='*.axb', title= 'pad mask file',/LPATH, DEFPATH=defpath)
	if file EQ '' then return
	tmp = axb_load(file=file, /silent)
    if  n_tags(tmp) EQ 0 THEN return
 ; ---- prepare buffers for average and standard deviation of each pad (including undamaged)
	npixel = fltarr(10)
	average = fltarr(10)
	stand_dev = fltarr(10)

	CurBufa = CurBuf			; save # of buffer with 9-pad image
	t = ax_name(file)
	t_last = strlen(t(1))-1
	base = t(0) + strmid(t(1),0,t_last)		; establish base name for the pad mask files
	HANDLE_VALUE, Data(CurBufa), tmp
	damage = tmp.d
	fileshort = tmp.dl
; set up column headers
	axis_log, 'Pad   area    mean (std dev)'
; iterate over the 9 pad-mask files
	dmg_img = tmp.d*0.
	for i = 0,9 do begin
;	read next mask file
		file = base + strtrim(i,2) + '.axb'
		tmp = axb_load(file=file, /silent)
		mask = tmp.d
; multiply 9-pad image by pad mask
		tmp.d = damage * mask
		dmg_img =  dmg_img + tmp.d
; extract numbers
		nz = where(tmp.d NE 0, n_pix)
		print, 'number of non-zero pixels = ', n_pix
		print, 'number of zero pixels = ', n_elements(tmp.d)-n_pix
		print, 'total number of pixels = ', n_elements(tmp.d)
		dat_tmp = reform(tmp.d(nz), n_pix)
		t = moment(dat_tmp,/nan,sdev=sdev,/double)
		npixel(i) = n_pix
		average(i) = t(0)
		stand_dev(i) = sdev/2.
; report values
		text = strtrim(i,2) + '  ' + string(n_pix,format='(i8," px   ")')
		text = text + string(average(i),stand_dev(i), format='(g11.3," (",g11.3, ")")')
		axis_log, text
; display pads as they are analysed
		tmp.d = dmg_img
		HANDLE_VALUE, Data(0), tmp, /set
		PlotBuf, 0

	ENDFOR

; --------- generate a plot of damage versus time (ask user for Time Multiplier used
; this calculation assumes the undamaged and the 9 damage pads have a base time sequence (in ms) of
; 0	(undmg)		0.0125	0.025	0.05			0.1	0.25	0.375			0.5	0.75	1

	tm = get_num(prompt='time multiplier', title = '9-pad generation time multiplier', group = axis_ID)
	x =fltarr(10)
	x(0) = 0. & x(1) = 0.0125	 & x(2) = 0.025	 	& x(3) = 0.05
	    		x(4) = 0.1		 & x(5) = 0.25		& x(6) = 0.375
	    		x(7) = 0.50		 & x(8) = 0.75		& x(9) = 1.0
	x = x*tm/1000.
; ------- construct & save spectrum of damage signals
	tmp = {t:'1d', d:average, x: x, yl: 'Damage signal', xl: ' pad exposure time (s)', $
	             dn: average, dl: fileshort + ' 9-pad ' }
	t = spc_save(tmp)
	CurBuf = buffer(group = AXIS_ID, prompt = 'Buffer for 9-pad values')
	HANDLE_VALUE, Data(CurBuf), tmp, /set
	Label(CurBuf) = tmp.dl
	PlotBuf, CurBuf

; ------- construct stand_dev
	tmp = {t:'1d', d:stand_dev, x: x, yl: 'Damage signal', xl: ' pad exposure time (s)', $
	             dn: stand_dev, dl: fileshort + ' 9-pad std_dev ' }
	t = spc_save(tmp)
	CurBuf = buffer(group = AXIS_ID, prompt = 'Buffer for standard deviation')
	HANDLE_VALUE, Data(CurBuf), tmp, /set
	Label(CurBuf) = tmp.dl
	PlotBuf, CurBuf

; -------- save  mean and standard deviation in multicolumn, *.xas format
	y = fltarr(3,10)
	y(0,*) = npixel
	y(1,*) = average
	y(2,*) = stand_dev
	filename=PICKFILE2(/Write, FILTER='*.xas', title= '9-pad damage data',/LPATH, DEFPATH=defpath)
	write_xas, filename(0), x, y , tmp.dl

; --------- display image of analysed regions
	HANDLE_VALUE, Data(CurBufa), tmp
  	CurBuf = 0
  	tmp.d = dmg_img
	HANDLE_VALUE, Data(CurBuf), tmp, /set
	Label(CurBuf) = tmp.dl + ' 9-pad'
	PlotBuf, CurBuf
END
