; Copyright (c) 1998-2024 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		H5_READ_CLS_SGM
;
;LAST CHANGED: ----------------------------------- 	20-Jan-2024 (aph)
;
; PURPOSE:
;	This function reads images from hdf5 files written from the SGM acquisition system
; using the IDL H5_browser for input   [h5_parse(file, /read_data)] (which is a KLUGE)
; and returns an aXis2000  format *.(ncb, dat) stack file
;
; CATEGORY:
;	data read-in; operates stand alone or in aXi2000
;
; CALLING SEQUENCE:
;	Result = H5_READ_CLS_SGM(file=file, silent = silent,  tey = tey, struct = struct, io=io,
;             dwell=dwell,diode=diode,pfyr=pfyr, sddavg = sddavg, sddnum=sddnum, dsave=dsave
;
; INPUTS: keyword only
;
; KEYWORDS:
;	FILE	name of file
;	SILENT 	suppress all tag lines
;	STRUCT	return all content of (entry1\data)  as IDL structure
;	TEY		total electron yield channel
;	DIODE	transmission (diode) channel
;	DWELL	dwell signal
;	IO		Io channel
;	PFYR	read partial fluorescence yield signals in user-defined peaks
;	DSAVE	if set, write result to disk from this function
;   SDDAVG	average of all SDD's and all (x,y) positions (to determine channels
;	SDDNUM  number of SDD detector
;
; OUTPUTS:
;	Based on keywords selected this function eturns
;			(i)  XRF spectrum from of SDDx averaged over all (x,y) sampled
;	        (ii) an aXis format image with the signal
;				 (TEY) (Io) (dwell) (PD) (PFY - of SDDx and channel range)
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;	bsif_com
;	stack_process_com
;   volume_data, image_stack
;

; DETAILS OF SGM_IMAGE HDF5 data file [version of Dec 2024, from Yuan Hao's data set)

;  2023-12-14t13-03-50-0600_528.h5 (sent by Hao Yuan ===>  SUPERCEDED  18-Jan-2024 (Zach Arthur)
;help, tmp.entry3.data ,/struct
;** Structure <8a91d78>, 24 tags, length=71338728, data length=71338608, refs=2:
;   _NAME           STRING    'data'
;   _ICONTYPE       STRING    ''
;   _TYPE           STRING    'GROUP'
;   _FILE           STRING    'Y:\data\XRM\CLS-SGM\2023-Hao\2023-12-14t13-03-50-0600_528.h5'
;   _PATH           STRING    '/entry3'
;   _COMMENT        STRING    ''
;   AUX1            STRUCT    -> <Anonymous> Array[1]
;   AUX2            STRUCT    -> <Anonymous> Array[1]
;   AUX3            STRUCT    -> <Anonymous> Array[1]
;   CLOCK           STRUCT    -> <Anonymous> Array[1]  time when each channel sampled (variable)
;   I0              STRUCT    -> <Anonymous> Array[1] Io
;   PD              STRUCT    -> <Anonymous> Array[1]PHOTODIODE  (transmission
;   SDD1            STRUCT    -> <Anonymous> Array[1]
;   SDD2            STRUCT    -> <Anonymous> Array[1]
;   SDD3            STRUCT    -> <Anonymous> Array[1]
;   SDD4            STRUCT    -> <Anonymous> Array[1]  best single SDD detector
;   TEMP1           STRUCT    -> <Anonymous> Array[1]
;   TEMP2           STRUCT    -> <Anonymous> Array[1]
;   TEY             STRUCT    -> <Anonymous> Array[1] TEY signal
;   XP              STRUCT    -> <Anonymous> Array[1] X-values
;   YP              STRUCT    -> <Anonymous> Array[1] Y-values
;   SIGNAL          STRUCT    -> <Anonymous> Array[1]
;   NX_CLASS        STRUCT    -> <Anonymous> Array[1]
;   AXES            STRUCT    -> <Anonymous> Array[1]
;
; 18-Jan-2024 (Zach Arthur)
; help, a.entry19 ,/struct
;** Structure <11986228>, 8 tags, length=86713928, data length=86713640, refs=2:
;   _NAME           STRING    'entry19'
;   _ICONTYPE       STRING    ''
;   _TYPE           STRING    'GROUP'
;   _FILE           STRING    'Y:\data\XRM\CLS-SGM\2023-Hao-2\test_539eV.h5'
;   _PATH           STRING    '/'
;   _COMMENT        STRING    ''
;   DATA            STRUCT    -> <Anonymous> Array[1]
;   NX_CLASS        STRUCT    -> <Anonymous> Array[1]
;IDL> help, a.entry19.data ,/struct
;** Structure <1163e260>, 46 tags, length=86713760, data length=86713472, refs=2:
;   _NAME           STRING    'data'
;   _ICONTYPE       STRING    ''
;   _TYPE           STRING    'GROUP'
;   _FILE           STRING    'Y:\data\XRM\CLS-SGM\2023-Hao-2\test_539eV.h5'
;   _PATH           STRING    '/entry19'
;   _COMMENT        STRING    ''
;   AUX1            STRUCT    -> <Anonymous> Array[1]
;   AUX2            STRUCT    -> <Anonymous> Array[1]
;   AUX3            STRUCT    -> <Anonymous> Array[1]
;   CLOCK           STRUCT    -> <Anonymous> Array[1]
;   CURRENT         STRUCT    -> <Anonymous> Array[1]
;   EMISSION        STRUCT    -> <Anonymous> Array[1]
;   EN              STRUCT    -> <Anonymous> Array[1]
;   EXIT_SLIT       STRUCT    -> <Anonymous> Array[1]
;   EXS             STRUCT    -> <Anonymous> Array[1]
;   GAP             STRUCT    -> <Anonymous> Array[1]
;   HEX_X           STRUCT    -> <Anonymous> Array[1]
;   HEX_Y           STRUCT    -> <Anonymous> Array[1]
;   HEX_Z           STRUCT    -> <Anonymous> Array[1]
;   I0              STRUCT    -> <Anonymous> Array[1]
;   KBHB_D          STRUCT    -> <Anonymous> Array[1]
;   KBHB_U          STRUCT    -> <Anonymous> Array[1]
;   KBLH            STRUCT    -> <Anonymous> Array[1]
;   KBLV            STRUCT    -> <Anonymous> Array[1]
;   KBVB_D          STRUCT    -> <Anonymous> Array[1]
;   KBVB_U          STRUCT    -> <Anonymous> Array[1]
;   PD              STRUCT    -> <Anonymous> Array[1]
;   PERIOD          STRUCT    -> <Anonymous> Array[1]
;   PHI_E           STRUCT    -> <Anonymous> Array[1]
;   PHI_L           STRUCT    -> <Anonymous> Array[1]
;   SDD1            STRUCT    -> <Anonymous> Array[1]
;   SDD2            STRUCT    -> <Anonymous> Array[1]
;   SDD3            STRUCT    -> <Anonymous> Array[1]
;   SDD4            STRUCT    -> <Anonymous> Array[1]
;   SR_ENERGY       STRUCT    -> <Anonymous> Array[1]
;   TEMP1           STRUCT    -> <Anonymous> Array[1]
;   TEMP2           STRUCT    -> <Anonymous> Array[1]
;   TEY             STRUCT    -> <Anonymous> Array[1]
;   VAY             STRUCT    -> <Anonymous> Array[1]
;   VAZ             STRUCT    -> <Anonymous> Array[1]
;   XP              STRUCT    -> <Anonymous> Array[1]
;   YP              STRUCT    -> <Anonymous> Array[1]
;   ZP              STRUCT    -> <Anonymous> Array[1]
;   SIGNAL          STRUCT    -> <Anonymous> Array[1]
;   NX_CLASS        STRUCT    -> <Anonymous> Array[1]
;   AXES            STRUCT    -> <Anonymous> Array[1]


; MODIFICATION HISTORY:
; (07-Jan-24 aph) first version , adapted from h5_read_sgm_image.pro
; (12-Jan-24 aph)  solve variable 'entry#' problem using tmp.(6) where tmp is full hdf5 content
;           extract E from filename; Io, Dwell,
;     loop structure to generate stacks of any of these are done in aXis2000~read~stacks~CLS_SGM~
;     using a list of file names so can do 1 or many automatically
; (14-Jan-24 aph) clean up and set up calls inside aXis2000
; (20-Jan-24 aph) add extract energy from (-.en);  add A to read in sum of all 4 SDD signals
;-

function H5_READ_CLS_SGM, file=file,tey=tey, pfyr=pfyr, struct = struct, io=io, diode=diode,  $
      dwell = dwell, sddavg=sddavg, sddnum = sddnum, ch_lo=ch_lo, ch_hi=ch_hi, silent=silent, dsave = dsave
@axis_com
@bsif_com
@stack_process_com
COMMON volume_data, image_stack
on_error,2

; determine if AXIS is running (therefore may have called ax_cgo)
; either when AXIS keyword is supplied or if any widget active
if  keyword_set(axis) then axis_on = 1 else axis_on = widget_info(/active)

reply = ' '
IF NOT keyword_set(file) then begin
	fltr='*.h5'
	file = pickfile2(/READ, FILTER=fltr, /LPATH, DEFPATH=defpath)
	if file EQ '' then return, reply
ENDIF
s = ''
if file EQ '' THEN RETURN, s  ; bail-out if no filename
check = findfile(file)		 ; bail-out if non-existent file
if strlen(check(0)) EQ 0 then begin
	axis_log, 'Cannot find ' + file
	return, s
endif
t = ax_name(file)
cd, t(0), current=current_dir
; print, 'changed default folder from ', current_dir, '  to  ',t(0)

; --------- get label from acquisition and construct short name
t = ax_name(file) & fileshort = t(1)

; ----- use h5f_parse to open file
tmp = h5_parse(file, /read_data)
;print, ''
;print, ' ============  whole structure top level  =======
;help, tmp, /struct

n = tmp.(6)		;  this is the entry# b=name which is different for each *.h5 file in the stack

if keyword_set(struct) then return, n.data

sgm_im = n.data

; ------------ get photon energy from en  ;  if not present (pre 18-Jan2024)
;     try to extract from file name e.g. 2023-12-14t15-24-39-0600_536.0.h5

if n_elements(sgm_im.en._data) GT 0 then begin
	energy = sgm_im.en._data
endif else begin
	uscore=strpos(file,'_', /reverse_search) +1
	dot = strpos(file,'.h5', /reverse_search)
	E_len= dot - uscore + 1
	energy = float(strmid(file,uscore, E_len))
endelse
if n_elements(energy) EQ 0 then energy = -1.


; ----------- get dwell array (NB  dwell at each pixel varies if fast scan was used)
 clock = sgm_im.clock._data
dwell = mean(clock)
; axis_log, 'dwell (s) = ' + string(dwell, format='(F7.3)')

; -------- get x & y positions
x_pos = sgm_im.xp._data			; (x,y)_pos, (x,y)_start, (x,y)_stop
y_pos = sgm_im.yp._data			; are trasferred to calling program by bsif.com
nx = n_elements(x_pos)
ny = n_elements(y_pos)
x_start = x_pos(0) &  x_stop = x_pos(nx-1)
y_start = y_pos(0) &  y_stop = y_pos(ny-1)

xl = 'x (um)     E =' + string(energy) + ' eV     dwell (ms) = ' +  string(dwell, format='(F7.1)')

;============ read requested data channels ===============
WIDGET_CONTROL, /Hourglass


;-----------  read in average XRF spectrum from selected SDD device (1,2,3,4, A)
if keyword_set(sddavg) then begin
	if not keyword_set(sddnum) then begin
		sddnum = string('3')
		if axis_on EQ 1 then begin
			 sddnum = get_text(prompt = 'which SDD ? (1,2,3,4, <A>ll)', val = sddnum, group = Axis_ID)
		endif else sddnum = get_text(prompt = 'which SDD ? (1,2,3,4, <A>ll)', val = sddnum)
		sddnum = strtrim(strupcase(string(sddnum)),2)
		if sddnum NE '1' and sddnum NE '2' and sddnum NE '3' and sddnum NE '4' and sddnum NE 'A' then begin
			axis_log, 'SDD # must be 1, 2, 3 or 4 or <A>ll'
			return, sddnum
		endif
	endif
; ---------- read and sum all 4 SDD signals
	print, 'Reading average XRF spectrum from sdd ', sddnum
	if sddnum EQ 'A' then begin
		sdd_xrf = fltarr(256)
		sdd_xrf_sum = fltarr(256)
		for i = 1, 4 do begin
			if i EQ 1 then sdd_img = sgm_im.sdd1._data
			if i EQ 2 then sdd_img = sgm_im.sdd2._data
			if i EQ 3 then sdd_img = sgm_im.sdd3._data
			if i EQ 4 then sdd_img = sgm_im.sdd4._data
			 print, 'reading SDD' + strtrim(string(i))
			for h = 0, 255 do begin
					sdd_xrf(h) = mean(sdd_img(h,*,*))
			endfor
			sdd_xrf_sum = sdd_xrf_sum + sdd_xrf
		endfor
		sdd_xrf = sdd_xrf_sum
	endif else begin
;  --------  read in sdd full array of sdd spectra
		print, 'reading XRF spectrum from SDD # ', sddnum
		CASE sddnum OF
		   	'1': sdd_img = sgm_im.sdd1._data
			'2': sdd_img = sgm_im.sdd2._data
			'3': sdd_img = sgm_im.sdd3._data
			'4': sdd_img = sgm_im.sdd4._data
		   	ELSE: begin
		   		PRINT, 'SDD is not 1,2,3,4 or <A>ll'
		   		return, sddnum
		   	ENDELSE
		ENDCASE
		sdd_xrf = fltarr(256)
		for h = 0, 255 do begin
				sdd_xrf(h) = mean(sdd_img(h,*,*))
		endfor
	endelse

;  --------  SGM operates SDD's such that 1 channel = 10 eV of XRF energy
	dl = fileshort + ' XRF from SDD'+sddnum
	xl = 'XRF energy    Xray E-inc =' + string(energy)
	yl = 'Counts        Xray E-inc = ' + string(energy)
	xrf={t:'1d', d:	sdd_xrf, x: 10*findgen(256), xl: xl, yl: yl, dn: sdd_xrf, dl: dl}
	if keyword_set(dsave) then reply = spc_save(xrf)
	return, xrf
endif


;----------- read in TEY image ----------
if keyword_set(tey) then begin
;	print, 'reading TEY image'
	img = sgm_im.tey._data
	tey_img = {t:'2d', x:x_pos, y:y_pos, xl:xl, yl:'TEY', e: energy, d: img, dl:fileshort + '  TEY'}
	out_file=fileshort +'_tey'
	if keyword_set(dsave) then reply = axb_save(tey_img, file=out_file)
	return, tey_img
endif


;----------- read in peak signal from <A>ll or a SINGLE partial XRF-yield image ----------
if keyword_set(pfyr) then begin

; ------------- for now, get user to identify channel limits in the call aXis2000
	lo = ch_lo
	hi = ch_hi
	yl = 'PFY:  lo = '+ strtrim(string(fix(lo)),2) +'   hi = ' + strtrim(string(fix(hi)),2)
	axis_log, 'extracting '+ yl

	pfy_img = fltarr(nx, ny)

;	get user to select SDD(s) to analyse if not identified in the call to h5_read_cls_sgm
	if not keyword_set(sddnum) then begin
		sddnum = string('3')
		if axis_on EQ 1 then begin
			 sddnum = get_text(prompt = 'which SDD ? (1,2,3,4, <A>ll)', val = sddnum, group = Axis_ID)
		endif else sddnum = get_text(prompt = 'which SDD ? (1,2,3,4, <A>ll)', val = sddnum)
		sddnum = strtrim(strupcase(string(sddnum)),2)
		if sddnum NE '1' and sddnum NE '2' and sddnum NE '3' and sddnum NE '4' and sddnum NE 'A' then begin
			axis_log, 'SDD # must be 1, 2, 3 or 4 or <A>ll'
			return, sddnum
		endif
	endif
; ---------- read and sum all 4 SDD signals
	if sddnum EQ 'A'  then begin
		print, 'Generating pFY stack from all 4 sdd ', sddnum
		sdd_xrf = fltarr(256)
		sdd_xrf_sum = fltarr(256)
		for k = 1, 4 do begin
			if k EQ 1 then sdd_img = sgm_im.sdd1._data
			if k EQ 2 then sdd_img = sgm_im.sdd2._data
			if k EQ 3 then sdd_img = sgm_im.sdd3._data
			if k EQ 4 then sdd_img = sgm_im.sdd4._data
			print, 'reading pFY stack / image from SDD', strtrim(string(k))
			for i = 0, nx-1 do begin
				for j = 0, ny-1 do begin
					for h = lo-1, hi-1 do begin
						pfy_img(i,j) = pfy_img(i,j) + sdd_img(h,j,i)
					endfor
				endfor
			endfor
		endfor
	endif else begin

;  --------  read in image  from a single sdd spectra
		print, 'reading pFY stack / image  from SDD # ', sddnum
		CASE sddnum OF
		   	'1': sdd_img = sgm_im.sdd1._data
			'2': sdd_img = sgm_im.sdd2._data
			'3': sdd_img = sgm_im.sdd3._data
			'4': sdd_img = sgm_im.sdd4._data
		   	ELSE: begin
		   		PRINT, 'SDD is not 1,2,3,4 or <A>ll'
		   		return, sddnum
		   	ENDELSE
		ENDCASE
	endelse
	for i = 0, nx-1 do begin
		for j = 0, ny-1 do begin
			for h = lo-1, hi-1 do begin
				pfy_img(i,j) = pfy_img(i,j) + sdd_img(h,j,i)
			endfor
		endfor
	endfor



;  --------- define lo and hi channel limits inside h5_read_cls_sgm.pro
;	lo = 45   & hi = 60
;	if axis_on then begin
;		lo=fix(get_num(prompt='lower limit of peak', val = lo), group = Axis_ID)
;		hi=fix(get_num(prompt='upper limit of peak', val = hi), group = Axis_ID)
;	endif else begin
;		lo=fix(get_num(prompt='lower limit of peak', val = lo))
;		hi=fix(get_num(prompt='upper limit of peak', val = hi))
;	endelse


	pfy_img = {t:'2d', x:x_pos, y:y_pos, xl:xl, yl:yl, e: energy, d: pfy_img, dl:fileshort + '  PFY ' + sddnum}
	out_file=fileshort+'_pfy'
	if keyword_set(dsave) then reply = axb_save(pfy_img, file = out_file)
	return, pfy_img
endif


if keyword_set(io) then begin
; ---------- get Io image at this energy  ----
	im_io = sgm_im.i0._data
	yl = 'y (um)     CLS_SGM stack  Io channel'
	dl = fileshort + ' Io'
	tmp = {t:'2d', x: x_pos, y: y_pos, xl: xl, yl: yl, e: energy, d: im_io, dl: dl}
	out_file=fileshort + '_Io'
	if keyword_set(dsave) then reply = axb_save(tmp, file = out_file)
;	help, tmp, /struct
	return, tmp
endif

if keyword_set(diode) then begin
; ---------- get Transmission image data ----
	im_diode = sgm_im.pd._data
	dl_tag = 'tranmsission'
	yl = 'y (um)     CLS-SGM TRANSMISSION'
	dl =  fileshort + ' ' + dl_tag
	print, 'header = ', dl
	tmp = {t:'2d', x: x_pos, y: y_pos, xl: xl, yl: yl,e: energy, d: im_diode, dl:dl}
	out_file=fileshort + '_PD'
	if keyword_set(dsave) then reply = axb_save(tmp, file = out_file)
	return, tmp
endif

; =========  ASSUME that the Dwell is constant

if keyword_set(dwell) then begin
; ---------- get array of dwell times  ----
	clock = sgm_im.clock._data
	yl = 'y (um)     SGM image  DWELL '
	dl =  fileshort + ' dwell'
	tmp = {t:'2d', x: x_pos, y: y_pos, xl: xl, yl: yl,e: energy, d: clock, dl:dl}
	out_file=fileshort + '_dwell'
	if keyword_set(dsave) then reply = axb_save(tmp, file = out_file)
	return, tmp
endif

end

;if keyword_set(pfya) then begin
;; --------- get average Fluorscence spectrum (all 4 detectors)
;	im_pfy = sgm_im.sdd_processed._data
;	fys = fltarr(256)
;	for ch = 0, 3 do begin
;		for i = 0,255 do fys(i) = fys(i) + mean(im_pfy(ch,i,*,*))
;	endfor
;	fys = fys/4
;	x = findgen(255)
;	dl =  fileshort + ' average X-ray fluorescence ' + ' E= ' + string(energy)
;	tmp = {t:'1d', x: x, xl: xl, yl:'Average X-ray fluorescence spectrum', dn: fys, d: fys, dl: dl}
;endif

;if keyword_set(pfyr) then begin
;; -------- get average maps for each peak identified in a *-peaks.txt file
;	im_pfy = sgm_im.sdd_processed._data
;; ----- read in peak positions
;	peak_file = pickfile2(TITLE='Peak limit file', /READ, FILTER='*.txt', /LPATH, DEFPATH=defpath)
;	peaks = text_read1(peak_file)
;	t = size(peaks)
;	n_peaks = t(2)
;	for i = 0, n_peaks-1 do print, i,  string(peaks(*,i), format='(i3)')		; add element symbol
;; ------ read in map from average of all 4 detectors integreated within the peak limits
;	WIDGET_CONTROL, /Hourglass
;	map_all = fltarr(nx, ny)
;	for i = 0, n_peaks-1 do begin
;		map = fltarr(nx, ny)		; rset to all 0's for each map
;		for ch = 0, 3 do begin
;			for j = peaks(0,i), peaks(1,i)-1 do map = map + im_pfy(ch,j,*,*)
;		endfor
;; ----- generate axis axb format image of PFY data
;		dl_tag = 'PFY channel ' + string(i)
;		yl = 'y (um)     SGM image ' + dl_tag
;		dl =  fileshort + ' ' + dl_tag
;		tmp = {t:'2d', x: x_pos, y: y_pos, xl: xl, yl: yl, e: energy, d: map, dl:dl}
;; ----- write out map
;		q = axb_save(tmp)
;		map_all= map_all + map
;	endfor
;	; ----- generate axis axb format image of PFY data
;	dl_tag = 'total FY'
;	yl = 'y (um)     SGM image ' + dl_tag
;	dl =  fileshort + ' ' + dl_tag
;	tmp = {t:'2d', x: x_pos, y: y_pos, xl: xl, yl: yl, e: energy, d: map_all, dl:dl}
;endif
;
;
;ax_wait, /off
;;return, tmp
