; Copyright (c) 1998-2022 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	AX_SGM_CSV_LOAD
;
;LAST CHANGED: -------- 28 Aug 2022 (aph, from 11 Oct 2019, ax_file_load)
;
;PURPOSE:
;	This set of procedures is a widget to read spectra from a multi-column ascii file in *.csv format
; It assumes the first line is a list of column heaaders, as inn ones written by CLS-SGM software
; Optional Io normalization for  either OD = -ln(signal/Io) or Yield = signal/Io
; The widget uses SPC_LOAD to execute the read-in
;
;CATEGORY:
;  AXIS: runs in aXis2000 or independently (except READ_ALL)
;
; AX_SPC_LOAD_EVENT ROUTINES:
;
; TEXT_AX_SGM_CSV_LOAD_PATH			change path
; TEXT_AX_SGM_CSV_LOAD_FNAME		modify filename; when changed, execute BUT_AX_SGM_CSV_LOAD_FNAME_BROWSE to get parameters
; BUT_AX_SGM_CSV_LOAD_FNAME_BROWSE browse for new filename; if found, load new parameters
; TEXT_AX_SGM_CSV_LOAD_EXT			change extension
; BUT_AX_SGM_CSV_LOAD_DISPLAY_HDR   show file in chunks of 6 lines
; BUT_AX_SGM_CSV_LOAD_MORE			show 6 more lines
; AX_SGM_CSV_ENERGY					select column for photon energy
; AX_SGM_CSV_SIGNAL					select column for signal (e.g.  tey, pd, sdd#, sdd_total)
; AX_SGM_CSV_IO						select column for incident flux (Io)
; AX_SGM_CSV_MODE					select  mode: Signal only, Yield (S/Io). OD (ln(Io/S)
; TEXT_AX_SGM_CSV_LOAD_ROWS			change number of rows to skip (default = 1)
; BUT_AX_SGM_CSV_LOAD_OK			proceed with loading multi-column file into AXIS
; BUT_AX_SGM_CSV_LOAD_CANCEL		abort read in
; BUT_AX_SGM_CSV_LOAD_ALL			read all columns with x-axis being row number
;
;COMMON BLOCKS:
;	@AXIS_COM	standard set of common blocks (inludes mcf_com)
;
;MODIFICATION HISTORY:
; (28-Aug-22 aph) customize ax_file_load for reading SGM *.csv files prepared by Zach Arthur (CLS)
;-
;

;-----------------------------------------------------------------
pro TEXT_AX_SGM_CSV_LOAD_PATH, Event
on_error,2
@axis_com
@ax_sgm_csv_com

; get the changed path name
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_AX_SGM_CSV_LOAD_PATH')
WIDGET_CONTROL,id, Sensitive=1, Get_value = file_path

t = ax_name(mcf_lastfile)
test = file_path + t(1) + '.' + mcf_ext
; -- check if the new filename exists - if not assume this is just a trap out of
; -- the user starting to type a changed name or a mistake
t = findfile(test(0), count=t_count)
if t_count EQ 0 then begin
	print, 'Changed path does not exist. Revert to exisitng path'
	WIDGET_CONTROL,id, Sensitive=1, set_value = t(0)
endif else begin
	fname = test
	print, 'New path is ', fname(0)
 	WIDGET_CONTROL,id, Sensitive=1, set_value = file_path
 	mcf_lastfile = fname(0)
endelse

end

;-----------------------------------------------------------------
pro TEXT_AX_SGM_CSV_LOAD_FNAME, Event
on_error,2
@axis_com
@ax_sgm_csv_com

; get the changed short name
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_AX_SGM_CSV_LOAD_FNAME')
WIDGET_CONTROL,id, Sensitive=1, Get_value = file_short

t = ax_name(mcf_lastfile)
test = t(0) + file_short + '.' + mcf_ext
; -- check if the new filename exists - if not assume this is just a trap out of
; -- the user starting to type a changed name or a mistake
t = findfile(test(0), count=t_count)
if t_count EQ 0 then begin
	print, 'Changed filename does not exist. Revert to exisitng name'
	WIDGET_CONTROL,id, Sensitive=1, set_value = t(1)
endif else begin
	fname = test
	print, 'New filename is ', fname(0)
 	WIDGET_CONTROL,id, Sensitive=1, set_value = file_short
 	mcf_lastfile = fname(0)
endelse

end

;-----------------------------------------------------------------
pro TEXT_AX_SGM_CSV_LOAD_EXT, Event
on_error,2
@axis_com
@ax_sgm_csv_com

; get the changed extension
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_AX_SGM_CSV_LOAD_EXT')
WIDGET_CONTROL,id, Sensitive=1, Get_value = tmp
mcf_ext = tmp(0)
print, 'New extension is ', mcf_ext

end

;-----------------------------------------------------------------
pro BUT_AX_SGM_CSV_LOAD_FNAME_BROWSE, Event, file=file
on_error,2
@axis_com
@ax_sgm_csv_com

; get the current path name
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_AX_SGM_CSV_LOAD_PATH')
WIDGET_CONTROL,id, Sensitive=1, Get_value = file_path
; get the current file name
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_AX_SGM_CSV_LOAD_FNAME')
WIDGET_CONTROL,id, Sensitive=1, Get_value = file_short
; get the current file extension
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_AX_SGM_CSV_LOAD_EXT')
WIDGET_CONTROL,id, Sensitive=1, Get_value = mcf_ext

fname = file_path + file_short + '.' + mcf_ext
print, 'Filename changed to ', fname

if not keyword_set(file) then begin
	if strlen(mcf_ext) EQ 0 then mcf_ext = '*'
	fname = pickfile2(Title='Select file', Filter = '*.' + mcf_ext)
endif else fname = file
if strlen(fname) GT 0 then begin
	mcf_lastfile = fname(0)	; ensure the filename is saved for future use
;	print, 'New filename is ', mcf_lastfile

; update name in the text box

	t = ax_name(fname)
	file_path = t(0)
	file_short=t(1)
	mcf_ext=t(2)
	id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_AX_SGM_CSV_LOAD_PATH')
	WIDGET_CONTROL,id, Set_value = file_path
	id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_AX_SGM_CSV_LOAD_FNAME')
	WIDGET_CONTROL,id, Set_value = file_short
	id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_AX_SGM_CSV_LOAD_EXT')
	WIDGET_CONTROL,id, Set_value = mcf_ext
	mcf_lastfile = fname		; save for future use if valid text file

; read file and update contents of header box
	openr, iunit, fname, /get_lun
	text = ''
	readf, iunit, text
	id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_AX_SGM_CSV_LOAD_HEADER')
	WIDGET_CONTROL,id, Set_value = text
	for i = 1, 6 do begin
		readf, iunit, text
		WIDGET_CONTROL,id, Set_value = text, /append
	endfor
	mcf_lastrow = 6
	close, iunit & free_lun, iunit
	id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_AX_SGM_CSV_LOAD_LASTROW')
	WIDGET_CONTROL,id, Set_value = strtrim(string(mcf_lastrow),2)
endif
end

;-----------------------------------------------------------------

pro BUT_AX_SGM_CSV_LOAD_OK, Event, group=wGroup
on_error,2
@axis_com
@ax_sgm_csv_com


; get the filename
fname =mcf_lastfile
t = ax_name(fname)
short_fname = t(1)
;------------- read in selected columns using SPC_LOAD --------------
;  SPC_LOAD returns an aXis2000 1D structure:
;   	mcf ={t:'1d', d: y-data, x: x, xl: xl,  yl: yl, dn: y-data, dl: t(1)}

; print, 'sgm_mode is ', sgm_mode, '    read ', sgm_mode_list(sgm_mode)
CASE sgm_mode OF
0: BEGIN		; signal only
		print, 'Reading in ONLY SIGNAL  from ', short_fname
		print, 'Energy = ', col_header(sgm_energy)
		print, 'Signal = ', col_header(sgm_signal)
		mcf  = spc_load(file=fname(0), typ = 'mc', xcol=sgm_energy+1, ycol=sgm_signal+1,skip= mcf_rows(0))
		mcf.dl = mcf.dl + '  ' + col_header(sgm_signal)
   END
1: BEGIN		; yield  S/Io
		print, 'Reading in SIGNAL/Io from ', short_fname
		print, 'Energy = ', col_header(sgm_energy)
		print, 'Signal = ', col_header(sgm_signal)
		print, 'Io = ',     col_header(sgm_io)
		mcf  = spc_load(file=fname(0), typ = 'mc', xcol=sgm_energy+1, ycol=sgm_signal+1,skip= mcf_rows(0))
		mcf0 = spc_load(file=fname(0), typ = 'mc', xcol=sgm_energy+1, ycol=sgm_io+1,    skip= mcf_rows(0))
		mcf_yield = mcf.d/mcf0.d
		mcf.d = mcf_yield
		mcf.dl = mcf.dl + '  ' + col_header(sgm_signal) + '/' + col_header(sgm_io)
	END
2: BEGIN		;  OD  ln(I0/S)
		print, 'Reading in OPTICAL DENSITY from ', short_fname
		print, 'Energy = ', col_header(sgm_energy)
		print, 'Signal = ', col_header(sgm_signal)
		print, 'Io = ',     col_header(sgm_io)
		mcf  = spc_load(file=fname(0), typ = 'mc', xcol=sgm_energy+1, ycol=sgm_signal+1,skip= mcf_rows(0))
		mcf0 = spc_load(file=fname(0), typ = 'mc', xcol=sgm_energy+1, ycol=sgm_io+1,    skip= mcf_rows(0))
		mcf_od = alog(mcf0.d/mcf.d)
		mcf.d = mcf_od
		mcf.dl = mcf.dl + '  ln['  + col_header(sgm_io) + '/' +  col_header(sgm_signal) + ']'
	END
ENDCASE
WIDGET_CONTROL, Event.Top, /DESTROY
return

end

;-----------------------------------------------------------------

pro BUT_AX_SGM_CSV_LOAD_MORE, Event, group=wGroup
on_error,2
@axis_com
@ax_sgm_csv_com

; get the filename
fname =mcf_lastfile

; read file and update contents of header box
openr, iunit, fname, /get_lun
text = ''
readf, iunit, text
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_AX_SGM_CSV_LOAD_HEADER')
WIDGET_CONTROL,id, Set_value = text
for i = 1, mcf_lastrow + 6 do begin
	readf, iunit, text
	WIDGET_CONTROL,id, Set_value = text, /append
endfor
close, iunit & free_lun, iunit
mcf_lastrow = mcf_lastrow + 6
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_AX_SGM_CSV_LOAD_LASTROW')
WIDGET_CONTROL,id, Set_value = strtrim(string(mcf_lastrow),2)
end

;-----------------------------------------------------------------
pro AX_SGM_CSV_ENERGY, Event
on_error,2
@axis_com
@ax_sgm_csv_com

; get the changed region
id = widget_info(Event.top, FIND_BY_UNAME = 'AX_SGM_CSV_ENERGY')
tmp = WIDGET_INFO(id, /Droplist_select)
sgm_energy = fix(tmp)
print, 'sgm_energy set to ', col_header(sgm_energy)
end

;-----------------------------------------------------------------
pro AX_SGM_CSV_SIGNAL, Event
on_error,2
@axis_com
@ax_sgm_csv_com

; get the column with the Signal  (PD, TEy, SDD)
id = widget_info(Event.top, FIND_BY_UNAME = 'AX_SGM_CSV_SIGNAL')
tmp = WIDGET_INFO(id, /Droplist_select)
sgm_signal = fix(tmp)
;print, 'sgm_signal set to ', col_header(sgm_signal)
end

;-----------------------------------------------------------------
pro AX_SGM_CSV_IO, Event
on_error,2
@axis_com
@ax_sgm_csv_com

; get the changed region
id = widget_info(Event.top, FIND_BY_UNAME = 'AX_SGM_CSV_IO')
tmp = WIDGET_INFO(id, /Droplist_select)
sgm_io = fix(tmp)
;print, 'sgm_io set to ', col_header(sgm_io)
end


;-----------------------------------------------------------------
pro AX_SGM_CSV_MODE, Event
on_error,2
@axis_com
@ax_sgm_csv_com

; get the changed region
id = widget_info(Event.top, FIND_BY_UNAME = 'AX_SGM_CSV_MODE')
tmp = WIDGET_INFO(id, /Droplist_select)
sgm_mode = tmp(0)
;print, 'SGM mode set to ', sgm_mode_list(sgm_mode)

end

;-----------------------------------------------------------------
pro TEXT_AX_SGM_CSV_LOAD_ROWS, Event
on_error,2
@axis_com
@ax_sgm_csv_com

; get the number of rows to skip (normally 1 for CLS-SGM *.csv files,
; but if in futere there are more header lines used, this can accomodated it
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_AX_SGM_CSV_LOAD_ROWS')
WIDGET_CONTROL,id, Sensitive=1, Get_value = tmp
mcf_rows = fix(tmp)
print, 'mcf_rows set to ', mcf_rows
end


;-----------------------------------------------------------------
pro BUT_AX_SGM_CSV_LOAD_CANCEL, Event
on_error,2
@axis_com
@ax_sgm_csv_com

WIDGET_CONTROL, Event.Top, /DESTROY
mcf= 0
end


;-----------------------------------------------------------------
pro BUT_AX_SGM_CSV_LOAD_ALL, Event
on_error,2
@axis_com
@ax_sgm_csv_com


; ------------ should only execute if axis2000 is running
; get the  name
fname =mcf_lastfile

widget_control, /hourglass

;------------- read ALL columns using SPC_LOAD --------------
nspec = spc_load(file=fname(0), typ = 'un', /multi, /print, skip=mcf_rows(0))  ; find number of columns
print, 'There are ', nspec,' columns'

if nspec(0) EQ 1 then begin
	tmp = spc_load(file=fname(0), xcol = 1, ycol =2, typ = 'mc', skip=mcf_rows(0))
	HANDLE_value, Data(CurBuf),tmp,/set
	Plotbuf,CurBuf
endif else begin
	CurBuf = CurBuf - 1
	for i = 2, nspec(0)+1 do begin
		CurBuf = CurBuf + 1
		if CurBuf GT 9 then CurBuf = 0
		tmp=spc_load(file=fname(0), xcol = 1, ycol = i, typ = 'mc', skip=mcf_rows(0))
		tmp.dl = tmp.dl + '  ' + col_header(i)
	  	HANDLE_value, Data(CurBuf),tmp,/set
	   	Plotbuf,CurBuf
	endfor
endelse
mcf = 0
WIDGET_CONTROL, Event.Top, /DESTROY
return
end

;
; **************************************************************
;
pro BASE_AX_SGM_CSV_LOAD_event, Event
@axis_com
@ax_sgm_csv_com

  wWidget =  Event.top

  case Event.id of

    Widget_Info(wWidget, FIND_BY_UNAME='TEXT_AX_SGM_CSV_LOAD_PATH'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH' )then $
        TEXT_AX_SGM_CSV_LOAD_PATH, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='TEXT_AX_SGM_CSV_LOAD_FNAME'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH' )then $
        TEXT_AX_SGM_CSV_LOAD_FNAME, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='TEXT_AX_SGM_CSV_LOAD_EXT'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH' )then $
        TEXT_AX_SGM_CSV_LOAD_EXT, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_AX_SGM_CSV_LOAD_FNAME_BROWSE'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_AX_SGM_CSV_LOAD_FNAME_BROWSE, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_AX_SGM_CSV_LOAD_MORE'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_AX_SGM_CSV_LOAD_MORE, Event
    end
      Widget_Info(wWidget, FIND_BY_UNAME='AX_SGM_CSV_ENERGY'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_DROPLIST' )then $
        AX_SGM_CSV_ENERGY, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='AX_SGM_CSV_SIGNAL'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_DROPLIST' )then $
        AX_SGM_CSV_SIGNAL, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='AX_SGM_CSV_IO'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_DROPLIST' )then $
        AX_SGM_CSV_IO, Event
    end
     Widget_Info(wWidget, FIND_BY_UNAME='AX_SGM_CSV_MODE'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_DROPLIST' )then $
        AX_SGM_CSV_MODE, Event
    end

    Widget_Info(wWidget, FIND_BY_UNAME='TEXT_AX_SGM_CSV_LOAD_ROWS'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH' )then $
        TEXT_AX_SGM_CSV_LOAD_ROWS, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_AX_SGM_CSV_LOAD_ALL'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_AX_SGM_CSV_LOAD_ALL, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_AX_SGM_CSV_LOAD_OK'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_AX_SGM_CSV_LOAD_OK, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_AX_SGM_CSV_LOAD_CANCEL'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_AX_SGM_CSV_LOAD_CANCEL, Event
    end
    else:    print, Event.id, ' not found'
  endcase

end
;-----------------------------------------------------------------

pro BASE_AX_SGM_CSV_LOAD, file=file, ext=ext, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_
@axis_com
@ax_sgm_csv_com

;  Resolve_Routine, 'ax_file_load_eventcb'     ; Load event callback routines
t = size(wGroup)
IF t(1) EQ 0 THEN modal_val=0 else modal_val=1
IF t(1) EQ 0 THEN read_all_sens = 0 else read_all_sens = 1

if widget_info(/active) EQ 0 then read_all_sens = 0 else read_all_sens = 1

; set up first entries to widget
	mcf_lastrow = 0
	if  n_elements(sgm_energy) EQ 0 then sgm_energy = 0
	if  n_elements(sgm_signal ) EQ 0 then sgm_signal  = 7
	if  n_elements(sgm_io) EQ 0 then sgm_io = 3
	if  n_elements(sgm_mode) EQ 0 then sgm_mode = 0
	if  n_elements(mcf_rows) EQ 0 then mcf_rows = 1
	if  keyword_set(ext) then mcf_ext = ext else mcf_ext = '*.csv'

  BASE_AX_SGM_CSV_LOAD = Widget_Base( GROUP_LEADER=wGroup,  $
      UNAME='BASE_AX_SGM_CSV_LOAD',  modal = modal_val $
      ,TITLE='Read SGM spectra from *.csv files' ,SPACE=3 ,XPAD=10, YPAD=10)

  TEXT_AX_SGM_CSV_LOAD_PATH_label = Widget_Text(BASE_AX_SGM_CSV_LOAD, UNAME='TEXT_AX_SGM_CSV_LOAD_PATH_label'  $
      ,FRAME=1 ,XOFFSET=0 ,YOFFSET=6, XSIZE=4 ,YSIZE=1, VALUE='Path ' )

  TEXT_AX_SGM_CSV_LOAD_PATH = Widget_Text(BASE_AX_SGM_CSV_LOAD, UNAME='TEXT_AX_SGM_CSV_LOAD_PATH'  $
      ,FRAME=1 ,XOFFSET=40 ,YOFFSET=6 ,SCR_XSIZE=300 ,SCR_YSIZE=20  $
      ,/EDITABLE ,/ALL_EVENTS ,XSIZE=26 ,YSIZE=1)

  TEXT_AX_SGM_CSV_LOAD_FNAME_label = Widget_Text(BASE_AX_SGM_CSV_LOAD, UNAME='TEXT_AX_SGM_CSV_LOAD_FNAME_label'  $
      ,FRAME=1 ,XOFFSET=0 ,YOFFSET=30, XSIZE=4 ,YSIZE=1, VALUE='File ' )

  TEXT_AX_SGM_CSV_LOAD_FNAME = Widget_Text(BASE_AX_SGM_CSV_LOAD, UNAME='TEXT_AX_SGM_CSV_LOAD_FNAME'  $
      ,FRAME=1 ,XOFFSET=40 ,YOFFSET=30 ,SCR_XSIZE=180 ,SCR_YSIZE=20  $
      ,/EDITABLE ,/ALL_EVENTS ,XSIZE=8 ,YSIZE=1)

  TEXT_AX_SGM_CSV_LOAD_EXT_label = Widget_Text(BASE_AX_SGM_CSV_LOAD, UNAME='TEXT_AX_SGM_CSV_LOAD_EXT_label'  $
      ,FRAME=1 ,XOFFSET=260 ,YOFFSET=30, XSIZE=4 ,YSIZE=1, VALUE='ext ' )


  TEXT_AX_SGM_CSV_LOAD_EXT = Widget_Text(BASE_AX_SGM_CSV_LOAD, UNAME='TEXT_AX_SGM_CSV_LOAD_EXT'  $
      ,FRAME=1 ,XOFFSET=300 ,YOFFSET=30 ,SCR_XSIZE=40 ,SCR_YSIZE=20  $
      ,/EDITABLE ,/ALL_EVENTS ,XSIZE=20 ,YSIZE=1, VALUE = [mcf_ext(0)])

  BUT_AX_SGM_CSV_LOAD_FNAME_BROWSE = Widget_Button(BASE_AX_SGM_CSV_LOAD,  $
      UNAME='BUT_AX_SGM_CSV_LOAD_FNAME_BROWSE' ,XOFFSET=398 ,YOFFSET=6  $
      ,SCR_XSIZE=73 ,SCR_YSIZE=30 ,/ALIGN_CENTER ,VALUE='Browse')

  BUT_AX_SGM_CSV_LOAD_DISPLAY_HDR = Widget_Button(BASE_AX_SGM_CSV_LOAD,  $
      UNAME='BUT_AX_SGM_CSV_LOAD_MORE' ,XOFFSET=484 ,YOFFSET=6  $
      ,SCR_XSIZE=112 ,SCR_YSIZE=25 ,/ALIGN_CENTER ,VALUE='Display more')

  TEXT_AX_SGM_CSV_LOAD_HEADER = Widget_Text(BASE_AX_SGM_CSV_LOAD,  $
      UNAME='TEXT_AX_SGM_CSV_LOAD_HEADER' ,XOFFSET=1 ,YOFFSET=60  $
      ,SCR_XSIZE=603 ,SCR_YSIZE=140 ,/SCROLL ,/EDITABLE ,XSIZE=20  $
      ,YSIZE=1)

 TEXT_AX_SGM_CSV_LOAD_LASTROW_TITLE = Widget_Text(BASE_AX_SGM_CSV_LOAD,  $
      UNAME='TEXT_AX_SGM_CSV_LOAD_X_AXIS_TITLE' ,XOFFSET=0 ,YOFFSET=200  $
      , VALUE=[ 'LastRow displayed ' ])

 TEXT_AX_SGM_CSV_LOAD_LASTROW = Widget_Text(BASE_AX_SGM_CSV_LOAD,  $
      UNAME='TEXT_AX_SGM_CSV_LOAD_LASTROW' ,XOFFSET=125 ,YOFFSET=200  $
      ,SCR_XSIZE=26 ,SCR_YSIZE=20, /ALL_EVENTS ,VALUE=['1'])

 TEXT_AX_SGM_CSV_LOAD_ROWS_TITLE = Widget_Text(BASE_AX_SGM_CSV_LOAD,  $
      UNAME='TEXT_AX_SGM_CSV_LOAD_ROWS_TITLE' ,XOFFSET=170 ,YOFFSET=200  $
      ,SCR_XSIZE=68 ,SCR_YSIZE=18 ,VALUE=[ 'rows to skip '])

 TEXT_AX_SGM_CSV_LOAD_ROWS = Widget_Text(BASE_AX_SGM_CSV_LOAD,  $
      UNAME='TEXT_AX_SGM_CSV_LOAD_ROWS' ,XOFFSET=250 ,YOFFSET=200  $
      ,SCR_XSIZE=26 ,SCR_YSIZE=20 ,/EDITABLE ,/ALL_EVENTS ,VALUE=[  $
      strtrim(string(mcf_rows),2)] ,XSIZE=15 ,YSIZE=1)

; ----------  only turn on when this routine is called from aXis2000
  BUT_AX_SGM_CSV_LOAD_ALL = Widget_Button(BASE_AX_SGM_CSV_LOAD,  $
      UNAME='BUT_AX_SGM_CSV_LOAD_ALL' ,XOFFSET=465 ,YOFFSET=265  $
      , SENSITIVE = read_all_sens $
      ,SCR_XSIZE=70 ,SCR_YSIZE=30 ,/ALIGN_CENTER ,VALUE='READ ALL')

  BUT_AX_SGM_CSV_LOAD_OK = Widget_Button(BASE_AX_SGM_CSV_LOAD,  $
      UNAME='BUT_AX_SGM_CSV_LOAD_OK' ,XOFFSET=420 ,YOFFSET=265  $
      ,SCR_XSIZE=40 ,SCR_YSIZE=30 ,/ALIGN_CENTER ,VALUE='OK')

  BUT_AX_SGM_CSV_LOAD_CANCEL = Widget_Button(BASE_AX_SGM_CSV_LOAD,  $
      UNAME='BUT_AX_SGM_CSV_LOAD_CANCEL' ,XOFFSET=550 ,YOFFSET=265  $
      ,SCR_XSIZE=50 ,SCR_YSIZE=30 ,/ALIGN_CENTER ,VALUE='Cancel')


; ----------- force user to select file ------------
	if keyword_set(file) then begin
		fname = file
	endif else begin
		fname = pickfile2(Title='Select file', Filter = '*.csv')
		if fname EQ '' then return
		mcf_lastfile = fname(0)	; ensure the filename is saved for future use
		print, 'New filename is ', mcf_lastfile
	endelse
t = findfile(fname, count=t_count)
if t_count NE 0 then begin			; file exists - is it readable ?
	openr, iunit, fname, /get_lun
	text = ''
	readf, iunit, text

; - assumes first line in file has definition of columns; seperate into each item
; - the Step scan data file format differs - has a comma at the beginning so there is an additional symbol is needed
	if strmid(text,0,1) EQ ',' then text = '#'+text  ; KLUGE to get around this
	WIDGET_CONTROL,TEXT_AX_SGM_CSV_LOAD_HEADER, Set_value = text, /append
	i_header = 0
	col_header = strarr(1)
	while strpos(text,',') GT 0 do begin
		col_header(i_header) = strmid(text,0,strpos(text,','))
		text = strmid(text, strpos(text,',')+1)
		col_header = [col_header, '']
		i_header = i_header + 1
	endwhile
;	this routine misses the last column header - extend by 1
;	print, text
	if strlen(text) GT 0 then begin
;		col_header = [col_header, '']
		col_header(i_header)=text
	endif
	print, '# of columns = ', n_elements(col_header)
	print, col_header

; - Read first 6 lines 		-- use modified first line with '#' as first character
	for i = 2, 6 do begin
		readf, iunit, text
		WIDGET_CONTROL,TEXT_AX_SGM_CSV_LOAD_HEADER, Set_value = text, /append
	endfor
	close, iunit & free_lun, iunit
	t = ax_name(fname)
	file_path = t(0)
	file_short=t(1)
	mcf_ext = t(2)
	WIDGET_CONTROL,TEXT_AX_SGM_CSV_LOAD_PATH, Set_value = file_path
	WIDGET_CONTROL,TEXT_AX_SGM_CSV_LOAD_FNAME, Set_value = file_short
	WIDGET_CONTROL,TEXT_AX_SGM_CSV_LOAD_EXT, Set_value = mcf_ext
	mcf_lastfile = fname		; save for future use if valid text file
	mcf_lastrow = 6
endif
WIDGET_CONTROL,TEXT_AX_SGM_CSV_LOAD_LASTROW, Set_value = strtrim(string(mcf_lastrow),2)

; ------- Droplist of the column labels  used for selection of column
;         for energy, signal, Io ---------------------------
; set to current value if sgm_energy, sgm_signal, sgm_io, sgm_mode are defined
;	if n_elements(sgm_energy) LE 0 then sgm_energy = 0
AX_SGM_CSV_ENERGY = Widget_Droplist(BASE_AX_SGM_CSV_LOAD, /DYNAMIC_RESIZE,  $
       VALUE = col_header,  XOFFSET=2 ,YOFFSET=230,  $
        UNAME='AX_SGM_CSV_ENERGY', TITLE='Energy ')
Widget_Control, AX_SGM_CSV_ENERGY, SET_DROPLIST_SELECT= sgm_energy

; if n_elements(sgm_signal) LE 0 then sgm_signal = 7
AX_SGM_CSV_SIGNAL = Widget_Droplist(BASE_AX_SGM_CSV_LOAD, /DYNAMIC_RESIZE,  $
       VALUE = col_header,  XOFFSET=130 ,YOFFSET=230,  $
       UNAME='AX_SGM_CSV_SIGNAL', TITLE='Signal')
Widget_Control, AX_SGM_CSV_SIGNAL, SET_DROPLIST_SELECT= sgm_signal

; if n_elements(sgm_io) LE 0 then sgm_io = 3
AX_SGM_CSV_IO = Widget_Droplist(BASE_AX_SGM_CSV_LOAD, /DYNAMIC_RESIZE,  $
       VALUE = col_header,  XOFFSET=240 ,YOFFSET=230,  $
       UNAME='AX_SGM_CSV_IO', TITLE='Io ')
Widget_Control, AX_SGM_CSV_IO, SET_DROPLIST_SELECT= sgm_io

; ------ droplist of modes for selection of processing
; if n_elements(sgm_mode) LE 0 then sgm_mode = 0
SGM_mode_list =['Signal','Yield (S/Io)', 'OD ln(Io/S)']
AX_SGM_CSV_MODE = Widget_Droplist(BASE_AX_SGM_CSV_LOAD, /DYNAMIC_RESIZE,  $
       VALUE = SGM_mode_list,  XOFFSET=340 ,YOFFSET=230,  $
      UNAME='AX_SGM_CSV_MODE', TITLE='Mode ')
Widget_Control, AX_SGM_CSV_MODE, SET_DROPLIST_SELECT= sgm_mode

  SDF_ID = BASE_AX_SGM_CSV_LOAD

  Widget_Control, /REALIZE, BASE_AX_SGM_CSV_LOAD

  XManager, 'BASE_AX_SGM_CSV_LOAD', BASE_AX_SGM_CSV_LOAD

end
;-----------------------------------------------------------------
;
; Empty stub procedure used for autoloading.
;
function AX_SGM_CSV_LOAD, file = file, ext = ext, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_
@axis_com
@ax_sgm_csv_com

mcf = ' '
if keyword_set(file) then begin
	BASE_AX_SGM_CSV_LOAD, GROUP_LEADER=wGroup, file=file, ext = ext, _EXTRA=_VWBExtra_
endif else begin
	BASE_AX_SGM_CSV_LOAD, GROUP_LEADER=wGroup,ext = ext, _EXTRA=_VWBExtra_
endelse

return, mcf

end
