; $Id: split_string.pro,v 1.2 1997/02/02 06:07:50 loo Exp $ 
; 
;+ 
; NAME: 
;       SPLIT_STRING 
; 
; PURPOSE: 
;       This function splits a string into its delimited components. 
; 
; CATEGORY: 
;       String processing. 
; 
; CALLING SEQUENCE: 
;       Result = SPLIT_STRING(Input) 
; 
; INPUTS: 
;       Input:     The string to be split. The string should consist of 
;                  multiple words separated by a user specified delimiter. 
; 
; KEYWORD PARAMETERS: 
;       DELIMITER: A string containing the delimiter between words. 
;                  The default delimiter is a single space. Also, if the 
;                  default delimiter is used, multiple spaces are compressed 
;                  to a single space before parsing. 
; 
; OUTPUTS: 
;       Returns each word of the input string as one element in the 
;       output string array, Result. 
; 
; EXAMPLE: 
;       If the input string is: 
;         input = 'Hello, world' 
; 
;       then the command sequence: 
;         result = split_string(input, DELIMITER=', ') 
; 
;       produces the result: 
;         IDL> print, result(0) 
;         Hello 
;         IDL> print, result(1) 
;         world 
; 
;       Note that the delimiter in this case includes a space. 
; 
; WRITTEN BY: 
;       Billy W. Loo, Jr. 
;       Bioengineering Graduate Group, UCSF / UCB 
;       School of Medicine, UCD 
;       Lawrence Berkeley National Lab 
;       June, 1996 
; 
; MODIFICATION HISTORY: 
; 
; $Log: split_string.pro,v $ 
; Revision 1.2  1997/02/02 06:07:50  loo 
; * Corrected bug that occurs if delimiter is first character in string. 
; 
; Revision 1.1  1996/06/09  07:56:00  loo 
; Initial revision 
; 
;- 
 
FUNCTION SPLIT_STRING, input, DELIMITER=delimiter 
  ;Return to main level on error. 
  ;ON_ERROR, 1 
 
  ;Check that input is a scalar string. 
  sz = SIZE(input) 
  IF (sz(sz(0)+1) NE 7) OR (sz(sz(0)+2) NE 1) THEN $ 
    MESSAGE, 'Input must be scalar string.' 
 
  ;Check if delimiter is specified and is of type string. 
  IF NOT(KEYWORD_SET(DELIMITER)) THEN BEGIN 
    delimiter = ' ' 
    in_string = STRCOMPRESS(input) 
  ENDIF $ 
 
  ELSE BEGIN 
    sz = SIZE(delimiter) 
    IF (sz(sz(0)+1) NE 7) OR (sz(sz(0)+2) NE 1) THEN $ 
      MESSAGE, 'Delimiter must be scalar string.' 
 
    ;Transfer input to temporary variable. 
    in_string = TEMPORARY(input) 
    transfer_flag = 1 
  ENDELSE 
 
  ;Some initializations. 
  ilength = STRLEN(in_string) ;Length of input string. 
  dlength = STRLEN(delimiter) ;Length of delimiter. 
  break_pos = -dlength        ;Positions of word breaks. 
  curr_pos = 0                ;Current search position. 
 
  ;Search through string. 
  WHILE (curr_pos LT ilength) DO BEGIN 
    curr_pos = STRPOS(in_string, delimiter, curr_pos) 
 
    ;Record break position if delimiter found. 
    IF (curr_pos GE 0) THEN BEGIN 
      break_pos = [break_pos, curr_pos] 
      curr_pos = curr_pos + dlength 
    ENDIF $ 
 
    ;Exit if no more delimiters found. 
    ELSE BEGIN 
      break_pos = [break_pos, ilength] 
      curr_pos = ilength 
    ENDELSE 
  ENDWHILE 
 
  ;Create array to hold result. 
  n_words = N_ELEMENTS(break_pos) - 1 
  result = STRARR(n_words) 
 
  ;Fill in result array. 
  FOR i = 1, n_words DO BEGIN 
    start = break_pos(i-1) + dlength  
    wlength = break_pos(i) - start 
    result(i-1) = STRMID(in_string, start, wlength) 
  ENDFOR 
 
  ;If there is only one word, convert to scalar. 
  IF (n_words EQ 1) THEN result = result(0) 
 
  ;Transfer data back to input, if necessary. 
  IF KEYWORD_SET(transfer_flag) THEN input = TEMPORARY(in_string) 
 
  RETURN, result 
END ;split_string 
