; Copyright(c) 2014 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		IRMAP_LOAD
;
;LAST CHANGED: ----------------------------------- 21-Jul-14 (aph)
;
; PURPOSE:
;   This prodecure reads an FTIR map from a set of csv files
;  consisting of IR spectra at an array of points
;  The data are converted to an aXis binary stack (*.ncb)
;
; CATEGORY:
;	utility; stand-alone or used through aXis2000
;
; CALLING SEQUENCE:
;IRMAP_LOAD, file=file, filter=filter, xcol=xcol, ycol=ycol, $
;         multi=multi, skip=skip, int_factor=int_factor, typ=typ, NOF=nof, _extra=e
;
; INPUTS: only by keyword
;
; KEYWORDS:
; FILE		filename (*.map - array dimensions)
; FILTER  	filter extension (*.map)
; XCOL		number of x-axis (if multi-column)
; YCOL		number of y-axis (if multi-column)
; SKIP		number of header lines to skip  (if multi-column)
; MULTI		returns number of columns -1 = number of spectra
; NOF		no filter (*.*)
; _EXTRA	pass through keywords
;
; OUTPUTS:
;	Result is  a stack (x,y,wavenumber)
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;
; MODIFICATION HISTORY:
; (18-dec-08 aph) first version  - data sent by Paul Dumas (Soleil)
; (21-Jul-14 aph) corrected header
;-

pro irmap_load,  file=file, filter=filter, xcol=xcol, ycol=ycol, $
           skip=skip, NOF=nof, int_factor=int_factor, multi=multi, _extra=e
@axis_com
@stack_process_com
COMMON volume_data, image_stack
@bsif_com
on_error,2

s=0
fltr='*.map'
if keyword_set(nof) then fltr='*'
if keyword_set(filter) then fltr=filter
if not keyword_set(file) then $
     file = pickfile2(/READ, FILTER=fltr, /LPATH, DEFPATH=defpath, TITLE = TITLE)
if strlen(file) LE 0 THEN return    ; bail-out if no filename

openr, iunit, file, /get_lun, error=err
if (err NE 0) then begin
	file = pickfile2(/READ, FILTER=fltr, /LPATH, DEFPATH=defpath)
	openr, iunit, file, /get_lun
endif

t = ax_name(file)
path = t(0)
root = ''
readf, iunit, root
print, 'name of spectral files: ', root
title = ''
readf, iunit, title
print, 'TITLE: ', title
readf, iunit, nimg
readf, iunit, ncols, nrows
ncols = fix(ncols)
nrows = fix(nrows)
print, 'Stack dimensions: ', ncols, ' columns by ', nrows, ' rows'
close,iunit
free_lun,iunit

; ------ read in the first spectrum to allow 3d array to be properly dimensioned
fn = path + root + '0001.csv'
read_csv,fn,ftir	; ftir is a 2-d array, with (0,*) = wavenumber; (1,*) = intensity
t=size(ftir) & print, t
n_energies = t(2)
image_stack = fltarr(ncols, nrows, n_energies)
ev = ftir(0,*)
mask=10000
ftir_data = fltarr(nimg, n_energies)
for i= 0, nimg-1 do begin
	t=mask+i+1
	ts=strtrim(string(t),2)
	fn = path + root + strmid(ts,1) + '.csv'
	print, 'reading spectrum ', i+1
	read_csv,fn,ftir
	ftir_data(i,*) = ftir(1,*)
endfor
help, ftir_data
image_stack=reform(ftir_data, ncols, nrows, n_energies)

filename_ev_msec_list = strarr(n_energies)
for i = 0, n_energies-1 do begin
	filename_ev_msec_list(i) = root + ' ' + string(ev(i))
endfor
x_start = 0  & x_stop = ncols
y_start = 0  & y_stop = nrows

; --------- ask user what file name they want - default name is 'root'
sname=pickfile2(/write, file=root +'.ncb', title = 'name of binary stack file')
stack_wb, sname
s=1
return
end
