; Copyright (c) 1998-2024 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_NANOMAX_IMAGE_PROCESS.PRO
;;
;LAST CHANGED: ----------------------------------- 	08 July 2024 (aph)

; PURPOSE:
;	This function reads an image from a single NanoMAX h5 file  when the Eiger camera is used.
;	The Eiger signal is the average (or sum ?) of the full camera image
;   The Io is the upstream ion chamber signal, which is very noisy (due to beam position changes?)
;
; CATEGORY:
;	    Runs stand alone or inside aXis2000.
;
; CALLING SEQUENCE:
;	READ_NANOMAX_IMAGE_PROCESS = READ_NANOMAX_IMAGE(file = file, path=path, silent = silent, verbose=verbose, group = group)
;
; INPUTS: none required - keywords used

;
; KEYWORDS:
;	FILE	filename.(*.h5)
;	Path	path
;	Silent	if set do not track in the log on xis2000
;	Verbose	print optinal messages
;	Group	(axis200_id; to allow call from aXis2000)
; OUTPUTS:
;
;
; COMMON BLOCKS:
; AXIS_COM    standard set of common blocks
; stack_process_com
; COMMON volume_data, image_stack
; bsif_com


; MODIFICATION HISTORY:
; (26-Mar-24 aph) first version
; (04-Apr-24 aph) give choice of I, Io or I/Io
; (18-Apr-24 aph) this code only works for image files in the process folder of MArch2024 NanoMAX (also no E)
; (28-Jun-24 aph) parameterize the scale factors used to keep within 32-bit limit
; (04-Jul-24 aph) exploring why stxm_map_eiger gives different results than [ln(map_It_Eiger/map_Io)]
;          in my IDL code but the same in python (Maik Kahnt. 27-Jun - 5 July 2024) np.log is ln
; (08-Jul-24 aph) automate conversion of stxm_map_Eiger to normalized transmission
;       remove offset (handling large # problem by ln(map_It_Eiger/map_Io); exp()
;-


FUNCTION  read_nanomax_image_process, file = file, path=path, silent = silent, verbose=verbose, group = group
@axis_com
@stack_process_com
COMMON volume_data, image_stack
@bsif_com

on_error,2

null = 0

if n_elements(group) EQ 0 then begin
	if widget_info(/active) then axis_on=1 else axis_on=0
endif else axis_on = 1

axis_log, 'reading NanoMAX images in format of Process folder (stxm_######.h5)'

; ------------ select NanoMAX  *.h5 file(s)

if NOT keyword_set(file) then begin
	if n_elements(last_image) EQ 0 then last_image = ' '
    file = dialog_pickfile(title='Select NanoMAX  *.h5 file',filter=['*.h5'],  file=last_image, path=DefPath, get_path=folder_path)
	last_image = file
	DefPath=folder_path    ; --- force path for writing to be the source folder
endif
t = ax_name(file)
DefPath=t(0)
if NOT keyword_set(silent) then print, 'default path set to  ', DefPath

; ----------- single image read-in -------
if strlen(file) EQ 0 then RETURN, null
if NOT keyword_set(silent) then print, 'NanoMAX_data_file = ' ,file
t =ax_name(file)
fileshort = t(1)


a = h5_parse(file, /read_data)

; ---------- ask user to supply the photon energy ==> not in the files in process folder

if n_elements(last_energy) EQ 0 then last_energy = 0.
energy = get_num(prompt = 'Photon energy', val=last_energy, group=group) ;a.info.energies._data
last_energy = energy

; ---------- get scan number
scan_num= a.scan._data
print, scan_num

; read x and y values

x = a.x._data
y = a.y._data
nx = n_elements(x)
ny = n_elements(y)


; ---- have user choose <S>TXM, <O> Ion chamber or <R>atio
if n_elements(last_type) EQ 0 then last_type =' S'
if axis_on EQ 1 then begin
	type = get_text(prompt = "Type =  <E>iger, <O> ion chamber or <S>TXM_map_Eiger ", val = last_type, group=group)
endif else begin
	type = get_text(prompt = "Type =  <E>iger, <O> ion chamber or <S>TXM_map_Eiger ", val = last_type)
endelse
type = strtrim(strupcase(type),2)
if type NE 'E' AND  type NE 'O' AND type NE 'S'  then begin
	axis_log, 'Type must be E, O or S'
	return, null
endif
last_type = type



;  read the STXM (integrated Eiger mage / Io) and seperate signals (Eiger map and  IonChamber signals
d =  a.map_it_eiger._data

d_ion = a.map_I0._data

; -------
; s =a.stxm_map_eiger._data
s = double(a.stxm_map_eiger._data)  ; "stxm_map_eiger" = log (map_It_eiger / map_I0)  ; log in np is ln

;s = 1.0e17*exp(double(a.stxm_map_eiger._data))		; a.stxm_map_eiger._data is  log(map_It_eiger/ map_I0)
;					where the eiger and ion chamber spignals have been corrected for 0-flux background

a1 = 2.e8		; scale factor for Ion_chamber
b1 = 5.e9		; scale factor for Eiger-map

CASE type OF
; ----- reading transmitted I
;
	'E' :  begin
		d =  d/a1		; get values to a 32 bit range (4e9)
		dl = fileshort + '  map_It_Eiger'
		END
	'O' :  begin
		d =  d_ion*b1	; get values to a 32 bit range (4e9)
		dl = fileshort + '  map_Io'
		END

	'S' : begin
		d = s
		dl = fileshort + '  stxm_map_eiger'   ;STXM = -ln(map_It_Eiger/map_Io)'
		d = d - (min(d) + 1.e-2*(max(d)-min(d)));   remove offset
		d = exp(d)
		END

ENDCASE


; ------ dwell not in the h5 file
; print, 'Dwell (s) ', h(9)
 dwell = 50.  ;  typically 20 or 50 or 100 ms was used in March 2024 run

; nx = get_num(prompt='Number of x-points ', val = nx)  ;,  group = Axis_ID)
; ny = get_num(prompt='Number of y-points ', val = ny)  ; ,  group = Axis_ID)
;energy = get_num(prompt='Photon energy ', val = energy)  ;  group = Axis_ID)

;  ----- populate parameters in bsif_com
xmin = x(0)
xmax = x(n_elements(2) -1)
ymin = y(0)
ymax = y(n_elements(2) -1)
x_start = xmin
x_stop  = xmax
x_step = (xmax - xmin) /(nx-1)
y_start = ymin
y_stop  = ymax
y_step = (ymax - ymin) /(ny-1)
n_rows = nx
n_cols = ny
xl = '  ' +  string(energy) + ' eV.   dwell = ' +  string(dwell) + ' s'
yl = ' '
;  -- verify
if NOT keyword_set(silent) then print, 'Opening file ', dl
if NOT keyword_set(silent) then print, " type , xmin , xmax, nx, ymin, ymax, ny, dwell(s) ", type , xmin , xmax, nx, ymin, ymax, ny, dwell


; ------- redimension the immage
;if (nx*ny) eq n_elements(d) then begin
;	d = reform (d, nx, ny)
;endif else begin
;	axis_log,'nx, ny dimensions do not match size of data array'
;	return, null
;endelse

; flip (x --> x,  y--> -y
d = rotate(d, 7)
; y = reverse(y)
img = {t:'2d', x:x, y:y, xl:xl, yl:yl, e:energy, d:d, dl:dl}


return, img

end