; Copyright (c) 2000-2025 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_CLS_XPEEM
;
;LAST CHANGED: ----------------------------------- 28 Nov 2025 (aph)
;
;PURPOSE:
;	This function converts a set of tiff files (unsigned integer) and associated parameters
;   into an axis2000 file (*.axb for single images; (*.dat, *.ncb) for stacks)
;
;CATEGORY:
;	STAND ALONE: utility & may be called from aXis2000
;
;CALLING SEQUENCE:
;	Result = READ_CLS_XPEEM(in_file=in_file, filter=filter, hdf5_only=hdf5_only)
;
;CALLED FROM AXIS:
;
;INPUTS:
;  binary *.hdf5 file containing instrument paramters; in Nov2025 version only the energies are used
;               written by the acquisition code for the CLS XPEEM;derived from read_lox, read cSTXM)
;				VERSION 1.0
; folder containing tiff files

;KEYWORDS
;	IN_FILE - name of *.hdf5	file
;	FILTER  -  extension (*.hdf5)
; 	hdf5_only - return the structure generated by the IDL command h5_parse(filename,/read_data)
;				works for both IDL 6.3 and 9.x
;COMMON BLOCKS:
;	@AXIS_COM	standard set of common blocks
;	@bsif_com
; @stack_process_com
; @nexus_common
;
;MODIFICATION HISTORY:
; (20-Nov-25 aph) First version adapted from read_als_stxm (Jun2025)
; 		h5_parse(filename,/read_data)
; (27-Nov-25 aph)  auto-square; aborted part-stacks; correct (x,y) for FoV, binning etc
; (28-Nov-25 aph) flip images to match display in  pyPEEM;  auto launch stack_process


FUNCTION READ_CLS_XPEEM, in_file=in_file, filter=filter, hdf5_only=hdf5_only

@AXIS_COM
@bsif_com
@stack_process_com
@nexus_com

on_error, 2

COMMON volume_data, image_stack

; check to see if run stand-alone (axis_on = 0) or from aXis2000 (axis_on = 1)
if n_elements(group) EQ 0 then begin
	if widget_info(/active) then axis_on=1 else axis_on=0
endif else axis_on = 1

; ----------------------
if not keyword_set(filter) then filter="*.hdf5"
if n_elements(def_path) EQ 0 then def_path = 'c:\aXis2000'  ; in case start without prior call to aXis2000
if n_elements(in_file) EQ 0 then begin
	in_file=pickfile2(LPATH=lp, filter=filter, TITLE="Read CLS XPEEM *.hdf5 file",  $
         LFILE=lf, group = group, _extra=e)
endif
t = ax_name(in_file)
fileshort=t(1)
s=0

print, ' Reading CLS XPEEM data from ', fileshort
print, '--------------------------------------------'

;  use h5_parse to convert an hdf5 file into an IDL structure
;  this works in IDL 6.3 and later versions
a = h5_parse(in_file, /READ_DATA)

if keyword_set(hdf5_only) then return, a

; ----------- if a keyword indicating the type of data (image, point spectra, stack etc)
;				gets added to the *.hdf5 file,this will be used to use the appropriate code for each type
;type = a.entry0.counter0.stxm_scan_type._data
;print, fileshort, ' is an CLS XPEEM scan of type: ', type


; --- read in energy from the *.hdf5 file
ev = a.entry1.data.energy._data
n_e_all = n_elements(ev)
; ----------- note, if a stack acquisition is cancelled the values of the energies in the hdf5 file from that point on are 0.0
; use this to read part stacks
tmp = where(ev NE 0, count)
if count NE 0 then eV= ev(tmp)
n_e = n_elements(ev)
if n_e LT n_e_all then axis_log, 'PART STACK: expected # of images is ' + strtrim(string(n_e_all),2) + ' Actual # is ' + strtrim(string(n_e),2)
  axis_log, 'STACK: # of images ' + strtrim(string(n_e),2)
n_img = n_e

IF n_e EQ 1 then print, 'Single image at E(eV) = ', ev(0)
IF n_e GT 1 then print, ' STACK: ', strtrim(string(n_e),2), ' energies from ', ev(0), '  to  ', ev(n_e-1)

if n_elements(ev) LE 0 then begin
  print, 'no energies found'
  return, s
endif

;  so if energy = 1, there is only 1 image
;  maybe that is enough to differentiate individual images and stacks

;===================  READING a CLS XPEEM stack =================
; define limiting paramters
; ram = 12	; jab10, windows computer has 16 Gb
; ram = get_num(prompt = 'size of RAM (Gb)', Val = ram, Title ='available memory limit')  ;,, group = Axis_ID))

; =================  read FIRST image
; print, 'path to hdf5 file ',  lpath
t = ax_name(in_file)
file_core = t(1) +'_'

stackpath = strmid(in_file, 0, rstrpos(in_file,'.hdf5')) + ax_sep()
print, ' Path to hdf5 file', stackpath
cd , stackpath
lpath = stackpath

; --------------  read one tiff file to set sizes and co-ordinates
fname = stackpath + file_core + '0001.tiff'
print, 'FIRST file is ', fname

first = read_tiff(fname)
; help, first
; ---------- identify centre (assumed to be (nx/2, ny/2)
t = size(first)
x_centre = t(1)/2
y_centre = t(2)/2
x_pix=t(1)			; use this to derive a pixel size corrected for additional biining
; ----- determine clipping parameters based on size of first
if x_pix GT 2600 then pre_bin = 1
if x_pix LT  700 then pre_bin = 4
if x_pix GT 800 and x_pix LT 2000 then pre_bin = 2
x_gap = 350/pre_bin
y_gap = 80/pre_bin
offset = 699/pre_bin		; this is for the full camera image (2688, 2020)  scale by any bin factor (2,4) applied in acquisition
tx_min = x_centre - offset - 1
tx_max = x_centre + offset
ty_min = y_centre - offset - 1
ty_max = y_centre + offset

test =  size(first, /dimensions)
print, 'BEFORE TRIM, tiff size is ', test

first = first(tx_min:tx_max, ty_min:ty_max)

print, 'TRIM to X: ', tx_min,' - ', tx_max, '.  Y: ',  ty_min, ' - ', ty_max
test =  size(first, /dimensions)
print, 'AFTER TRIM, tiff size is ', test



;trim = dialog_message('trim to centre ?', /question, /cancel, /center, title = 'trim')
;if trim EQ 'Yes' then begin
;	test =  size(first, /dimensions)
;int, 'tiff size is ', test
;	if test(0) EQ 2688 then begin
;		tx_min = 340   & tx_max = 2399	; no camera binning
;		ty_min = 100   & ty_max = 2159
;	endif
;	if test(0) EQ 1344 then begin
;		tx_min = 172   & tx_max = 120  ; X2 camera binning
;		ty_min = 50    & ty_max = 1080
;	endif
;	if test(0) EQ 672  then begin
;		tx_min = 86   & tx_max = 600  ; X4 camera binning
;		ty_min = 25    & ty_max = 540
;	endif
;
;	first = first(tx_min:tx_max, ty_min:ty_max)
;	print, 'TRIM to X: ', tx_min,' - ', tx_max, '.  Y: ',  ty_min, ' - ', ty_max
;	test =  size(first, /dimensions)
;	print, 'AFTER TRIM, tiff size is ', test
;endif

; -------------- reduce size, by a binning factor  (each image reduced by square of this factor)
if n_elements(xpeem_bin) EQ 0 then xpeem_bin = 4
if axis_on EQ 1 then begin
   xpeem_bin = get_num(prompt='CLS-XPEEM bin factor', val=xpeem_bin, group = axis_ID)
endif else	xpeem_bin = get_num(prompt='CLS-XPEEM bin factor', val=xpeem_bin)


; NB  if use rebin need, to restrict use of rebin to integer binning & use congrid for fractional cases
full = size(first, /dimensions)
;d_bin = intarr(2)
;d_bin(0) = full(0)/bin  &  d_bin(1) = full(1)/bin
ncol = full(0)/xpeem_bin
nrow = full(1)/xpeem_bin

; print, 'reducing dimensions from ', full, ' to ' ncol, nrow
; ------------- bin with rebin s restricted to integer factors;  congrid is slower but does not require integer factors
; first = rebin(first, d_bin)
first = congrid(first, ncol, nrow, cubic = -0.5)

; ======== ask user for information (that should be in *.hdf5 file !!) FoV, Dwell, Polarization . . .
if n_elements(fov) EQ 0 then fov=10
if axis_on EQ 1 then begin
   fov = get_num(prompt='Field_of_View', val=fov, group = axis_ID)
endif else fov = get_num(prompt='Field_of_View', val=fov)


; --------------- define  x,y-co-ordinates needed to write stack

xy_dim = size(first, /dimensions)
nx = xy_dim(0)
ny = xy_dim(1)
; -------  correct for FoV and additional binning
x_step = xpeem_bin*fov/x_pix
print, 'pixel size (nm) based on FoV & xpeem_bin is ', x_step/1000.
xr = indgen(nx)*x_step
yr = indgen(ny)*x_step
xmin = min(xr)    & xmax = max(xr)
ymin = min(yr)    & ymax = max(yr)
xstep = (xmax-xmin)/nx
ystep = (ymax-ymin)/ny

; ----------- generate container for the stack
image_stack = fltarr(nx, ny, n_e-1)
; help, image_stack
print, ' '
; 2-Dec-2025    ADD Progress bar
; read in stack
for i = 0, n_e-2 do begin
;	t = bar[IDLitIMessaging::]ProgressBar('converting XPEEM stack',Percent = i/n_e-2)
	num = strcompress(string(10001 + i))
	num = strmid(num,2,5)
	fname = stackpath + file_core + num +'.tiff'
;	print, 'reading ', file_core + num +'.tiff'
  	t = read_tiff(fname)
  	t = t(tx_min:tx_max, ty_min:ty_max)
	tmp = congrid(t, ncol, nrow, cubic = -0.5)
	tmp = rotate(tmp,7)		; ;   flip the image X1 = X0  and Y1 = -Y0)
  	image_stack(*,*,i) = tmp
endfor
; t = bar[IDLitIMessaging::]ProgressBar(/shutdown)

print, file_core, ' stack  read-in'
t = size(image_stack)
print,'Stack size   is ', t(5)/1.0e6 , '  Mb'

  ;  interpolate the data to match the number of requested samples
  ; NB initially I used image_data as a variable but that is in bsif_com where it can be changed
  ; also I suspect 'image' is a reserved word, so change to Rimage
; if n_e eq 1 then begin         ; interpolate single image
;    Rimage = congrid(ddd, nx, ny)
;
;  endif else begin               ; interpolate stack if recorded with rectilinear scan
;      help, ddd                  ; should not chage spiral scan data
;       image_stack = fltarr(nx, ny, n_e)
;       for i = 0, n_e-1 do begin
;          t = ddd(*,*,i)
;          tmp = congrid(t, nx, ny)
;          image_stack(*,*,i) = tmp
;       endfor
;       if type EQ 'Image' then $
;          print, 'stack interpolated to (nx,ny) ', strtrim(string(nx),2),' x ',strtrim(string(ny),2)
;       help, image_stack
;  endelse
;

 ; ----- populate parameters in bsif_com
  x_start = xmin
  x_stop  = xmax
  y_start = ymin
  y_stop  = ymax
  n_rows = nx
  n_cols = ny
  dwell = 1000		;  eventually ask for this value and the dimension of the image

;; preferred X-label and stack label formats
;; filename_ev_msec_list(i) = filenm  + ' : '+$
;  ;     strtrim(string(ev(i),format='(f10.2)'),2)+' eV '+$
;  ;     strtrim(string(dwell,format='(f7.2)'),2)+' msec'
;
;  xl = '  ' +  strtrim(string(ev(i),format='(f10.2)'),2) + ' eV.   dwell = ' $
;             + strtrim(string(dwell,format='(f7.2)'),2)  + ' ms'
;  print, 'X-label = ', xl
;  yl = 'ALS COSMIC *.stxm' + type
;  print, 'y-label = ', yl

; ==================================
;  so now should have x, y, ddd, and ev values matching
if n_e GT 1 then begin
  t = size(image_stack)
  print, 'STACK:  nx ',nx,'  ny ', ny, ' image  size ', t(1),' x ', t(2), '  # energies ', n_e
endif else begin
  print, 'Single image (nx,ny) ', nx, ny, '  energy (eV) = ', ev
endelse

; ==================================  SINGLE IMAGE ====================
;if n_elements(ev) EQ 1 then begin
;    ; ========  save as a single image =======================
;
;    filename = a.ENTRY0.COUNTER0._file
;    t = ax_name(filename)
;    dl = t(1)
;    x = xr    ; as read from *.stxm, all variables are doubles
;    y = yr
;    ; preferred X-label and stack label formats
;    ; filename_ev_msec_list(i) = filenm  + ' : '+$
;    ;     strtrim(string(ev(i),format='(f10.2)'),2)+' eV '+$
;    ;     strtrim(string(dwell,format='(f7.2)'),2)+' msec'
;
;    xl = '  ' +  strtrim(string(ev(0),format='(f10.2)'),2) + ' eV.   dwell = ' $
;      + strtrim(string(dwell,format='(f7.2)'),2)  + ' ms'
;    print, 'X-label = ', xl
;    yl = 'ALS COSMIC *.stxm' + type
;    print, 'y-label = ', yl
;    tmp = {t:'2d', x:xr, y:yr, xl:xl, yl:yl, e: ev(0), d: Rimage, dl:dl}
; ;   help, tmp
;    axis_log, 'read ALS COSMIC image ' + dl
;    return, tmp
;
;endif else begin
;  ==================================  STACK ====================
; ========  save as a stack. Stacks were acquired with both rectilinear scans and spiral scans
; TEST DATA
;
;   STXM_SCAN_TYPE = Spiral Image
; Y:\aXis-Pass-Test-data\00-READ\02-STXM-NEXUS\ALS_.stxm\NS_250525081.stxm
;
;   STXM_SCAN_TYPE = Image
; Y:\aXis-Pass-Test-data\00-READ\02-STXM-NEXUS\ALS_.stxm\NS_231029069.stxm

 ; Identify stack scan type from STXM_SCAN_TYPE
;  if a.entry0.counter0.STXM_SCAN_TYPE._data EQ 'Spiral Image' then begin
;
; NB  The DATA and X, Y co-ordinates for spiral scans are saved as
;     ENTRY0.binned_values.sample_x._data
;     different from the rectilinear scans

    print, 'Recorded stack (nx,ny,nE) = ', $
       strtrim(string(nx),2), '   ', strtrim(string(ny),2),'   ',  strtrim(string(n_e),2)

  ; generate filename_ev_msec_list
    t=ax_name(in_file)
    fileshort = t(1)
    if not keyword_Set(out_file) then out_file = t(2)
    out_file = pickfile2(filter='*.ncb', file=fileshort, title='binary stack file ', WRITE=write)
    IF strlen(out_file) EQ 0 THEN BEGIN
      axis_log, 'no output file name.'
      return, s
    ENDIF
    t=ax_name(out_file)
    fileshort_out = t(1)

    print, ' Output file: ',  fileshort_out, '.ncb'

    filename_ev_msec_list = strarr(n_e)
    ; (24-Apr-16 aph)  syntax of filename_ev_msec_list not consistent with stack_process
    ;       filename_ev_msec_list(i) = filenm  + ' : '+$
    ;     strtrim(string(ev(i),format='(f10.2)'),2)+' eV '+$
    ;     strtrim(string(dwell,format='(f7.2)'),2)+' msec'

    FOR i = 0, n_e-1 DO BEGIN
      filename_ev_msec_list(i) = fileshort_out + ' ' + strtrim(string(i),2)+' ' + string(ev(i), format = '(F10.2)') + $
        ' eV ' +  string(dwell(0), format='(F7.2)') + ' msec'
      print,  strtrim(string(i),2), '    ', filename_ev_msec_list(i)
    ENDFOR

; convert doubles to single floats
  image_stack = ax_dts(image_stack)

;  write stack to disk
    stack_wb, out_file

    t = ax_name(out_file)
    Data_File_Name = t(1)
    log_text = string(format='(i3," x",i3," stack with ",i3," energies read from ",A,".")',nx, ny, n_e, Data_File_Name)
    axis_log, log_text

    return, out_file

end
