; Stack_align.pro, Chris Jacobsen
; This routine aligns a stack.  It acts in ignorance to any previous
; alignment that was done - it just tries to align "image_stack" which
; is passed in a common block.
;
; Modified 25-mar-1998 to deal with 24 bit graphics
; Modified July 11 and 13, 1998 to deal with changes to align.pro
; Modified 28-aug-1998 to work with corrected 24 bit graphics
; Modified 31-Aug-1998 to add x_shift, y_shift to argument list, and
;   remove shift_filename!
; Modified quite a bit 12-Jul-2000.  Now the existing shift is in
;   a common block "stack_data_common".  This program works on
;   a local shift.  If the local shift is good, it is added onto
;   stack_x_shift and stack_y_shift.

PRO stack_align_common

COMMON stack_align_common, $
  stack_align_par, img_zoom, plot_rows, plot_cols, edgegauss_pixels, $
  x_shift_color_index, y_shift_color_index, white_color_index, $
  x_shift_plot_index, y_shift_plot_index, white_plot_index, $
  top_color_index, r, g, b, disp_min, disp_max, disp_gamma, $
  local_x_shift, local_y_shift, maxshift, filename_info_list, dragbox,$ 
  stack_align_idlversion
COMMON stack_data_common, image_stack, stack_x_shift, stack_y_shift

END

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO stack_align_sensitive,sensitive=sensitive,desensitive=desensitive
  
@bsif_common
COMMON stack_align_common
COMMON stack_data_common 
 
IF keyword_set(desensitive) THEN BEGIN
    widget_control, stack_align_par.sobel_edge_label, sensitive = 0
    widget_control, stack_align_par.roberts_edge_label, sensitive = 0
    widget_control, stack_align_par.no_edge_label, sensitive = 0
    widget_control, stack_align_par.precede_match_label, sensitive = 0
    widget_control, stack_align_par.constant_match_label, sensitive = 0
    widget_control, stack_align_par.alignto_list_label, sensitive = 0
    widget_control, stack_align_par.dragbox_label, sensitive = 0
    widget_control, stack_align_par.xcor_peak_label, sensitive = 0
    widget_control, stack_align_par.xcor_cm_label, sensitive = 0
    widget_control, stack_align_par.xcor_maxshift_label, sensitive = 0
    widget_control, stack_align_par.xcor_edgegauss_label, sensitive = 0
    widget_control, stack_align_par.doit_label, sensitive = 0
    widget_control, stack_align_par.skipit_label, sensitive = 0
ENDIF ELSE BEGIN
    widget_control, stack_align_par.sobel_edge_label, sensitive = 1
    widget_control, stack_align_par.roberts_edge_label, sensitive = 1
    widget_control, stack_align_par.no_edge_label, sensitive = 1
    widget_control, stack_align_par.precede_match_label, sensitive = 1
    widget_control, stack_align_par.constant_match_label, sensitive = 1
    widget_control, stack_align_par.alignto_list_label, sensitive = 1
    widget_control, stack_align_par.dragbox_label, sensitive = 1
    widget_control, stack_align_par.xcor_peak_label, sensitive = 1
    widget_control, stack_align_par.xcor_cm_label, sensitive = 1
    widget_control, stack_align_par.xcor_maxshift_label, sensitive = 1
    widget_control, stack_align_par.xcor_edgegauss_label, sensitive = 1
    widget_control, stack_align_par.doit_label, sensitive = 1
    widget_control, stack_align_par.skipit_label, sensitive = 1
    
    IF (stack_align_par.do_edge EQ 0) THEN BEGIN
        widget_control, stack_align_par.sobel_edge_label, set_button = 0
        widget_control, stack_align_par.roberts_edge_label, set_button = 0
        widget_control, stack_align_par.no_edge_label, set_button = 1
    ENDIF ELSE IF (stack_align_par.do_edge EQ 1) THEN BEGIN
        widget_control, stack_align_par.no_edge_label, set_button = 0
        widget_control, stack_align_par.roberts_edge_label, set_button = 0
        widget_control, stack_align_par.sobel_edge_label, set_button = 1
    ENDIF ELSE BEGIN
        widget_control, stack_align_par.no_edge_label, set_button = 0
        widget_control, stack_align_par.sobel_edge_label, set_button = 0
        widget_control, stack_align_par.roberts_edge_label, set_button = 1
    ENDELSE
    
    IF (stack_align_par.constant_match EQ 1) THEN BEGIN
        widget_control, stack_align_par.precede_match_label, set_button = 0
        widget_control, stack_align_par.constant_match_label, set_button = 1
    ENDIF ELSE BEGIN
        widget_control, stack_align_par.constant_match_label, set_button = 0
        widget_control, stack_align_par.precede_match_label, set_button = 1
    ENDELSE
    
    IF (stack_align_par.align_cm EQ 0) THEN BEGIN
        widget_control, stack_align_par.xcor_cm_label, set_button = 0
        widget_control, stack_align_par.xcor_peak_label, set_button = 1
    ENDIF ELSE BEGIN
        widget_control, stack_align_par.xcor_peak_label, set_button = 0
        widget_control, stack_align_par.xcor_cm_label, set_button = 1
    ENDELSE
ENDELSE

END 

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO stack_align_findshifts
  
@bsif_common
COMMON stack_align_common
COMMON stack_data_common
  
  stack_align_sensitive,/desensitive
  
  svec = size(image_stack)
  n_files=svec(3)
  
  sobel = 0
  roberts = 0
  IF (stack_align_par.do_edge EQ 1) THEN sobel = 1
  IF (stack_align_par.do_edge EQ 2) THEN roberts = 1
  
  wset,stack_align_par.plot_win
  plot_min = -10.
  plot_max = +10.
  plot,local_x_shift,/nodata,yrange=[plot_min,plot_max],$
    xtitle='File #', ytitle='Shift (pixels)',$
    color=white_plot_index
  xyouts,3,6,/data,color=x_shift_plot_index,'X'
  xyouts,3,-6,/data,color=y_shift_plot_index,'Y'
  
  start_time = systime(1.)
  IF (stack_align_par.constant_match EQ 0) THEN BEGIN
      ref_fft = image_stack[*,*,0]
  ENDIF ELSE BEGIN
      ref_fft = image_stack[*,*,stack_align_par.constant_image_index]
  ENDELSE
  
  ;; Do we want to wipe out all but a selected, high contrast
  ;; region as our reference?
  IF ((dragbox(0) NE 0) AND (dragbox(1) NE 0) AND $
      (dragbox(2) NE 0) AND (dragbox(3) NE 0)) THEN BEGIN
      xwidth = abs(dragbox(2)-dragbox(0)) > (2*edgegauss_pixels+2)
      yheight = abs(dragbox(3)-dragbox(1)) > (2*edgegauss_pixels+2)
      xleft = fix(0.5*(dragbox(0)+dragbox(2))-0.5*xwidth)
      ybot = fix(0.5*(dragbox(3)+dragbox(1))-0.5*yheight)
      xleft = (xleft > fix(0.9+edgegauss_pixels)) < $
        fix(n_cols-1-xwidth)
      ybot = (ybot > fix(0.9+edgegauss_pixels)) < $
        fix(n_rows-1-yheight)
      xright = fix(xleft+xwidth) < fix(n_cols-1)
      ytop = fix(ybot+yheight) < fix(n_rows-1)
      dragbox_image = ref_fft(xleft:xright,ybot:ytop)
      edgegauss,dragbox_image,edgegauss_pixels,/zero_edge
      ref_fft = fltarr(n_cols,n_rows)
      ref_fft(xleft:xright,ybot:ytop) = dragbox_image
  ENDIF
  
  ;; What we're doing here is to prepare the reference
  ;; image in a way that is made to be indentical to
  ;; how the sequence of images will be prepared.  We
  ;; don't care about the fact that "align" will assume
  ;; that its first argument is an image which has already
  ;; be Fourier transformed.
  this_image = ref_fft
  img_align,this_image,ref_fft,edgegauss=edgegauss_pixels,$
    /fftpass,sobel = sobel, roberts=roberts,maxshift=5
  ;; 18-nov-2001 CJJ stop using /meanfill
  
  widget_control,hourglass=1
  
  FOR i_file = 0,(n_files-1) DO BEGIN
      
      this_image = image_stack[*,*,i_file]
      
      ;; By saying /fftpass we assume that the first image has
      ;; already been Fourier transformed, and also we get
      ;; "this_image" back Fourier transformed.
      ;; 18-nov-2001 CJJ stop using /meanfill
      img_align,ref_fft,this_image,this_x_shift,this_y_shift,$
            shifted_image, sobel = sobel, roberts=roberts, $
            maxshift=(maxshift-1), /fftpass, $
            cm=stack_align_par.align_cm,edgegauss=edgegauss_pixels,$
            xcorimg_win = stack_align_par.diff_image_win, $
            xcorimg_zoom = img_zoom
      IF (stack_align_par.constant_match EQ 0) THEN BEGIN
          ref_fft = this_image
          IF (i_file EQ 0) THEN BEGIN
              ;; We really wouldn't expect any shift at all, but
              ;; we'll use the output of align anyway...
              local_x_shift[0] = this_x_shift
              local_y_shift[0] = this_y_shift
          ENDIF ELSE BEGIN
              ;; We must accumulate the shifts
              local_x_shift[i_file] = local_x_shift[i_file-1]+this_x_shift
              local_y_shift[i_file] = local_y_shift[i_file-1]+this_y_shift
          ENDELSE
      ENDIF ELSE BEGIN
          local_x_shift(i_file) = this_x_shift
          local_y_shift(i_file) = this_y_shift
      ENDELSE
      
      widget_control, stack_align_par.this_imagetitle_label, $
                      set_value = $
                      filename_info_list(stack_align_par.constant_image_index)
      widget_control, stack_align_par.this_imagecount_label,$
                      set_value=' ('+strtrim(string(i_file+1),2)+' of '+$
                      strtrim(string(n_files),2)+'; shift was ['+$
                      strtrim(string(this_x_shift,format='(f10.1)'),2)+','+$
                      strtrim(string(this_y_shift,format='(f10.1)'),2)+'])'
      wset,stack_align_par.this_image_win
      percent_image = 100.*shifted_image/max(shifted_image)
      byte_image = byte(0.5+float(top_color_index)*$
                        (( (((percent_image-disp_min) / $
                             float(disp_max-disp_min))^disp_gamma) $
                           >0.)<1.))<byte(top_color_index)
      tv,rebin(byte_image,n_cols*img_zoom,n_rows*img_zoom,/sample),0,0
      
      IF (i_file GT 0) THEN BEGIN
          wset,stack_align_par.plot_win
          plots,[i_file-1,i_file],local_x_shift(i_file-1:i_file),/data,$
                color=x_shift_plot_index
          plots,[i_file-1,i_file],local_y_shift(i_file-1:i_file),/data,$
                color=y_shift_plot_index
          ;; This flushes the graphics buffer
          wait,0.0001
      ENDIF
  ENDFOR
  
  plot_min = min([min(local_x_shift),min(local_y_shift)])<(-1.)
  plot_max = max([max(local_x_shift),max(local_y_shift)])>(1.)
  plot,local_x_shift,/nodata,yrange=[plot_min,plot_max],$
       xtitle='File #', ytitle='Shift (pixels)',$
       color=white_plot_index
  xyouts,3,(plot_min+0.7*(plot_max-plot_min)),/data,$
         color=x_shift_plot_index,'X'
  xyouts,3,(plot_min+0.3*(plot_max-plot_min)),/data,$
         color=y_shift_plot_index,'Y'
  oplot,findgen(n_files),local_x_shift,$
        color=x_shift_plot_index
  oplot,findgen(n_files),local_y_shift,$
        color=y_shift_plot_index
  
  widget_control,hourglass=0
  
  end_time = systime(1.)
  total_time = end_time - start_time
  minutes = fix(total_time/60)
  seconds = total_time-60.*minutes
  IF (minutes GT 0) THEN BEGIN
      print,'Elapsed time for stack alignment: '+$
            strtrim(string(minutes),2)+' minutes '+$
            strtrim(string(seconds,format='(f10.1)'),2)+' seconds.
  ENDIF ELSE BEGIN
      print,'Elapsed time for stack alignment: '+$
            strtrim(string(seconds,format='(f10.1)'),2)+' seconds.
  ENDELSE
  
  stack_align_par.findshifts_complete = 1
  
  stack_align_sensitive,/desensitive
  widget_control, stack_align_par.doit_label, $
                  set_value = 'Save new alignment; go to stack_analyze'
  widget_control, stack_align_par.skipit_label, $
                  set_value = 'Do not save new alignment; go to stack_analyze'
  widget_control, stack_align_par.doit_label, sensitive = 1
  widget_control, stack_align_par.skipit_label, sensitive = 1
  
  return
END

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; This routine both applies the shifts to image_stack,
;; and it also adds local_x_shift and local_y_shift into
;; stack_x_shift and stack_y_shift.

PRO stack_align_applyshifts
  
@bsif_common
COMMON stack_align_common
COMMON stack_data_common

  svec = size(image_stack)
  n_files=svec(3)
  
  widget_control,hourglass=1
  stack_x_shift = stack_x_shift+local_x_shift
  stack_y_shift = stack_y_shift+local_y_shift
  
  print,'Applying shifts to all images'
  FOR i_file=0,(n_files-1) DO BEGIN
      p = [local_x_shift(i_file),0.,1.,0.]
      q = [local_y_shift(i_file),1.,0.,0.]
      this_image = image_stack[*,*,i_file]
      ;; 18-nov-2001 CJJ stop using missing because it seems not to
      ;; work any more ?
      ;; missing = total(this_image)/(float(n_cols)*float(n_rows))
      IF (stack_align_idlversion LT 4.0) THEN BEGIN
          image_stack[*,*,i_file] = $
            poly_2d(this_image, p, q, 1)
      ENDIF ELSE BEGIN
          image_stack[*,*,i_file] = $
            poly_2d(this_image, p, q, 2, cubic=-0.5)
      ENDELSE
  ENDFOR 
  widget_control,hourglass=0
  
  return
END

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO stack_align_event,event
  
@bsif_common
COMMON stack_align_common
COMMON stack_data_common
  
IF ((event.id EQ stack_align_par.skipit_label) OR $
    (tag_names(event,/structure_name) EQ 'WIDGET_KILL_REQUEST')) THEN BEGIN
    widget_control, stack_align_par.main_base,/destroy
    IF (stack_align_par.calling_base NE 0) THEN BEGIN
        widget_control,stack_align_par.calling_base,sensitive=1
        stk_anl_refresh
    ENDIF
ENDIF ELSE BEGIN 
    CASE event.id OF
        stack_align_par.sobel_edge_label: BEGIN
            stack_align_par.do_edge = 1
            widget_control, stack_align_par.no_edge_label, set_button = 0
            widget_control, stack_align_par.roberts_edge_label, set_button = 0
            widget_control, stack_align_par.sobel_edge_label, set_button = 1
        END
        
        stack_align_par.roberts_edge_label: BEGIN
            stack_align_par.do_edge = 2
            widget_control, stack_align_par.no_edge_label, set_button = 0
            widget_control, stack_align_par.sobel_edge_label, set_button = 0
            widget_control, stack_align_par.roberts_edge_label, set_button = 1
        END
        
        stack_align_par.no_edge_label: BEGIN
            stack_align_par.do_edge = 0
            widget_control, stack_align_par.sobel_edge_label, set_button = 0
            widget_control, stack_align_par.roberts_edge_label, set_button = 0
            widget_control, stack_align_par.no_edge_label, set_button = 1
        END
        
        stack_align_par.precede_match_label: BEGIN
            stack_align_par.constant_match = 0
            widget_control, stack_align_par.constant_match_label,set_button = 0
            widget_control, stack_align_par.precede_match_label, set_button = 1
        END
        
        stack_align_par.constant_match_label: BEGIN
            stack_align_par.constant_match = 1
            widget_control, stack_align_par.precede_match_label, set_button = 0
            widget_control, stack_align_par.constant_match_label, set_button=1
        END
        
        stack_align_par.alignto_list_label: BEGIN
            stack_align_par.constant_image_index = event.index
            widget_control, stack_align_par.this_imagetitle_label, $
              set_value = $
              filename_info_list(stack_align_par.constant_image_index)
            wset,stack_align_par.this_image_win
            percent_image = $
              100.*image_stack[*,*,stack_align_par.constant_image_index] / $
              max(image_stack[*,*,stack_align_par.constant_image_index])
            byte_image = byte(0.5+float(top_color_index)*$
                              (((((percent_image-disp_min) / $
                                  float(disp_max-disp_min))^disp_gamma) $
                                >0.)<1.))<byte(top_color_index)
            tv,rebin(byte_image,n_cols*img_zoom,n_rows*img_zoom,/sample),0,0
        END
        
        stack_align_par.dragbox_label: BEGIN
            stack_align_sensitive,/desensitive
            
            wset,stack_align_par.this_image_win
            xv=[0,0]
            yv=[0,0]
            getbox,xv,yv,/device
            xleft = xv[0]/img_zoom
            xright = xv[1]/img_zoom
            ybot = yv[0]/img_zoom
            ytop = yv[1]/img_zoom
            IF (xleft GT xright) THEN BEGIN
                temp_int = xleft
                xleft = xright
                xright = temp_int
            ENDIF
            IF (ybot GT ytop) THEN BEGIN
                temp_int = ybot
                ybot = ytop
                ytop = temp_int
            ENDIF
            xleft = (xleft>0)<(n_cols-1)
            xright = (xright>0)<(n_cols-1)
            ybot = (ybot>0)<(n_rows-1)
            ytop = (ytop>0)<(n_rows-1)

            IF (((xright-xleft) GE 2) AND ((ytop-ybot) GE 2)) THEN BEGIN
                dragbox = [xleft,ybot,xright,ytop]
            ENDIF ELSE BEGIN
                dragbox = [0,0,(n_cols-1),(n_rows-1)]
            ENDELSE
            tvlct,r,g,b
            
            stack_align_sensitive,/sensitive
        END
        
        stack_align_par.xcor_peak_label: BEGIN
            ;; 0 for xcor_peak, 1 for xcor_cm
            stack_align_par.align_cm = 0
            widget_control, stack_align_par.xcor_cm_label, set_button = 0
            widget_control, stack_align_par.xcor_peak_label, set_button = 1
        END
        
        stack_align_par.xcor_cm_label: BEGIN
            ;; 0 for xcor_peak, 1 for xcor_cm
            stack_align_par.align_cm = 1
            widget_control, stack_align_par.xcor_peak_label, set_button = 0
            widget_control, stack_align_par.xcor_cm_label, set_button = 1
        END
        
        stack_align_par.xcor_maxshift_label: BEGIN
            temp_string = ''
            widget_control, stack_align_par.xcor_maxshift_label, $
              get_value = temp_string
            on_ioerror, xcor_maxshift_label_oops
            IF (strlen(temp_string(0)) EQ 0) THEN $
              goto, xcor_maxshift_label_oops
            reads,temp_string(0),maxshift
            IF (maxshift LT 1) THEN maxshift = 1
            IF (maxshift GT min([n_cols/2,n_rows/2])) THEN $
              maxshift = min([n_cols/2,n_rows/2])
            xcor_maxshift_label_oops:
            widget_control, stack_align_par.xcor_maxshift_label, $
              set_value = strtrim(string(maxshift,format='(i6)'),2)
        END
        
        stack_align_par.xcor_edgegauss_label: BEGIN
            temp_string = ''
            widget_control, stack_align_par.xcor_edgegauss_label, $
              get_value = temp_string
            on_ioerror, xcor_edgegauss_label_oops
            IF (strlen(temp_string(0)) EQ 0) THEN $
              goto, xcor_edgegauss_label_oops
            reads,temp_string(0),edgegauss_pixels
            xcor_edgegauss_label_oops:
            widget_control, stack_align_par.xcor_edgegauss_label, $
              set_value = strtrim(string(edgegauss_pixels,format='(f10.1)'),2)
        END
        
        stack_align_par.doit_label: BEGIN
            IF (stack_align_par.findshifts_complete EQ 0) THEN BEGIN
                stack_align_findshifts
            ENDIF ELSE BEGIN
                stack_align_applyshifts
                widget_control, stack_align_par.main_base,/destroy
                IF (stack_align_par.calling_base NE 0) THEN BEGIN
                    widget_control,stack_align_par.calling_base,sensitive=1
                    stk_anl_refresh
                ENDIF
            ENDELSE
        END
    ENDCASE
ENDELSE
  
END

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; For compatibility with the SLICER, this gets its
;; data from COMMON volume_data, image_stack.  

;; Considerably modified, July 2000.  You now need to pass all
;; of the following parameters.  "calling_base" should be the widget
;; base of the calling program, so it can be made insensitive.
;;
;;    stack_align,filename_list,stk_anl_par.zoom,stk_anl_par.main_base

PRO stack_align, filename_ev_msec_list, zoom, calling_base

@bsif_common
COMMON stack_align_common
COMMON stack_data_common

  stack_align_idlversion = idl_version()  

  svec = size(image_stack)
  IF (svec[0] NE 3) THEN BEGIN
      print,'Not a valid stack'
      return
  ENDIF 
  IF ((svec[1] NE n_cols) OR (svec[2] NE n_rows)) THEN BEGIN
      print,'Discrepancy between stack dimensions and n_cols, n_rows'
      return
  ENDIF
  
  n_files=svec(3)
  local_x_shift = fltarr(n_files)
  local_y_shift = fltarr(n_files)
  
  filename_info_list = filename_ev_msec_list
  dragbox=[0,0,0,0]
  
  img_zoom = 1
  svec = size(zoom)
  IF (svec(1+svec(0)) NE 0) THEN BEGIN
      img_zoom = zoom
  ENDIF
  
  edgegauss_pixels = 3.
  plot_rows = 200
  plot_cols = (n_cols*img_zoom)>300
  IF (n_tags(stack_align_par) EQ 0) THEN BEGIN
      stack_align_par = $
        { stack_align_par, $
          main_base: 0L, $
          calling_base: 0L, $
          sobel_edge_label: 0L, $
          roberts_edge_label: 0L, $
          no_edge_label: 0L, $
          do_edge: 0L, $
          constant_match_label: 0L, $
          precede_match_label: 0L, $
          constant_match: 0L, $
          constant_image_index: 0L, $
          alignto_list_label: 0L, $
          dragbox_label: 0L, $
          xcor_peak_label: 0L, $
          xcor_cm_label: 0L, $
          xcor_maxshift_label: 0L, $
          xcor_edgegauss_label: 0L, $
          align_cm: 0L, $
          doit_label: 0L, $
          skipit_label: 0L, $
          complete_base: 0L, $
          acknowledge_label: 0L, $
          this_imagetitle_label: 0L, $
          this_imagecount_label: 0L, $
          this_image_label: 0L, $
          this_image_win: 0L, $
          diff_image_label: 0L, $
          diff_image_win: 0L, $
          plot_label: 0L, $
          plot_win: 0L, $
          findshifts_complete: 0L $
        }
  ENDIF
  
  stack_align_par.calling_base = 0
  svec = size(calling_base)
  IF (svec(1+svec(0)) NE 0) THEN BEGIN
      stack_align_par.calling_base = calling_base
      widget_control, stack_align_par.calling_base, sensitive = 0
  ENDIF
  
  stack_align_par.findshifts_complete = 0
  
  stack_align_par.main_base = widget_base(title='Stack Align', /row,$
                                          /tlb_kill_request_events)
  col1 = widget_base(stack_align_par.main_base,/column)
  col2 = widget_base(stack_align_par.main_base,/column)
  
  frame = widget_base(col1,/column,/frame)
  edge_text = 'Edge enhance before aligning?'
  label = widget_text(frame,value=edge_text,xsize=(strlen(edge_text)+4))
  base = widget_base(frame,/column,/exclusive)
  stack_align_par.sobel_edge_label = $
    widget_button( base, value='Sobel')
  stack_align_par.roberts_edge_label = $
    widget_button( base, value='Roberts')
  stack_align_par.no_edge_label = $
    widget_button( base, value='None')
  
  frame = widget_base(col1,/column,/frame)
  label = widget_text(frame,value='Image to align to:')
  base = widget_base(frame,/column,/exclusive)
  stack_align_par.precede_match_label = $
    widget_button( base, value='Each preceding image') ;
  stack_align_par.constant_match_label = $
    widget_button( base, value='Constant reference image:') ;
  stack_align_par.alignto_list_label = $
    widget_list( frame, xsize=40, ysize=5, $
                 value = filename_info_list )
  
  frame = widget_base(col1,/column,/frame)
  label = widget_text(frame,value='How to align')
  stack_align_par.dragbox_label = $
    widget_button(frame, value='Select a subregion')
  base = widget_base(frame,/row,/frame,/exclusive)
  stack_align_par.xcor_peak_label = $
    widget_button( base, value='X-cor peak') ;
  stack_align_par.xcor_cm_label = $
    widget_button( base, value='X-cor CM') ;
  row = widget_base( frame, /row )
  label = widget_text( row, value='X-cor pixel shift limit: ')
  maxshift = min([n_cols/2,n_rows/2,30])
  stack_align_par.xcor_maxshift_label = $
    widget_text( row, xsize=8, /editable, /frame, $
                 value=strtrim(string(maxshift,format='(i6)'),2))
  row = widget_base( frame, /row )
  label = widget_text( row, value='Pixels for edgegauss: ')
  stack_align_par.xcor_edgegauss_label = $
    widget_text( row, xsize=8, /editable, /frame, $
                 value=strtrim(string(edgegauss_pixels,format='(f10.1)'),2))
  
  frame = widget_base(col1,/column,/frame)
  stack_align_par.doit_label = $
    widget_button(frame,value='Start aligning',/dynamic_resize)
  stack_align_par.skipit_label = $
    widget_button(frame,value='Skip alignment',/dynamic_resize)
  
  row = widget_base(col2,/row)
  base = widget_base(row,/column)
  stack_align_par.this_imagetitle_label = $
    widget_text(base,value='Image ',xsize=40)
  stack_align_par.this_imagecount_label = $
    widget_text(base,value='',xsize=40)
  stack_align_par.this_image_label = $
    widget_draw(base,xsize=n_cols*img_zoom,ysize=n_rows*img_zoom,$
                retain=2)
  
  base = widget_base(row,/column)
  label = widget_text(base,value='Cross correlation')
  stack_align_par.diff_image_label = $
    widget_draw(base,xsize=n_cols*img_zoom,ysize=n_rows*img_zoom,retain=2)
  
  stack_align_par.plot_label = $
    widget_draw(col2,xsize=plot_cols,ysize=plot_rows,retain=2)
  
  widget_control, stack_align_par.main_base, /realize
  
  widget_control,stack_align_par.this_image_label,get_value=window
  stack_align_par.this_image_win = window
  
  widget_control,stack_align_par.diff_image_label,get_value=window
  stack_align_par.diff_image_win = window
  
  widget_control,stack_align_par.plot_label,get_value=window
  stack_align_par.plot_win = window
  
  ;; Added 28-aug-98 CJJ
  IF (!d.n_colors EQ 16777216L) THEN  BEGIN
      device,decomposed=0
  ENDIF
  
  x_shift_color_index = !d.table_size-1
  y_shift_color_index = !d.table_size-2
  white_color_index = !d.table_size-3
  top_color_index = !d.table_size-4
  
  r = byte(0.5+255.*findgen(!d.table_size)/float(top_color_index))
  g = r
  b = r
  r(x_shift_color_index) = 255
  g(x_shift_color_index) = 0
  b(x_shift_color_index) = 0
  r(y_shift_color_index) = 0
  g(y_shift_color_index) = 255
  b(y_shift_color_index) = 0
  r(white_color_index) = 255
  g(white_color_index) = 255
  b(white_color_index) = 255
  
  white_plot_index = white_color_index
  x_shift_plot_index = x_shift_color_index
  y_shift_plot_index = y_shift_color_index
  
  wset,stack_align_par.this_image_win
  tvlct,r,g,b
  
  wset,stack_align_par.diff_image_win
  tvlct,r,g,b
  
  wset,stack_align_par.plot_win
  tvlct,r,g,b
  
  stack_align_par.do_edge = 0
  widget_control, stack_align_par.sobel_edge_label, set_button = 0
  widget_control, stack_align_par.roberts_edge_label, set_button = 0
  widget_control, stack_align_par.no_edge_label, set_button = 1
  
  stack_align_par.constant_match = 1
  widget_control, stack_align_par.precede_match_label, set_button = 0
  widget_control, stack_align_par.constant_match_label, set_button = 1
  
  ;; 0 for xcor_peak, 1 for xcor_cm
  stack_align_par.align_cm = 0
  widget_control, stack_align_par.xcor_cm_label, set_button = 0
  widget_control, stack_align_par.xcor_peak_label, set_button = 1
  
  disp_min = 0.
  disp_max = 110.
  disp_gamma = 1.
  stack_align_par.constant_image_index = 0
  widget_control, stack_align_par.this_imagetitle_label, $
    set_value = filename_info_list(stack_align_par.constant_image_index)
  wset,stack_align_par.this_image_win
  percent_image = $
    100.*image_stack[*,*,stack_align_par.constant_image_index] / $
    max(image_stack[*,*,stack_align_par.constant_image_index])
  byte_image = byte(0.5+float(top_color_index)*$
                    (( (((percent_image-disp_min) / $
                         float(disp_max-disp_min))^disp_gamma) $
                       >0.)<1.))<byte(top_color_index)
  tv,rebin(byte_image,n_cols*img_zoom,n_rows*img_zoom,/sample),0,0
  
  widget_control, stack_align_par.doit_label, $
    set_value='Start aligning'
  widget_control, stack_align_par.skipit_label, $
    set_value='Skip alignment'
  stack_align_sensitive,/sensitive
  
  IF (stack_align_par.calling_base EQ 0) THEN BEGIN
      xmanager, 'stack_align', stack_align_par.main_base
  ENDIF ELSE BEGIN
      widget_control
      xmanager, 'stack_align', stack_align_par.main_base, $
        group_leader = stack_align_par.calling_base
  ENDELSE
  
  return
END

