;; align.pro, Chris Jacobsen, Stony Brook
;;
;; Modified 25-mar-1998 to deal with 24 bit color.
;;   If you provide the white, red, and green colors, then
;;   it is assumed that the color table is preloaded.

;; Significantly modified on July 11, 1998.  Eliminated
;; the correl option, and eliminated the /peak keyword because
;; it is now the default.  Fit the three pixels about the
;; peak to z=a+bx*cx^2 and z=a+by+cy^2 for subpixel location.
;; Improve the center of mass peak location.
;;
;; Modified July 13, 1998 to let one do a sequence with 2N+1
;; FFTs rather than 3N by using the /fftpass option.  If this
;; option is selected, then it is expected that the "image1"
;; you provide is already transformed, and the transformed
;; version of image2 is returned to you (for use as a
;; pre-transformed image1 in a subsequent call to ALIGN).
;
;; Modified 28-aug-1998 to deal with 24 bit color by demanding
;; device,decomposed=0

PRO img_align,image1,image2,xshift,yshift,image2_shifted,$
          fftpass=fftpass,sobel=sobel,roberts=roberts,edgegauss=edgegauss,$
          cm=cm,maxshift=maxshift,$
          meanfill=meanfill,$
          xcorimg_win=xcorimg_win,xcorimg_zoom=xcorimg_zoom,$
          xplot_win=xplot_win,yplot_win=yplot_win,$
          debug=debug,help=help

  IF (keyword_set(help) OR (n_params() eq 0)) THEN BEGIN
      print,'align,image1,image2,xshift,yshift,image2_shifted'
      print,'  Modifiers to input images: /sobel or /roberts, edgegauss_sigma='
      print,'    (if you simply do /edgegauss, you will get edgegauss_sigma=4.)'
      print,'  Constraints on peak finding: /cm, maxshift='
      print,'  Constraints on shifted image: /meanfill'
      print,'  Display the correlation: xcorimg_win=, xcorimg_zoom='
      return
  ENDIF

  IF (n_elements(sobel) EQ 0) THEN sobel = 0
  IF (n_elements(roberts) EQ 0) THEN roberts = 0
  IF (n_elements(edgegauss) EQ 0) THEN BEGIN
      edgegauss_sigma = 0.
  ENDIF ELSE IF (float(edgegauss) LE 1.) THEN BEGIN
      edgegauss_sigma = 4.
  ENDIF ELSE BEGIN
      edgegauss_sigma = edgegauss
  ENDELSE
  IF (n_elements(cm) EQ 0) THEN cm = 0
  IF (n_elements(meanfill) EQ 0) THEN meanfill = 0
  IF (n_elements(xcorimg_zoom) EQ 0) THEN xcorimg_zoom = 1
  IF (xcorimg_zoom LT 1) THEN xcorimg_zoom = 1

  svec=size(image1)
  IF (svec(0) NE 2) THEN BEGIN
      print,'image1 has '+strtrim(string(svec(0)),2)+$
        ' rather than 2 dimensions'
      return
  ENDIF
  nx1=svec(1)
  ny1=svec(2)

  IF (n_elements(maxshift) EQ 0) THEN maxshift = fix(min(0.8*[nx1/2,ny1/2]))

  ;; Because tri_fit will look as far as adjacent pixels, we need to
  ;; take one away because the nearest neighbor distance is allowed.
  ;; CJJ 11-June-2001
  maxshift = maxshift-1

  svec=size(image2)
  IF (svec(0) NE 2) THEN BEGIN
      print,'image2 has '+strtrim(string(svec(0)),2)+$
        ' rather than 2 dimensions'
      return
  ENDIF
  nx2=svec(1)
  ny2=svec(2)

  IF ((nx1 NE nx2) OR (ny1 NE ny2)) THEN BEGIN
      print,'Images must match in size'
      return
  ENDIF
  nx=nx1
  ny=ny1
  xcenter = 0
  ycenter = 0

  ;; We make our own copy of the input image and call it fft in anticipation
  ;; of what we will do to it later.
  IF (sobel NE 0) THEN BEGIN
      IF (not keyword_set(fftpass)) THEN BEGIN
          fft1 = sobel(image1)
      ENDIF
      fft2 = sobel(image2)
  ENDIF ELSE IF (roberts NE 0) THEN BEGIN
      IF (not keyword_set(fftpass)) THEN BEGIN
          fft1 = roberts(image1)
      ENDIF
      fft2 = roberts(image2)
  ENDIF ELSE BEGIN
      IF (not keyword_set(fftpass)) THEN BEGIN
          fft1 = image1
      ENDIF
      fft2 = image2
  ENDELSE
  IF (edgegauss_sigma NE 0.) THEN BEGIN
      IF (not keyword_set(fftpass)) THEN BEGIN
          edgegauss,fft1,edgegauss_sigma,dc1,/zero_edge
      ENDIF
      edgegauss,fft2,edgegauss_sigma,dc2,/zero_edge
  ENDIF

  ;; Did we get the FFT of the image passed instead of the
  ;; image itself?
  IF keyword_set(fftpass) THEN BEGIN
      fft1 = image1
  ENDIF ELSE BEGIN
      fft1 = shift(temporary(fft1),nx/2,ny/2)
      fft1 = fft(fft1,-1,/overwrite)
      fft1 = shift(temporary(fft1),nx/2,ny/2)
  ENDELSE

  fft2 = shift(temporary(fft2),nx/2,ny/2)
  fft2 = fft(fft2,-1,/overwrite)
  fft2 = shift(temporary(fft2),nx/2,ny/2)

  ;; For these calculations, we are quite willing to mess with
  ;; fft1 but we don't want to modify fft2 since it will supply
  ;; fft1 in a subsequent call to ALIGN
  fft1 = conj(temporary(fft1))
  fft1 = fft2*temporary(fft1)

  fft1 = shift(temporary(fft1),nx/2,ny/2)
  fft1 = fft(fft1,1,/overwrite)
  fft1 = shift(temporary(fft1),nx/2,ny/2)
  fft1 = abs(temporary(fft1))

  ;; print,'Align has maxshift=',maxshift
  ;; Find the maximum
  xleft = (nx/2-maxshift)>0
  xright = (xleft+2*maxshift)<(nx-1)
  ybot = (ny/2-maxshift)>0
  ytop = (ybot+2*maxshift)<(ny-1)
  dummy = max(fft1(xleft:xright,ybot:ytop),index)
  ycenter = ybot+(index / (xright-xleft+1))
  xcenter = xleft+(index mod (xright-xleft+1))
  x_integer_center = xcenter
  y_integer_center = ycenter

  IF keyword_set(debug) THEN BEGIN
      print,'Maximum value is at ['+$
        strtrim(string(xcenter),2)+','+$
        strtrim(string(ycenter),2)+']'
  ENDIF

  IF (cm NE 0) THEN BEGIN
      max_fft1 = max(fft1,min=min_fft1)
      cm_scale=1./(max_fft1-min_fft1)
      cm_threshold=0.5
      cm_image=fft1(xleft:xright,ybot:ytop)
      cm_indices=where(((cm_image-min_fft1)*cm_scale) GE cm_threshold,n_cm)
      IF (n_cm GE 4) THEN BEGIN
          xpositions=xleft+(cm_indices mod (xright-xleft+1))
          ypositions=ybot+(cm_indices/(xright-xleft+1))
          inverse_mass_total=1./total(cm_image(cm_indices))
          xcenter=total(xpositions*cm_image(cm_indices))*inverse_mass_total
          ycenter=total(ypositions*cm_image(cm_indices))*inverse_mass_total
          IF keyword_set(debug) THEN BEGIN
              print,'Center of mass is at ['+$
                strtrim(string(xcenter,format='(f10.2)'),2)+','+$
                strtrim(string(ycenter,format='(f10.2)'),2)+']'
          ENDIF
      ENDIF
  ENDIF ELSE BEGIN
      xpts=xcenter+[-1,0,1]
      ypts=fft1((xcenter-1):(xcenter+1),ycenter)
      tri_fit,xpts,ypts,xfit,xpeak
      xpts=ycenter+[-1,0,1]
      ypts=fft1(xcenter,(ycenter-1):(ycenter+1))
      tri_fit,xpts,ypts,yfit,ypeak
      xcenter=xpeak
      ycenter=ypeak
      IF keyword_set(debug) THEN BEGIN
          print,'Linear+parabolic center is at ['+$
            strtrim(string(xcenter,format='(f10.2)'),2)+','+$
            strtrim(string(ycenter,format='(f10.2)'),2)+']'
      ENDIF
  ENDELSE

  IF (n_elements(xplot_win) NE 0) THEN BEGIN
      wset,xplot_win
      wshow,xplot_win
      xpts=x_integer_center+findgen(13)-6
      zpts=fft1((x_integer_center-6):(x_integer_center+6),y_integer_center)
      plot,xpts,zpts,xtitle='X center',psym=2,ystyle=16
      oplot,xcenter+[0.,0.],!y.crange,linestyle=1
  ENDIF

  IF (n_elements(yplot_win) NE 0) THEN BEGIN
      wset,yplot_win
      wshow,yplot_win
      ypts=y_integer_center+findgen(13)-6
      zpts=fft1(x_integer_center,(y_integer_center-6):(y_integer_center+6))
      plot,ypts,zpts,xtitle='Y center',psym=2,ystyle=16
      oplot,ycenter+[0.,0.],!y.crange,linestyle=1
  ENDIF

  IF (n_elements(xcorimg_win) NE 0) THEN BEGIN
      wset,xcorimg_win
      wshow,xcorimg_win

      red_index = !d.table_size-1
      green_index = !d.table_size-2
      white_index = !d.table_size-3
      r = byte(0.5+(255.*findgen(!d.table_size)/ $
                    float(!d.table_size-3))<255.)
      g = r
      b = r
      r(green_index:red_index) = byte([0,255])
      g(green_index:red_index) = byte([255,0])
      b(green_index:red_index) = byte([0,0])

      ;; Added 28-aug-98 CJJ
      if (!d.n_colors eq 16777216L) then begin
          device,decomposed=0
      endif

      tvlct,r,g,b

      red_plot_index = red_index
      green_plot_index = green_index
      white_plot_index = white_index

      byte_image = fft1
      byte_max = max(byte_image,min=byte_min)
      ;; byte_min = byte_min>(byte_max-alog(1.e3))
      byte_image = bytscl(byte_image,min=byte_min,max=byte_max,top=white_index)
      IF (xcorimg_zoom GT 1) THEN BEGIN
          tv,rebin(byte_image,(xcorimg_zoom*nx),(xcorimg_zoom*ny),/sample)
      ENDIF ELSE BEGIN
          tv,byte_image
      ENDELSE
      plots,(nx/2)*xcorimg_zoom+[-8,-4],$
        (ny/2)*xcorimg_zoom+[0,0],$
        /device,color=green_plot_index
      plots,(nx/2)*xcorimg_zoom+[4,8],$
        (ny/2)*xcorimg_zoom+[0,0],$
        /device,color=green_plot_index
      plots,(nx/2)*xcorimg_zoom+[0,0],$
        (ny/2)*xcorimg_zoom+[-8,-4],$
        /device,color=green_plot_index
      plots,(nx/2)*xcorimg_zoom+[0,0],$
        (ny/2)*xcorimg_zoom+[4,8],$
        /device,color=green_plot_index
      plots,[fix(0.5+xcenter*xcorimg_zoom)],$
        [fix(0.5+ycenter*xcorimg_zoom)],/device,$
        color=red_plot_index,psym=6
  ENDIF

  fft1 = 0
  xshift = xcenter - double(nx/2)
  yshift = ycenter - double(ny/2)

  ;; Only shift the image if the user wants it
  IF (n_params() GE 5) THEN BEGIN
      IF ((abs(xshift) GT 0.02) OR (abs(yshift) GT 0.02)) THEN BEGIN
          p = [xshift,0.,1.,0.]
          q = [yshift,1.,0.,0.]
          IF (meanfill NE 0) THEN BEGIN
              missing = total(image2)/float(nx*ny)
              IF (idl_version() LT 4.0) THEN BEGIN
                  image2_shifted = $
                    poly_2d(image2,p,q,1,missing=missing)
              ENDIF ELSE BEGIN
                  image2_shifted = $
                    poly_2d(image2,p,q,2,cubic=-0.5,missing=missing)
              ENDELSE
          ENDIF ELSE BEGIN
              IF (idl_version() LT 4.0) THEN BEGIN
                  image2_shifted = $
                    poly_2d(image2,p,q,1)
              ENDIF ELSE BEGIN
                  image2_shifted = $
                    poly_2d(image2,p,q,2,cubic=-0.5)
              ENDELSE
          ENDELSE
      ENDIF ELSE BEGIN
          image2_shifted = image2
      ENDELSE

  ENDIF

  ;; Save fft2 for use as fft1 for a subsequent call to ALIGN
  IF keyword_set(fftpass) THEN BEGIN
      image2 = fft2
  ENDIF

  return
END

