; Copyright (c) 1998-2024 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_NANOMAX_IMAGE_EARLY.PRO
;;
;LAST CHANGED: ----------------------------------- 	03 Apr 2024 (aph)

; PURPOSE:
;	This function reads an image from a single NanoMAX h5 file  when the DIODE was used !!
;
; CATEGORY:
;	    Runs stand alone or inside aXis2000.
;
; CALLING SEQUENCE:
;	READ_NANOMAX_IMAGE = READ_NANOMAX_IMAGE(file = file, path=path, silent = silent, verbose=verbose, group = group)
;
; INPUTS:
;	FILE	filename.
;
; KEYWORDS:
;	KEY1:  none
;
; OUTPUTS:
;
;
; COMMON BLOCKS:
; AXIS_COM    standard set of common blocks
; stack_process_com
; COMMON volume_data, image_stack
; bsif_com


; MODIFICATION HISTORY:
; (26-Mar-24 aph) first version
; (03-Apr-24 aph) give choice of I, Io or I/Io

;-


FUNCTION  read_nanomax_image_early, file = file, path=path, silent = silent, verbose=verbose, group = group
@axis_com
@stack_process_com
COMMON volume_data, image_stack
@bsif_com

on_error,2

null = 0

; ------------ select NanoMAX  *.h5 file(s)

if NOT keyword_set(file) then begin
    file = dialog_pickfile(title='Select NanoMAX  *.h5 file',filter=['*.h5'], path=DefPath, get_path=folder_path)
	DefPath=folder_path    ; --- force path for writing to be the source folder
endif
t = ax_name(file)
DefPath=t(0)
if NOT keyword_set(silent) then print, 'default path set to  ', DefPath

; ----------- single image read-in -------
if strlen(file) EQ 0 then RETURN, null
if NOT keyword_set(silent) then print, 'NanoMAX_data_file = ' ,file
t =ax_name(file)
fileshort = t(1)



a = h5_parse(file, /read_data)

; --------- get dimensions of image from the header string
; ----------- TYPICAL string      npointflyscan  sx   -8   0   40   sy   0.4   4.0   18   0.1
hdr =  a.entry.description._data
h =strsplit(hdr,/extract)
;' print, 'HEADER   ', h
type = h(0)
xmin = float(h(2))
xmax = float(h(3))
nx =   fix(h(4))+1
ymin = float(h(6))
ymax = float(h(7))
ny =   fix(h(8))+1
; print, 'Dwell (s) ', h(9)
dwell = float(h(9))
; -- verify
if NOT keyword_set(silent) then print, 'Opening file ', fileshort
if NOT keyword_set(silent) then print, " type , xmin , xmax, nx, ymin, ymax, ny, dwell(s) ", type , xmin , xmax, nx, ymin, ymax, ny, dwell


; header content  (type, x_lbl, x_start, x_end, x_points, y_lbl, y_start, y_end, y_points)


; ------------ generate (x,y), energies, and dwell values from *.h5 file

; nx = get_num(prompt='Number of x-points ', val = nx)  ;,  group = Axis_ID)
; ny = get_num(prompt='Number of y-points ', val = ny)  ; ,  group = Axis_ID)
;energy = get_num(prompt='Photon energy ', val = energy)  ;  group = Axis_ID)


energy = reform(a.entry.snapshots.pre_scan.energy._data)
x_start = xmin
x_stop  = xmax
x_step = (xmax - xmin) /(nx-1)
x = findgen(nx)*x_step + xmin
y_start = ymin
y_stop  = ymax
y_step = (ymax - ymin) /(ny-1)
y = findgen(ny)*y_step + ymin
n_rows = nx
n_cols = ny
xl = '  ' +  string(energy) + ' eV.   dwell = ' +  string(dwell) + ' s'
yl = ' '


; ---- have user choose I, (Io) Ion chamber or  I/Io
type = get_text(prompt = "Type = <I>-transmitted, <O> ion chamber or <R> ratio (I/O) ", val = last_type, group=group)
type = strtrim(strupcase(type),2)
if type NE 'I' AND  type NE 'O' AND type NE 'R'  then begin
	axis_log, 'Type must be I, O or R'
	return, null
endif
last_type = type



;  read the transmitted (integrated Eiger mages) and ionChamber signals
d =  a.entry.measurement.alba2._3._data

d_ion = a.entry.measurement.alba2._1._data

CASE type OF
; ----- reading transmitted I
;
	'I' :  d =  d

	'O' :  d =  d_ion

	'R' :  d = d/d_ion

ENDCASE

; ------- redimension the immage
if (nx*ny) eq n_elements(d) then begin
	d = reform (d, nx, ny)
endif else begin
	axis_log,'nx, ny dimensions do not match size of data array'
	return, null
endelse

; flip (x --> x,  y--> -y
d = rotate(d, 7)
y = reverse(y)
img = {t:'2d', x:x, y:y, xl:xl, yl:yl, e:energy, d:d, dl:fileshort}


return, img

end