; Copyright (c) 1998-2017 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_DUV_POSITIONS.PRO
;
;LAST CHANGED: ----------------------------------- 19-Jul-2017(aph)
;
;PURPOSE:
;	read laser parameters and (x,y) co-ordinates for DUV Raman/fluorescence stacks
;
;CATEGORY:
;	 read in utility
;
;CALLING SEQUENCE:
;	Result = READ_DUV_POSITIONS(FILNENAME)
;
;CALLED FROM AXIS:
;	READ_DUV_STACK
;
; INPUTS:
; 	FNAME  	filename.
;
; KEYWORDS: none
;
; OUTPUTS:
;	Results (x-array, y-array) transferred via duv_com  common block
;
;
; KEYWORDS: none
;
;COMMON BLOCKS:
;	DUV_COM
;
; EXAMPLE - file used to develop this function
;LaserInfo
;40	500	12.5	12.5	961
;
; are 40 and 500 the offsets relative to the origin of the sub_image ?
;
;Points			THESE ARE CO_ORDINATES (in um) THE SUB_IMAGE SAVED TO SHOW WHAT THE REGION LOOKS LIKE
;2421	1520
;2440	1520
;2459	1520
;2478	1520
; . . . . .
;SubPoints	    THESE ARE CO_ORDINATES (in um) OF THE REGION WHERE THE RAMAN / FLUORESCENCE measured
;0	0
;19	0
;38	0
;57	0
; ' ' ' ' '
;MotoEnc	    	    THESE ARE MOTOR ENCODER VALUES (in counts ?) OF THE REGION (?)
;438351	169289
;440327	169289
;442303	169289
;444279	169289
;446256	169289
;
;MODIFICATION HISTORY:
; (19-Jul-17 aph) first version
;-
; *******************************************************************************

function READ_DUV_POSITIONS, file=file, NOF=nof
 @duv_com
 on_error,2

title = 'Select name of DUV stack positions file'
fltr='*.txt'
s = 0
if keyword_set(nof) then fltr='*
; -------- check if fname exists; if not ask to find it
IF NOT keyword_set(file) then begin
	file = pickfile2(/READ, title=title, FILTER=fltr, /LPATH, DEFPATH=defpath)
	if file EQ '' THEN RETURN, s  ; bail-out if no filename
ENDIF
check = findfile(file)
if strlen(check(0)) EQ 0 then begin
	print, 'problems finding ', file
	file = pickfile2(/READ, title=title, FILTER=fltr, /LPATH, DEFPATH=defpath)
endif
check = findfile(file)
if strlen(check(0)) EQ 0 then begin
	axis_log, 'READ_DUV_POSITIONS: cannot find ' + file
	return, s
endif
fname=file
; ---- print header (laser info)
  axis_log, 'Laser parameters'
  tmp = text_read(fname)
  axis_log, string(tmp)
  laser_log_1 = tmp(0)
  laser_log_2 = tmp(1)
  laser_log_3 = tmp(2)
  laser_log_4 = tmp(3)


; ------ get co-ordinates of sub_image (labelled POINTS) ------------------------
  pos = text_read(fname, block=1)

; ----- convert parameters
; NB  X varies fastest, Y increments slowest
;  so seek first repeat of x to determine x,y,
; RESULTS stored in DUV.COM as duv_img_nx, duv_img_ny,duv_img_x,duv_img_y
  i=1
  test = pos(0,0)
  while pos(0,i) NE test do begin
;    print, pos(0,*)
  	i = i + 1
  endwhile
  nx = i
  ny = fix(n_elements(pos(1,*))/nx)
;  print, '# of x-points = ', nx
;  print, '# of y-points = ', ny
  duv_img_x=fltarr(nx)
  duv_img_y=fltarr(ny)
  duv_img_x(*)=pos(0,0:nx-1)
  for i=0,ny-1 do begin
	duv_img_y(i)= pos(1,i*(nx+1))
  endfor
;  print, 'X-values', duv_img_x
;  print, 'Y-values', duv_img_y
  duv_img_nx = nx  &  duv_img_ny = ny
; print, 'Dimensions of sub-image ', duv_img_nx, ' * ', duv_img_ny

; ------------ REGION READ-IN -------------------
; ------ get positions of user selected SUB POINTS within the sub-image  - the stack (x,y) axes
  pos = text_read(fname, block=2)

; ----- convert parameters
; NB  X varies fastest, Y increments slowest
;  so seek first repeat of x to determine x,y, array
  i=1
  test = pos(0,0)
  while pos(0,i) NE test do begin
;    print, pos(0,*)
  	i = i + 1
  endwhile
  nx = i
  ny = fix(n_elements(pos(1,*))/nx)
;  print, '# of x-points = ', nx
;  print, '# of y-points = ', ny
  duv_x=fltarr(nx)
  duv_y=fltarr(ny)
  duv_x(*)=pos(0,0:nx-1)
  for i=0,ny-1 do begin
	duv_y(i)= pos(1,i*(nx+1))
  endfor
;  print, 'X-values', duv_x
;  print, 'Y-values', duv_y
  duv_nx = nx  &  duv_ny = ny

  return, 1

  end


; ---------- original text read1 block text file read
; reads the bth block (starting from zero) in the file and returns as a matrix
;
;  if keyword_set(bi) eq 0 then a=chkfileblocks(fname) else a=bi
;  if keyword_set(b) ne 0 then begin
;    start=a(b,2)
;    nrow=a(b,0)
;    ncol=a(b,1)
;  endif else begin
;    start=a(0,2)
;    nrow=a(0,0)
;    ncol=a(0,1)
;  endelse
;  openr, unit, fname, /get_lun
;  if start gt 0 then begin
;    s=strarr(start)
;    readf,unit,s
; endif
;  ans=fltarr(ncol, nrow)
;  readf,unit,ans
;  as=size(ans)
;  if (as(0) eq 1) and (as(1) eq 1) then ans=ans(0,0) ;single #,convert to scalar
;  if (as(0) eq 2) and (as(1) eq 1) then begin
;    ;single column, convert to 1-dim array
;    b=fltarr(as(2))
;    for i=0,as(2)-1 do b(i)=ans(0,i)
;    ans=b
;  endif
;  close, unit
;  free_lun,unit
;  return,ans
;end

