; Copyright (c) 1998-2005 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		PLOTSECT
;
;LAST CHANGED: ----------------------------------- 	14-Jun-05
;
; PURPOSE:
;	This procedure plots multisection (non-monotonic) spectra data using splot
;
; CATEGORY:
;	Spectral display - captive to aXis2000
;
; CALLING SEQUENCE:
;	PlotSect, BufNum, AUTOSCALE=auto, SPLIT=split, _extra=e
;
; INPUTS:
; 	BufNum	Number (0-9) of aXis data buffer to display.
;
; KEYWORDS:
;	AUTOSCALE	- if 1, then display with min/max range
;	SPLIT		- if 1, separate the parts of the non-montonic data
;                 into individual, +ve X-going sections in separate buffers
;				- if 0, overlpot in one display buffer, but increment colors
;			       to indicate non-monotonic
;
; OUTPUTS: none
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;
; MODIFICATION HISTORY:
; (20-dec-97 aph) generates multisection MainImg plot using SPLOT
; (19-jun-98 aph) isolated from AXIS_C
; (26-sep-99 aph) updated with thumbnails separated
; (15-aug-04 aph) fix problem with processing files which do not have an Xlabel
;                 standard header
; (14-Jun-05 aph) add correction (kluge!) to allow oplot with 1 point sections
;-

PRO PlotSect, BufNum, AUTOSCALE=auto, SPLIT=split, _extra=e
@axis_com		; include COMMON from file
on_error, 2

CurBufTmp=BufNum
HANDLE_VALUE, Data(CurBufTmp),tmp	;Retrieve data from handle

; parse the file for non-monotonic; plot overlapping multi-section if needed
delta = tmp.x(1:*) - tmp.x(0:*)
sect = where(delta LT 0,n_plot)
if (n_plot GT 0) then BEGIN             ; check if truly multi-section
; print, '#plots, boundaries', n_plot+1, sect
; set x,y limits - autoscale if first plot without zoom
    xrang = !x.range  & yrang = !y.range
	if keyword_set(auto) then begin
    	xrang = [min(tmp.x), max(tmp.x)]
    	yrang = [min(tmp.d), max(tmp.d)]
    endif
; plot first section
	sect_thick = 2
	xax=tmp.x(0:sect(0))
	dat = tmp.d(0:sect(0))
	datn = tmp.dn(0:sect(0))
	test = tag_names(tmp)
	n_test = n_tags(tmp)
	Xlabel = 'Energy (eV)'
	for i = 0, n_test-1 do begin
		if strpos(test(i),'XL') GE 0 then Xlabel = tmp.xl
	endfor

; kluge to allow splot / oplot o display single points
	if n_elements(xax) EQ 1 then begin
;		axis_log, 'PlotSect: extending single point non-monotonic region'
		delta = 0.001*xax(0)
		del_dat = 0.001*dat(0)
		del_datn = 0.001*datn(0)
		xax = [xax, xax(0)+delta]
		dat = [dat, dat(0)+del_dat]
		datn = [datn, datn(0)+del_datn]
	endif

	s={t:tmp.t, d:dat, x:xax, xl:Xlabel,dn:datn, dl:tmp.dl}
	splot,s, color=bcolor(CurBufTmp), charsize=0.7, thick = sect_thick, $
	    xstyle=1, ystyle=1, xrange = xrang, yrange = yrang
  if keyword_set(auto) then begin
	Xrng = !x.crange           ; establish range values from first plot ( for overplot)
	Yrng = !y.crange
	Xscl(CurBufTmp,*) = !X.S
   	Yscl(CurBufTmp,*) = !Y.S
   	setGraf, 'MainImg',/SET
  endif
  if keyword_set(split) then begin
    HANDLE_VALUE, Data(CurBufTmp), s, /SET	;Store sections in sequential buffers
    Label(CurBufTmp) = 'sect. 1 ' + tmp.dl
    pBufs(0) = CurBufTmp
    Xscl(CurBufTmp,*) = !X.S
   	Yscl(CurBufTmp,*) = !Y.S
   	Zscl(CurBufTmp,*) = !Z.S
    bt=intarr(n_plot+1)
    bt(0) = CurBufTmp
  endif
  text = 'Non-monotonic - '+strcompress(string(fix(n_plot+1)))+' regions'
  axis_log, text
  colnum = CurBufTmp
	for i = 0, n_plot-2 do begin
	    xax=tmp.x(sect(i)+1:sect(i+1))
	    dat = tmp.d(sect(i)+1:sect(i+1))
	    datn = tmp.dn(sect(i)+1:sect(i+1))
		s={t:tmp.t, d:dat, x:xax, xl:XLabel, dn:datn, dl:tmp.dl}
		colnum = colnum + 1
		if colnum GT 9 then colnum = 0
		oplot, xax, dat, color = bcolor(colnum), thick = sect_thick
		if keyword_set(split) then begin
			s.dl = 'sect. '+ strcompress(string(i+2))+ ' ' + tmp.dl
    		HANDLE_VALUE, Data(colnum), s, /SET	;Store sections in sequential buffers
    		Label(colnum) = s.dl
    		Xscl(colnum,*) = !X.S
   			Yscl(colnum,*) = !Y.S
   			Zscl(colnum,*) = !Z.S
    		bt(i+1) = colnum
  		endif
	endfor
	xax=tmp.x(sect(n_plot-1)+1:*)
	dat = tmp.d(sect(n_plot-1)+1:*)
	datn = tmp.dn(sect(n_plot-1)+1:*)
; kluge to allow splot / oplot o display single points
	if n_elements(xax) EQ 1 then begin
;		axis_log, 'PlotSect: extending single point non-monotonic region'
		delta = (tmp.x(sect(n_plot-1)+1) - tmp.x(sect(n_plot-1)))/10
		del_dat = (tmp.d(sect(n_plot-1)+1) - tmp.d(sect(n_plot-1)))/10
		del_datn =(tmp.dn(sect(n_plot-1)+1) - tmp.dn(sect(n_plot-1)))/10
		xax = [xax, xax(0)+delta]
		dat = [dat, dat(0)+del_dat]
		datn = [datn, datn(0)+del_datn]
	endif
	s ={t:tmp.t, d:dat, x:xax, xl:XLabel, dn:datn, dl:tmp.dl}
	colnum = colnum + 1
	if colnum GT 9 then colnum = 0
	oplot, xax, dat, color = bcolor(colnum), thick = sect_thick
	if keyword_set(split) then begin
		s.dl = 'sect. '+ strcompress(string(n_plot+1))+ ' ' + tmp.dl
   		HANDLE_VALUE, Data(colnum), s, /SET	;Store sections in sequential buffers
   		Label(colnum) = s.dl
   		Xscl(colnum,*) = !X.S
   		Yscl(colnum,*) = !Y.S
   		Zscl(colnum,*) = !Z.S
   		bt(n_plot) = colnum
  		Blbl_upd & thumbplot,colnum    ; update thumbnails
   		pNum = n_plot+1
; plot the whole sequence from multiple buffers
		pBufs=intarr(pNum)
		FOR i = 0, pNum-1 DO BEGIN
			HANDLE_VALUE, Data(bt(i)), tmpo
			HANDLE_VALUE, pData(i), tmpo, /set      ;and update temporary storage
			pBufs(i) = bt(i)
   	   ENDFOR
    	text = 'No rescale multiplot: '        ; message to user
		FOR i = 0, pNum - 1 DO BEGIN
		text = text + ' ' + STRCOMPRESS(string(pBufs(i)))
	  ENDFOR
	 axis_log, text
	 Refresh
	endif
ENDIF ELSE axis_log, 'PlotSect: not multi-section data'
return
END