; Copyright (c) 1998-2024 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_NANOMAX_PTYCHO_SINGLE_STACK.PRO
;;
;LAST CHANGED: ----------------------------------- 	29 Mar 2024 (aph)

; PURPOSE:
;	This function reads a set of images using the read_nanomax_image routine to read NanoMAX h5 files
;
; CATEGORY:
;	    Runs stand alone or inside aXis2000.
;
; CALLING SEQUENCE:
;	RESULT = READ_NANOMAX_PTYCHO_STACK(path=path, verbose=verbose, group = group)
;
; INPUTS: none
;
; KEYWORDS:
;	PARTH
;	VERBOSE
;	GROUP
;
; OUTPUTS:  stack as (*.dat, *.ncb)
;
;
; COMMON BLOCKS:
; AXIS_COM    standard set of common blocks
; stack_process_com
; COMMON volume_data, image_stack
; bsif_com


; MODIFICATION HISTORY:
; (26-Mar-24 aph) first version
; (29-Mar-24 aph) evolve from sets of files o single h5 file with stxm, ptycho-amp and ptycho-pha

;-

FUNCTION  read_nanomax_ptycho_single_stack, path=path, silent=silent, verbose=verbose, group = group
@axis_com
@stack_process_com
COMMON volume_data, image_stack
@bsif_com

on_error,2

null = 0


; ------------ select NanoMAX  *.h5 file(s)

file = dialog_pickfile(title='Select NanoMAX stack  *.h5 files',filter=['*.h5'], path=DefPath, get_path=folder_path)

DefPath=folder_path
cd, DefPath
print, "new default path is ", DefPath

a = h5_parse(file, /read_data)

; ---------- get photon energies
energy = a.info.energies._data

; print, 'energies ', energy

n_e = n_elements(energy)

; ---------- get sequential scan numbers
scan_num= a.info.scans._data
;print, scan_num


; ---- Ptycho reconstructions (NB ion chamber was incorporated when prepare sets of DI's for eeconstruction

pixel_sizes =  a.maps.ptycho_opticaldensity._psizes._data/1e8
; pixel size at each energy, All images should be resampled using the pixel size at the highest photon energy

pix_size = pixel_sizes(n_elements(pixel_sizes)-1)
pix_size = pix_size*1e-9

; ----- reading ptcho OD stack
image_stack =  a.maps.ptycho_opticaldensity.data._data

print, 'Interpolating all images to pix_size

help, image_stack

; get sizes to make x,y axes
tmp = size(image_stack)
nx = tmp(1)
ny = tmp(2)

; ------- generate x, y axes  &  store in bsif_com parameters
ev = energy

pix_size_array = a.maps.raw_stxm._psizes._data
pix_size=mean(pix_size_array*1e9)
x = findgen(nx)
x = pix_size*x
y = findgen(ny)
y = pix_size*y

x_start = x(0)
x_stop  = x(n_elements(x)-1)
x_step = (x_stop - x_start) /(nx-1)

y_start = y(0)
y_stop  = y(n_elements(y)-1)
y_step = (y_stop - y_start) /(ny-1)

n_rows = nx
n_cols = ny


print, ' nx   ny   ', nx, ny

; generate tag lines for each image of the stack
filename_ev_msec_list = strarr(n_e)

; ---------- loop through to build tag line of stack and flip thhe images (Y ==> -Y)
for i = 0, n_e-1 do begin
	d = image_stack(*,*,i)
	d = rotate(d, 7)
	image_stack(*,*,i) = exp(d)
	filename_ev_msec_list(i) = 'scan ' + strtrim(string(scan_num(i)),2) + '  ' + string(energy(i))
	axis_log, filename_ev_msec_list(i)
endfor

if n_elements(stack_file) LE 0 then stack_file = ' '
stack_file = get_text(prompt = 'name of stack ', val = stack_file, group = group)

stack_file = DefPath + stack_file
stack_wb, stack_file  ;, /real

return, stack_file

end




; ---------- loop through to get all images of the stack
;for i = 0, n_e-1 do begin
;	tmp = read_nanomax_ptycho(file=filelist(i), /silent)
;	image_stack(*,*,i) = exp(tmp.d)								; exponentiate since Maik Khn logged prior to normaization to alda2.1 ion chamber signal
;	t = ax_name(filelist(i))
;	ev(i) = tmp.e
;	filename_ev_msec_list(i) = t(1) + '  '+ string(ev(i))
;	axis_log, filename_ev_msec_list(i)
;endfor
;
;if n_elements(stack_file) LT 0 then stack_file = ' '
;stack_file = get_text(prompt = 'name of stack ', val = stack_file)
;stack_file = DefPath + stack_file
;stack_wb, stack_file, /real
