; Copyright (c) 1998-2016 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	AXIS_READ_IMAGE
;
;LAST CHANGED: ----------------------------------- 14-Feb-16
;
;  This function reads image files from formats defined
; by the filename extension into the standard AXIS 2d data structure.
; If a file name is not supplied or if the file cannot be found
; the user is asked to identify the file of interest.
; The extension of the file is set as the default extension
; for future image reading.
;
; CATEGORY:
;	AXIS: utility
;
;CALLED FROM AXIS:
;	used in some procedures (SVD ..)
;
; CALLING SEQUENCE:
;	Result = AXIS_READ_IMAGE( [file=file, /axis] )
;
; INPUTS: none
;
; KEYWORD PARAMETERS:
;	FILE:	name of file to be read (including path)
;	AXIS:	flag to indentify if called from AXIS
;
; OUTPUTS:
;  This function returns a 2d AXIS data structure
;   s = {t:'2d', x:x, y:y, d:d, xl:x_label, yl: y_label, dl: data_label}
;
; COMMON BLOCKS:
;   @AXIS_COM - set of common blocks for AXIS
;   @BSIF_COM - common block for netCDF
;
; PROCEDURE:
; 	This function is a front end for a number of routines used to read in
;   the various data formats that AXIS supports.
;	FORMAT         PROCEDURE	EXTENSION
; ---------------------------------------
;	AXIS			AXB_LOAD	*.axb
;	ALS-STXM		READ_ALS	*.im%
;	NSLS-old		READ_BNL	*.nc
;	NSLS-cryo		READ_CRYO	*.sxm
;	ALS-PEEM		READ_PEEM	*.tif
;
; Although not implemented (as of 3-Jan-00) the idea is
; to develop auto file identification and read-in capability.
;
; EXAMPLE:
; 	a = axis_read_image(file='f:\stxm\9909\914\90914100.im1')
;
; MODIFICATION HISTORY:
; (17-Oct-99 aph) first version
; (03-jan-00 aph) AXIS standard documentation
; (14-Feb-15 aph) add FILTER keyword
;-

Function AXIS_READ_IMAGE, file = file, axis = axis, filter = filter
@axis_com
on_error,2

; determine if AXIS is running (therefore may have called axis_read_image )
; either when AXIS keyword is supplied or if any widget active
if  keyword_set(axis) then axis_on = 1 else axis_on = widget_info(/active)

if keyword_set(file) then begin
; ---------- confirm file exists and file extension known
	check = findfile(file)
	if strlen(check(0)) EQ 0 then begin
		data_path = get_path(file)
		if data_path EQ '' then goto, ask_user
		t = ax_name(file)
		file = data_path + t(1) + t(2)
	endif
endif else begin
; ---------- user dialog to request image file --------
	ask_user:
	if axis_on then begin
		file = dialog_pickfile(path = DefPath, $
	    	 filter = last_image_ext, Get_path = DefPath, group = axis_ID)
	endif else begin
		file = dialog_pickfile(path = DefPath, $
	    	 filter = last_image_ext, Get_path = DefPath)
	endelse
	if strlen(file(0)) EQ 0 then return,0
endelse

; --------- read in file using the extension as a keyword
t = ax_name(file)
image_type = '*.' + strcompress(t(2))
if strmid(image_type,2,2) EQ 'im' then image_type = '*.im'
;print, 'Reading in image of type: ', image_type
CASE image_type OF
	'*.axb' : begin
		s = axb_load(file=file)
	end
	'*.im' : begin
		s = read_als(file)
	end
	'*.nc' : begin
		s = read_bnl(file)
	end
	'*.sxm' : begin
		s = read_cryo(file=file)
	end

    else : begin
    	text = ' Unknown image type ' + file
    	print, text
    	widget_control, Bad_ID=Bad_ID, Uprompt, SET_VALUE=text
    	return, 0
    endelse
ENDCASE

; ---------- set last_image_extension and return result
last_image_ext = '*.' + t(2)
;print, 'Setting last_image_ext to ', last_image_ext
return, s

end
