; Copyright (c) 1998-2010 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		PICKFILE2
;
;LAST CHANGED: ----------------------------------- 	25-Dec-2024 (from 22 Jul 2010)
;
; PURPOSE:
;	This function is a dialog which allows user to select a filename.
;   It is a wrapper over the old IDL PICKFILE dialog
;    with last path, last file options and forcing extension
;
; CATEGORY:
;	utility. stand alone
;
; CALLING SEQUENCE:
;	Result = PICKFILE2 (LPATH=lp, TITLE=TITLE, $
;                       WRITE=write, LFILE=lf, INFO=info, _extra=e)
;
; INPUTS: none
;
; KEYWORDS:
;	FILTER - extension (*.hdr)
;	INFO - if set, print tracking info on this routine
;	LPATH - last path used
;	LFILE - last file used
;	TITLE - user supplied title for dialog
;	WRITE - if set, when selecting a filename, checks & warns for existence
;
; OUTPUTS: none
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;	common lastpath, lpath, lfile				SHOULD NOT BE NEDED - these are in axis_com
;
; MODIFICATION HISTORY:
; (06-jul-98 aph) force DefPath to be LastPath from exit
; (04-jan-00 aph) add WRITE, AXIS keywords and build a wrap that warns user of overwriting
; (25-apr-00 aph) attempt to get lpath working properly; allow use outside of axis
; (19-apr-01 aph) add TITLE
; (14-may-02 aph) AXIS standard header added; force filter extension
; (22-nov-08 aph) use axis_log for warnings
; (20-Apr-10 aph) remove 'forcing extension' warning
; (22-Jul-10 aph) accomodate calls where defpath is not defined
; (25-Dec-24 aph) force DefPath to be set to the path to the file name; make sure lpath and lfile are set (in axis_com.pro)
;-

function PICKFILE2, LPATH=lp, filter=filter, TITLE=TITLE, WRITE=write, $
         LFILE=lf, INFO=info, group = group, _extra=e
@axis_com
; common lastpath, lpath     ;, lfile  this is already in aXis_com  (comment out 25-Dec-2024)
on_error,2

;   ----------- establish sensible starting parameters
if n_elements(lpath) eq 0 then lpath=defpath
lp = ''
lf = ''
if strlen(DefPath) gt 0 then begin
	if n_elements(lpath) eq 0 then lpath=defpath
	if n_elements(gpath) eq 0 then gpath=defpath
endif

; --------- troubleshoot changes to lpath & defpath  (25-Dec-2024)
; after definitions
if keyword_set(info) then begin
	print, 'after definitions    Defpath= ', Defpath
	print, ' lpath= ', lpath
	print, ' lfile= ', lfile
	print, ' gpath= ', gpath
	print, ' '
endif

pick_again:		; tag used to re-select file if the file already exists (in case of writing)

file=PICKFILE( PATH=lpath, GET_PATH=gpath,$  ; FILE=lfile, $
               filter=filter, title=title, /noconfirm, _extra=e)
if strlen(file) EQ 0 then return, file

; ------ force extension same as filter
if keyword_set(filter) AND keyword_set(write) then begin
	t = ax_name(file)
	t2 = ax_name(filter)
;	print, ' forcing exension to ', '.' + t2(2)
	file = t(0) + t(1) + '.' + t2(2)
endif

; after reading filename using pickfile (obsolete IDL routine)
if keyword_set(info) then begin
	print, 'after selecting file     Defpath= ', Defpath
	print, ' lpath= ', lpath
	print, ' lfile= ', lfile
	print, ' gpath= ', gpath
	print, ' '
endif

; ------ check if this file already exists [ ONLY if writing - i.e. /write keyword)
if n_elements(overwrite_all) EQ 0 then overwrite_all = 0
if overwrite_all EQ 1 then goto, continue	; do not check if switched for all files
if keyword_set(write) then begin
	test = findfile(file)
	if test(0) EQ file then begin
		t=ax_name(file)
		fileshort = t(1) + '.' + t(2)
		axis_log, ' WARNING: ' + fileshort + ' already exists. Overwrite ?'
		if widget_info(/active) EQ 1 then begin  ; assumes if a widget is used, it is AXIS
			dec = file_overwrite(val=1, group = axis_ID)
		endif else  dec = file_overwrite( val=1)
		if dec EQ 0 then goto, pick_again
		if dec EQ 2 then overwrite_all = 1
	endif
endif
continue:

; ------------------------ set Defpath, lpath, lfile to values from the file selected (or written)
if file NE '' then begin
	defpath = gpath
	lpath = gpath
	lfile = file
endif
if keyword_set(info) then begin
	print, 'exiting pickfile2()     Defpath= ', Defpath
	print, ' lpath= ', lpath
	print, ' lfile= ', lfile
	print, ' gpath= ', gpath
	print, ' '
endif
return, file
end


