; Copyright (c) 1998-2009 A.P. Hitchcock  All rights reserved
;+
;NAME:
; AX_PEEM_READ
;
;LAST CHANGED: ----------------------------------- 30-Jul-09 (aph)
;
; PURPOSE:
;	This file contains the codes for a WIDGET used to define parameters
; and read in one or a sequence of PEEM data files from various PEEMs
; 12-bit or 16-bit ALS PEEM2 format, ALS PEEM3 format (ALS PEEM3 read with separate widget)
; and 16-bit SPHINX (SRC, Labview generated) files are read.
;
; CATEGORY:
;	Image input (stand alone, axis2000)
;
; CALLING SEQUENCE:
; for procedures:
;	Result = AX_READ_PEEM()
;
; INPUTS: none - widget defines inputs for RD_PEEM
;
; KEYWORDS
; SPHINX - adapt to Sphinx format data (16-bit tif; dat multicolumn energy)
; AXIS - called from axis
; MULTI - multiple files (image sequence)
; COL12 - 12-bit data (old ALS camera)
; PEEM3 - data format for ALS PEEM3
; GROUP_LEADER - group leader variable
;  _EXTRA - extra parameters
;
; CONTENTS
;	BUT_Cancel		cancel job; kill widget
;	BUT_Read_data	execute job (parameters from AX_PEEM_COM set in widget)
;	BUT_no_correct	set no_correction
;	BUT_subtract_dark set subtract_dark
;	BUT_gain  		set gain_correction
;	BUT_dark_gain	set dark & gain correction
;	BUT_median		switch for mediam smooth
;	BUT_bits12		switch for 12-bit / 16-bit
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;   stack_process_com     stack_process common
;	BSIF_COM	common for netCDF format data
; 	AX_PEEM_COM	default path and filename information specific to PEEM
;	-------- COMMON PEEM -------------------
;	axis_on			if set, axis2000 running
;	dark_path		dark file path
;	gain_path		gain file dath
;	peem_aoi		Area of Interest file
;	peem_aoi_dark	if set, apply AOI to dark file
;	peem_aoi_gain	if set, apply AOI to gain file
;	peem_bin		binning (nXn) reduced to 1 pixel (default = 1)
;	peem_bits12		if set, all data files are 12-bit (ALS PEEM-2 camera default)
;	peem_col12		gives number of columns to drop for 12-bit readin (to avoid header noise)
;	peem_dark		dark data file
;   peem_dark_avg	if set, average multiple dark files
;   peem_dark_last	number of final dark of a sum of dark files
;	peem_dwell		exposure time (sec)
;   peem_E			energy (eV)
;	peem_first		first/single data file
;	peem_gain		gain data file
;	peem_ID			Widget ID
;	peem_last		last data file
;	peem_median		if set, 3-point median smooth
;	peem_multi		if set, then multiple file conversion to net_CDF
;   peem_ncb        if set, write directly to ncb file (no *.nc)
;	peem_process	string defining action
;				('no correct', dark correct', 'gain correct', dark and gain correct')
;	peem_region		if set, select a subregion, in real space parameters
;	peem_scale		size of pixel (um)
;	peem_sphinx     flag that data format is sphinx
;   peem_3			flag that data format is ALS peem3 (jan07)
;	peem_struct		flag indicating success (1) or failure (0) of conversion
;	region 			4-vector defining xmin, xmax, ymin, ymax of region
;	source_path		source path
;
; SIDE EFFECTS:
;	A widget is created/destroyed. If run standalone, result is plotted.
;
; MODIFICATION HISTORY:
; (05-jun-01 aph) first version; AXIS standard header added
; (31-jun-01 aph) enabled, non-functional
; (12-aug-01 aph) got working in Aug 1-12 period
; (08-may-01 aph) adapt to convert Elmitec PEEM (sphinx); through *.nc and directly to NCB
; (16-jul-02 aph) adapt to 8-bit tif file name convention (Sphinx)
; (05-oct-02 aph) auto-sort energies
; (02-nov-02 aph) increase file numbering scheme for sphinx data to full 5-digit format
; (17-feb-03 aph) fixed problem of extension in filename parsing
; (18-mar-03 aph) correction for file numbering error introcuded by ALS change into sphinx !
; (02-may-04 aph) change header
; (20-nov-04 aph) fix problem of not working in VM mode
; (13-aug-05 aph) fix not working in IDL6.1 (SGU bug report Aug-05: PEEM_ID not valid for tmp files)
; (22-feb-06 aph) fix not working - problem was that single images are not sent back to aXis2000 !!
; (20-jan-07 aph) adapt for ALS PEEM3 images (but eventually used a separate widget based on Scholl routines)
; (12-Sep-07 aph) clean up; adapt for use of ax_cimage to remove extrema to remove columns (NB NOT changed for stacks)
; (30-jul-09 aph) changed analcom to stack_process_com (update to stack_process)
;                 added on_error,2 to all routines
;-
; ###################################################

pro BUT_Cancel, Event
@ax_peem_com
on_error, 2

	peem_struct = 0
   	 WIDGET_CONTROL, Event.Top, /DESTROY
end

; ###################################################
; --------- AX_READ_PEEM:  main read in routine -----
; ###################################################
pro BUT_Read_data, Event, _extra=e
@ax_peem_com
@axis_com
@stack_process_com
@bsif_com
COMMON volume_data, image_stack
on_error,2

; -------- get current value of relevant parameters
id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_first_data_file')
WIDGET_CONTROL, id, Get_Value = test
PEEM_first = string(test(0))
if strlen(PEEM_first) EQ 0 then return
t = ax_name(PEEM_first) & source_path = t(0)

; get energy, scale and bin parameters
id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_energy')
WIDGET_CONTROL, id, Get_Value = test
PEEM_E = float(test(0))
id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_scale')
WIDGET_CONTROL, id, Get_Value = test
PEEM_scale = 0.001*float(test(0))
id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_bin')
WIDGET_CONTROL, id, Get_Value = test
PEEM_bin = fix(test(0))
id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_dwell')
WIDGET_CONTROL, id, Get_Value = test
PEEM_dwell = fix(test(0))

; --- get region
if peem_region EQ 1 then begin
	region = fltarr(4)
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_xmin')
	WIDGET_CONTROL, id, Get_Value = test
	region(0) = float(test(0))
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_xmax')
	WIDGET_CONTROL, id, Get_Value = test
	region(1) = float(test(0))
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_ymin')
	WIDGET_CONTROL, id, Get_Value = test
	region(2) = float(test(0))
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_ymax')
	WIDGET_CONTROL, id, Get_Value = test
	region(3) = float(test(0))
endif else region = 0

; ------ get Area of Interest if needed
id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_AOI_file')
WIDGET_CONTROL, id, Get_Value = test
PEEM_aoi = test(0)
if PEEM_AOI_dark EQ 1 OR PEEM_AOI_gain EQ 1 then begin
	if strlen(PEEM_aoi) NE 0 then begin
		if PEEM_AOI_dark EQ 1 then print, 'Applying AOI file to dark file'
		if PEEM_AOI_gain EQ 1 then print, 'Applying AOI file to gain file'
		openr, lun, PEEM_aoi, /get_lun
		txt = '' &	aoi_box = intarr(4)
		readf, lun, txt
		close, lun & free_lun, lun
;		print, 'input to Area of interest = ', txt   ; xmin, ymin, xmax, ymax
		start = strpos(txt,'Box1')
		txt = strmid (txt, start+5)
		reads, txt, aoi_box
		t = aoi_box(2) & aoi_box(2) = aoi_box(1) & aoi_box(1) = t
		aoi_width = aoi_box(1) - aoi_box(0) + 1
		aoi_height = aoi_box(3) - aoi_box(2) + 1
		print, string(format = $
		 '("Area of interest: X0=", I4," Y0=",I4,".   width=", I4," height=",I4)', $
		  aoi_box(0), aoi_box(2), aoi_width, aoi_height)
	endif else return  ; dump back to Widget to get name of AOI
endif else aoi_box = 0

; ------ define col12 for 12-bit readin
col12 = peem_col12
; print, 'ax_peem_read: col12 = ', col12


; --------- get dark signal if needed
if peem_process EQ 'subtract dark' OR peem_process EQ 'dark and gain correct' then begin
; --- get first dark data
; - get dark average if needed
	if peem_dark_avg EQ 1 then begin
		id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_last_dark')
		WIDGET_CONTROL, id, Get_Value = test
		PEEM_dark_last = fix(test(0))
		dkpos = strpos(PEEM_dark,'dk')
		darkbase = strmid(PEEM_dark,0,dkpos+2)
		first_dark = fix(strmid(PEEM_dark, dkpos+2, strpos(PEEM_dark,'.')-2-dkpos))
		n_dark = PEEM_dark_last - first_dark + 1
		if n_dark LT 1 then return
		s = 0
		FOR i_dark = 0, n_dark-1 DO BEGIN
			next = darkbase + strtrim(string(first_dark+i_dark),2)+'.tif'
			t = Rd_PEEM(next, bits12= PEEM_bits12, group = PEEM_ID, axis = axis_on, $
			aoi_box = aoi_box, aoi_data=peem_aoi_dark, col12=peem_col12, $
			energy = peem_E, scale = peem_scale)
			s = s + t.d
		ENDFOR
		dark = s / n_dark
		print, 'average of one and ', n_dark,' dark files', mean(t.d), mean(dark)
	endif else begin
		id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_dark_file')
		WIDGET_CONTROL, id, Get_Value = test
		PEEM_dark = string(test(0))
		if strlen(PEEM_dark) EQ 0 then return
		tmp = Rd_PEEM(PEEM_dark, bits12= PEEM_bits12, group = PEEM_ID, axis = axis_on, $
				aoi_box = aoi_box, aoi_data=peem_aoi_dark, col12=peem_col12, $
				energy = peem_E, scale = peem_scale)
		if n_tags(tmp) EQ 0 then return				; return to widget if do not get dark
		n_dark = 1
		dark = tmp.d
	endelse
endif

; --------- get gain signal if needed
if peem_process EQ 'gain correct' OR peem_process EQ 'dark and gain correct' then begin
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_gain_file')
	WIDGET_CONTROL, id, Get_Value = test
	PEEM_gain = string(test(0))
	gain = Rd_PEEM(PEEM_gain, bits12= PEEM_bits12, axis = axis_on, group = PEEM_ID, $
			energy = peem_E, region = region, bin = PEEM_bin, $
			aoi_box = aoi_box, aoi_data=peem_aoi_gain,  col12=peem_col12, $
			dwell = PEEM_dwell, scale = peem_scale, _extra=e)
endif

; ---- set up the loop for either single file read in (to AXIS format) or conversion to netCDF via AXIS format
if peem_multi EQ 1 then begin
; ---- set up the loop for multiple files  - named as at ALS-PEEM !! -----
	if strlen(PEEM_last) EQ 0 then return   ; force user to supply last file name
	file1 = PEEM_first
	file2 = PEEM_last
	if peem_sphinx EQ 1 then begin		; sphinx - assume last 5 before dot is number
; ----- (02-nov-02(aph) - often get more than 1000 files/day or sample - increase to full 5-digit format)
		fileroot=strmid(file1, 0, strpos(file1,'.')-5)
; -------- remove .8bit. if needed
		check = ax_name(file1)
		ext = check(2)
		t = ax_name(fileroot) & fileshort=t(1) & source_path = t(0)
		first = fix(strmid(file1, strpos(file1,'.')-5, 5))
		last  = fix(strmid(file2, strpos(file2,'.')-5, 5))
	endif else begin
		check = ax_name(file1) & ext = check(2)
		fileroot=strmid(file1, 0, strpos(file1,'#')+1)
		t = ax_name(fileroot) & fileshort=t(1) & source_path = t(0)
		first = fix(strmid(file1, strpos(file1,'#')+1, strpos(file1,'.')-strpos(file1,'#')-1))
		last  = fix(strmid(file2, strpos(file2,'#')+1, strpos(file2,'.')-strpos(file2,'#')-1))
	endelse
; determine number of digits used in file numbering scheme - kluge for ALS PEEM read for Cynthia !!
	if 	strpos(file1,'#') GT 0 then begin
		digits =  strpos(file1,'.') - strpos(file1,'#')-1
		base = fix(10.^digits)
	endif else base = 100000
;   	print, 'ax_peem_read: CHECK filename parsing: '
;   	print,  ' PATH: ', fileroot
;   	print,  ' SHORT NAME   ', fileshort
;   	print,  ' EXTENSION   ',  ext
;   	print,  ' first, last ', first, last
;   	print,  ' base for # ', base
	filename_list = strarr(last - first + 1)
	for i = first, last do begin
		num = strcompress(string(base + i))
;		print, ' constructing filename ', i, ' number ', num
		num = strmid(num,2,5)
		file=fileshort + num + '.' + ext                ; replace '.tif' to cover '.8bit.tif' Sphinx convention
		filename_list(i-first) = strcompress(file)
	endfor
	n_files = n_elements(filename_list)

	filebase= t(0) + 't_'

; get Energy values from spectrum file - switch from pem_load to spc_load depending on source
	WIDGET_CONTROL, Uprompt, Bad_ID = badID, SET_VALUE='Select Energy file'
	if keyword_set(peem_sphinx) then begin
		e_tmp = spc_load(filt='*.dat', xcol=1, ycol=2, title='Select stack energy file')
		t_dir = dialog_message(/question,'Scanned high-E to low-E ?')
		if t_dir EQ 'Yes' then E_vals = reverse(e_tmp.x) else $
			E_vals = e_tmp.x
	endif else begin
		e_tmp = pem_load()
		E_vals = e_tmp.x
	endelse
endif else begin

; ---- set up the loop if needed for one file -----
	t = ax_name(PEEM_first)
	filename_list = t(1) + '.' + t(2)
	E_vals = PEEM_E
	first = 1  & n_files = 1
endelse

; ------- LOOP PROCESSING
IF peem_sphinx EQ 1 then begin
	axis_log, 'converting SPHINX image or image sequence'
	PEEM_bits12 = 0
	bits12 = 0
	col12=0
ENDIF  ELSE  axis_log, 'converting ALS PEEEM2 image or image sequence'

; ---- set up to write to stack directly
If peem_ncb EQ 1 then begin
; ---- read & process first file to determine the array size
	datafile = source_path + filename_list(0)
	peem_E = E_vals(0)
	tmp = Rd_PEEM(datafile, sphinx = peem_sphinx, bits12= PEEM_bits12, $
	        axis = axis_on, group = PEEM_ID, $
			energy = peem_E, region = region, bin = PEEM_bin, $
			dwell = PEEM_dwell, scale = peem_scale, col12=peem_col12, _extra=e)
	if peem_bits12 and peem_col12 LT 5 then begin
		a = tmp.d  & col_drop = 5 - peem_col12
		t = size(a) &	b=a(col_drop:(t(1)-1),0:(t(2)-1)) & a = b
     	t = size(a)
     	x = tmp.x(col_drop:n_elements(tmp.x)-1)
     	tmp = {t:'2d', x:x, y:tmp.y, e:tmp.e, d:a, xl: tmp.xl, yl: tmp.yl, dl:tmp.dl}
	endif
; ----- apply binning if requested
	if peem_bin NE 1 then begin
		n_cols = n_elements(tmp.x) & n_rows = n_elements(tmp.y)
		bin = PEEM_bin
; (aph) - temporarily remove the check for 'loose' points to see if binning works on Elmitec (16-jul-02)
;		if (float(n_cols)/float(bin))-fix(n_cols/bin) GT 0 $
;         OR (float(n_rows)/float(bin))-fix(n_rows/bin) then begin
	        nct =fix(n_cols/bin)*bin  &  nrt = fix(n_rows/bin)*bin
	        xtra_c = n_cols - nct + 1
	        xtra_r = n_rows - nrt + 1
			x_stop = tmp.x(n_cols-1) & y_stop = tmp.y(n_rows-1)
	        x_stop = x_stop*float(nct)/float(n_cols)  ; correct axis length for truncation
	        y_stop = y_stop*float(nrt)/float(n_rows)
	        tmp2 = tmp.d                      ;force size to integer mutiple of bin
	        d = fltarr(nct,nrt)
			d=tmp2(0:n_cols-xtra_c,0:n_rows-xtra_r)
;      	endif
      	n_cols=fix(n_cols/bin)
      	n_rows=fix(n_rows/bin)
      	x = findgen(n_cols)*PEEM_scale*bin       ; get correct scale if bin
      	y = findgen(n_rows)*PEEM_scale*bin
       	tmp = {t:'2d', x: x, y:y, d:d, xl:tmp.xl, E:tmp.E, yl:tmp.yl, dl:''}
	endif

; -------------- define storage arrays for full loop
	npts = n_files
	nx = n_elements(tmp.x) & ny = n_elements(tmp.y)
	image_stack = fltarr(nx,ny,npts)
	filename_ev_msec_list = strarr(npts)
;	print, 'Dimension of volume data is ', nx, ny, npts
endif

; --------- MAIN LOOP to read and convert images -----------
FOR i_file = 0, n_files-1 DO BEGIN
	datafile = source_path + filename_list(i_file)
	peem_E = E_vals(i_file)
	print, 'Process ', filename_list(i_file), '  E(eV): ', peem_E,' with ', PEEM_process

	CASE strtrim(PEEM_process,2) of

	'no correct': BEGIN
		tmp = Rd_PEEM(datafile, sphinx = peem_sphinx, bits12= PEEM_bits12, $
		        axis = axis_on, group = PEEM_ID, $
				energy = peem_E, region = region, bin = PEEM_bin, $
				dwell = PEEM_dwell, scale = peem_scale, col12=peem_col12, _extra=e)
		tmp.yl = 'no correct ' + tmp.yl
	  END

	'subtract dark': BEGIN
		tmp = Rd_PEEM(datafile,  sphinx = peem_sphinx, bits12= PEEM_bits12, axis = axis_on, group = PEEM_ID, $
				energy = peem_E, region = region, bin = PEEM_bin,  col12=peem_col12, $
				dwell = PEEM_dwell, scale = peem_scale,CCD_bgnd = dark)
		if peem_dark_avg EQ 1 then $
		   tmp.yl = tmp.yl + '- <dark> - avg '+strtrim(string(n_dark),2) else $
		   tmp.yl = tmp.yl + '- dark'
	  END

	'gain correct': BEGIN
		tmp= Rd_PEEM(datafile,  sphinx = peem_sphinx, bits12= PEEM_bits12, axis = axis_on, group = PEEM_ID, $
				energy = peem_E, region = region, bin = PEEM_bin, $
				aoi_box = aoi_box, col12=peem_col12, $
				dwell = PEEM_dwell, scale = peem_scale, _extra=e, CCD_gain = gain.d)
		tmp.yl = tmp.yl + ' /gain'
	  END

	'dark and gain correct': BEGIN
		tmp = Rd_PEEM(datafile,  sphinx = peem_sphinx, bits12= PEEM_bits12, axis = axis_on, group = PEEM_ID, $
		        energy = peem_E, region = region, bin = PEEM_bin, $
		        aoi_box = aoi_box,  col12=peem_col12, $
				dwell = PEEM_dwell, scale = peem_scale,  _extra=e, CCD_bgnd = dark, CCD_gain = gain.d)
		if peem_dark_avg EQ 1 then $
			tmp.yl = '('+ tmp.yl + ' - <dark>)/gain' $
			else 	tmp.yl = '('+ tmp.yl + ' - dark)/gain'
	  END

	  ELSE: BEGIN
	  	print, 'un-matched case ', peem_process
	  	return
	  ENDELSE
	ENDCASE

; -----  if bits12 and col12 <5 then cut up to 5th column to remove header noise
	if peem_bits12 and peem_col12 LT 5 then begin
		a = tmp.d  & col_drop = 5 - peem_col12
		t = size(a) &	b=a(col_drop:(t(1)-1),0:(t(2)-1)) & a = b
     	t = size(a) & print,'AX_PEEM_READ: converted to ', t(1), ' x ', t(2)
     	x = tmp.x(col_drop:n_elements(tmp.x)-1)
     	tmp = {t:'2d', x:x, y:tmp.y, e:tmp.e, d:a, xl: tmp.xl, yl: tmp.yl, dl:tmp.dl}
	endif

; ----- apply binning if requested
	if peem_bin NE 1 then begin
		n_cols = n_elements(tmp.x) & n_rows = n_elements(tmp.y)
		bin = PEEM_bin
; (aph) - temporarily remove the check for 'loose' points to see if binning works on Elmitec (16-jul-02)
;		if (float(n_cols)/float(bin))-fix(n_cols/bin) GT 0 $
;         OR (float(n_rows)/float(bin))-fix(n_rows/bin) then begin
	        nct =fix(n_cols/bin)*bin  &  nrt = fix(n_rows/bin)*bin
	        xtra_c = n_cols - nct + 1
	        xtra_r = n_rows - nrt + 1
;		    print, 'truncate image to ', fix(nct), fix(nrt)
			x_stop = tmp.x(n_cols-1) & y_stop = tmp.y(n_rows-1)
	        x_stop = x_stop*float(nct)/float(n_cols)  ; correct axis length for truncation
	        y_stop = y_stop*float(nrt)/float(n_rows)
	        tmp2 = tmp.d                      ;force size to integer mutiple of bin
	        d = fltarr(nct,nrt)
			d=tmp2(0:n_cols-xtra_c,0:n_rows-xtra_r)
 ;     	endif
      	n_cols=fix(n_cols/bin)
      	n_rows=fix(n_rows/bin)
;      	print, 'bin image ', fix(bin), ' times,  to ',n_cols,'  x', n_rows
      	d=rebin(d,n_cols,n_rows)
      	x = findgen(n_cols)*PEEM_scale
      	y = findgen(n_rows)*PEEM_scale
      	dl = 'B'+string(format='(I1)',PEEM_bin)+ ' ' + tmp.dl
      	tmp = {t:'2d', x: x, y:y, d:d, xl:tmp.xl, E:tmp.E, yl:tmp.yl, dl:dl}
	endif

; ------ apply median smooth if requested
	if PEEM_median EQ 1 then begin
		tmp.d = median(tmp.d,3)
		tmp.dl = 'M3 ' + tmp.dl
	endif

; -------- convert to netCDF or store in image_stack for *.ncb
	IF PEEM_multi EQ 1 then begin
; ------ define the output file name  FORCE 3-character scheme with no blanks !!
		zero = '000'
		t_num = strtrim(strcompress(string(i_file+first)),2)
		strput, zero, t_num, 3-strlen(t_num)
	  	IF peem_ncb NE 1 then begin
			file = strlowcase(filebase + zero) + '.nc'
			if !VERSION.OS_FAMILY EQ 'unix' then sep = '/' else sep = '\'
			spos=rstrpos(file,sep)
			fileshort = strmid(file,spos+1,strlen(file))
			data_title = byte(fileshort)
			image_title = byte(fileshort)
			filename_list(i_file) = file		; save output name in this list
			file = source_path + file
;			help, tmp, /struct
			x_start = min(tmp.x)    &   x_step = tmp.x(1) - tmp.x(0)
			y_start = min(tmp.y)    &   y_step = tmp.y(1) - tmp.y(0)
			n_cols = n_elements(tmp.x)       &   n_rows = n_elements(tmp.y)
			x_stop = x_start + x_step*n_cols
			y_stop = y_start + y_step*n_rows
			sd.clock_frequency = 1.
			sd.dwell_time = PEEM_dwell
			if (PEEM_E NE 0)then begin
				sd.wavelength=12398./PEEM_E    ; save photon energy as wavelength
			endif else sd.wavelength=43.
; ------- establish variable set needed for nsls SD
			x_title = 'X'   & 		y_title = 'Y'
			x_normal=1      &		y_normal=1
			rotated=0
			image_title = byte(file)
			tab_char = string(byte(9))
			i_data = 0
			i_col = 0
			i_row = 0
			n_data = 2
			image_data = fltarr(n_cols,n_rows,n_data)
			image_data(*,*,0) = tmp.d
			image_data(*,*,1)= 1		; unit for clock normalization
			wrstx_ax, file

; write directly to NCB - load data file into image_stack
	   ENDIF ELSE BEGIN
			image_stack(*,*,i_file) = tmp.d
			filename_ev_msec_list(i_file) = filebase + zero + ' ' + $
			    string(E_vals(i_file), format = '(F7.2)') + '  ' +  string(PEEM_dwell, format='(F5.2)')
		ENDELSE
	; ------ provide tracking for user
	axis_log, string(PEEM_E, format="('PEEM stack ',/,'E= ',f8.3)")
	ENDIF
ENDFOR

; ------------- bail if only single image to be read ---------------
if peem_multi NE 1 then begin
   	peem_struct=tmp
	axis_log, string(format = '("Energy = ",F7.2, " eV. Pixel size = ",I3, " nm.  Bin = ", I1,"*",I1,".  Exposure = ",I3," sec.")', $
	  	  peem_E, 1000*peem_scale, peem_bin, peem_bin, peem_dwell)
	WIDGET_CONTROL, Event.Top, /DESTROY
	return
endif

; ---------- prepare to return the data or a flag through the parent function
if peem_multi EQ 1 then begin
	if peem_ncb NE 1 then begin
; ------- write stack_list file
		t = strmid(filebase, 0, strpos(filebase,'_'))
		outlist = source_path + t + '.sl'
		openw,unitw, outlist,/get_lun
		printf, unitw, strcompress(source_Path)		; write path at top of file
		for i_file = 0, n_elements(filename_list)-1 DO printf, unitw, filename_list(i_file)
		close, unitw
		text = 'stack_list saved in ' + strcompress(outlist)
		axis_log, text
		peem_struct = 1	; signify successful completion
	endif else begin
; --------------- write ncb file  ----------------------
;       	filename = source_path + strmid(filebase, 0, strpos(filebase,'_')) +'.ncb'
;       	print, filename
		filename = pickfile2(/write, filter = '*.ncb', /lpath, DefPath=DefPath)
        widget_control,hourglass=1
		x_start = tmp.x(0) & x_stop = tmp.x(n_elements(tmp.x)-1)
		y_start = tmp.y(0) & y_stop = tmp.y(n_elements(tmp.y)-1)
		ev = E_vals(0:n_files-1)
; reverse data if in wrong order
	;	test = dialog_message(/question, 'Energies out of order. Sort ?')
	;	if test EQ 'No' then goto, end_ev_sort
		sort_ev = sort(ev)
		timg = image_stack
		tlbl = filename_ev_msec_list
		for i = 0,n_elements(ev)-1 do begin
			image_stack(*,*,i) = timg(*,*,sort_ev(i))
			filename_ev_msec_list(i) = tlbl(sort_ev(i))
		endfor
		ev = ev(sort(ev))
		end_ev_sort:
		stack_wb, filename
        widget_control,hourglass=0
        peem_struct = 1
    endelse
 endif ; else begin
; 	peem_struct = 1
; endelse

WIDGET_CONTROL, Event.Top, /DESTROY

return

end


; ###################################################
; ----- routines to set state of control buttons ----
; ###################################################
pro BUT_no_correct, Event
@ax_peem_com
on_error,2

	PEEM_process = 'no correct'
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_dark_file')
	WIDGET_CONTROL, id, sensitive = 0
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_dark_browse')
	WIDGET_CONTROL, id, sensitive = 0
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_gain_file')
	WIDGET_CONTROL, id, sensitive = 0
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_gain_browse')
	WIDGET_CONTROL, id, sensitive = 0
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_last_dark')
 	WIDGET_CONTROL, id, sensitive = 0
end
; ###################################################
pro BUT_subtract_dark, Event
@ax_peem_com
on_error,2

	PEEM_process = 'subtract dark'
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_dark_file')
	WIDGET_CONTROL, id, sensitive = 1
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_dark_browse')
	WIDGET_CONTROL, id, sensitive = 1
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_gain_file')
	WIDGET_CONTROL, id, sensitive = 0
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_gain_browse')
	WIDGET_CONTROL, id, sensitive = 0
	IF PEEM_dark_avg EQ 1 then dark_avg_sens = 1 else dark_avg_sens = 0
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_last_dark')
 	WIDGET_CONTROL, id, sensitive = dark_avg_sens
end
; ###################################################
pro BUT_gain, Event
@ax_peem_com
on_error,2

	PEEM_process = 'gain correct'
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_dark_file')
	WIDGET_CONTROL, id, sensitive = 0
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_dark_browse')
	WIDGET_CONTROL, id, sensitive = 0
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_gain_file')
	WIDGET_CONTROL, id, sensitive = 1
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_gain_browse')
	WIDGET_CONTROL, id, sensitive = 1
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_last_dark')
 	WIDGET_CONTROL, id, sensitive = 0
end
; ###################################################
pro BUT_dark_gain, Event
@ax_peem_com
on_error,2

	PEEM_process = 'dark and gain correct'
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_dark_file')
	WIDGET_CONTROL, id, sensitive = 1
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_dark_browse')
	WIDGET_CONTROL, id, sensitive = 1
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_gain_file')
	WIDGET_CONTROL, id, sensitive = 1
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_gain_browse')
	WIDGET_CONTROL, id, sensitive = 1
	IF PEEM_dark_avg EQ 1 then dark_avg_sens = 1 else dark_avg_sens = 0
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_last_dark')
 	WIDGET_CONTROL, id, sensitive = dark_avg_sens
end
; ###################################################
pro BUT_median, Event
@ax_peem_com
on_error,2

    if PEEM_median EQ 1 then PEEM_median = 0 else PEEM_median = 1
end
; ###################################################
pro BUT_bits12, Event
@ax_peem_com
on_error,2

    if PEEM_bits12 EQ 1 then PEEM_bits12 = 0 else PEEM_bits12 = 1
;	print, '12 bit data flag to', PEEM_bits12
end

; ###################################################
pro BUT_region, Event
@ax_peem_com
on_error,2

    if PEEM_region EQ 1 then PEEM_region = 0 else PEEM_region = 1
;	print, 'region flag set to', PEEM_region
	if peem_region EQ 0 then reg_sens = 0 else reg_sens = 1
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_xmin')
	WIDGET_CONTROL, id , sensitive = reg_sens
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_xmax')
	WIDGET_CONTROL, id , sensitive = reg_sens
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_ymin')
	WIDGET_CONTROL, id , sensitive = reg_sens
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_ymax')
	WIDGET_CONTROL, id , sensitive = reg_sens
end

; ###################################################
; ----- routines to set parameters -----
; ###################################################
pro TXT_energy, Event
@ax_peem_com
on_error,2

	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_energy')
	WIDGET_CONTROL, id, Get_Value = test
	PEEM_E = float(test(0))
	print, 'Energy (eV) changed to', PEEM_E
end
; ###################################################
pro TXT_scale, Event
@ax_peem_com
on_error,2

	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_scale')
	WIDGET_CONTROL, id, Get_Value = test
	PEEM_scale = 0.001*float(test(0))
	print, 'Pixel size (nm) changed to', PEEM_scale
end
; ###################################################
pro TXT_bin, Event
@ax_peem_com
on_error,2

	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_bin')
	WIDGET_CONTROL, id, Get_Value = test
	PEEM_bin = fix(test(0))
	print, 'Binning changed to', PEEM_bin, ' x ', PEEM_bin
end
; ###################################################
pro TXT_dwell, Event
@ax_peem_com
on_error,2

	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_dwell')
	WIDGET_CONTROL, id, Get_Value = test
	PEEM_dwell = fix(test(0))
	print, 'Dwell (s) changed to', PEEM_dwell
end
; ###################################################
; ------ routines to set file names ----
; ###################################################
pro BUT_first_data_Browse, Event
@axis_com
@ax_peem_com
on_error,2

; display intent in log file
	axis_log, 'getting name of first PEEM data file'
; get current value of first_data_file
   	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_first_data_file')
	WIDGET_CONTROL, id, Get_Value = test
	PEEM_first = string(test(0))
;	print, 'Current first data file is ', PEEM_first
	PEEM_first = dialog_pickfile(file=PEEM_first, path = source_path, $
		filter='*.tif',title='PEEM read in: first data file', Get_Path=stest)
	source_path = stest
; update text widget with name of first data file unless no value
 	IF PEEM_first NE '' then begin
		id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_first_data_file')
	 	WIDGET_CONTROL, id, Set_Value = PEEM_first
	ENDIF
end
; ###################################################
pro TXT_first, Event
@ax_peem_com
on_error,2

	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_first_data_file')
	WIDGET_CONTROL, id, Get_Value = test
	PEEM_first = test(0)
end
; ###################################################
pro BUT_last_data_Browse, Event
@axis_com
@ax_peem_com
on_error,2

; get current value of last_data_file
   	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_last_data_file')
	WIDGET_CONTROL, id, Get_Value = test
	PEEM_last = string(test(0))
; display intent in log file
	axis_log, 'getting name of last PEEM data file'
;	print, 'Current last data file is ', PEEM_last
	PEEM_last = dialog_pickfile(file=PEEM_last, path = source_path, $
		filter='*.tif',title='PEEM read in: last data file', Get_Path=stest)
; update text widget with name of last data file unless no value
 	IF PEEM_last NE '' then begin
		source_path = string(stest(0))
 		id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_last_data_file')
 		WIDGET_CONTROL, id, Set_Value = PEEM_last
	ENDIF
end
; ###################################################
pro TXT_last, Event
@ax_peem_com
on_error,2

	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_last_data_file')
	WIDGET_CONTROL, id, Get_Value = test
	PEEM_last = test(0)
end
; ###################################################
pro BUT_dark_Browse, Event
@axis_com
@ax_peem_com
on_error,2

; ---------- get name of dark file
 	PEEM_dark = pickfile2(file=PEEM_dark, filter='*.tif',title='PEEM read in: dark correction file')
; update text widget with name of first data file unless no value
 	IF PEEM_dark NE '' then begin
		dark_path = DefPath
		id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_dark_file')
	 	WIDGET_CONTROL, id, Set_Value = PEEM_dark
	ENDIF
end
; ###################################################
pro TXT_dark, Event
@ax_peem_com
on_error,2

	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_dark_file')
	WIDGET_CONTROL, id, Get_Value = test
	PEEM_dark = test(0)
end
; ###################################################
pro BUT_dark_avg, Event
@axis_com
@ax_peem_com
on_error,2

 	IF PEEM_dark_avg EQ 1 then  PEEM_dark_avg = 0 else PEEM_dark_avg = 1
 	IF PEEM_process EQ 'subtract dark' OR peem_process EQ 'dark and gain correct' then begin
		IF PEEM_dark_avg EQ 1 then dark_avg_sens = 1 else dark_avg_sens = 0
		id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_last_dark')
	 	WIDGET_CONTROL, id, sensitive = dark_avg_sens
	ENDIF
end
; ###################################################
pro TXT_last_dark, Event
@ax_peem_com
on_error,2

	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_last_dark')
	WIDGET_CONTROL, id, Get_Value = test
	PEEM_dark_last = test(0)
end
; ###################################################
pro BUT_gain_Browse, Event
@axis_com
@ax_peem_com
on_error,2

; get current value of gain file
 	PEEM_gain = pickfile2(file=PEEM_gain, filter='*.tif',title='PEEM read in: gain correction file')
 	IF PEEM_gain NE '' then begin
		gain_path = DefPath
		id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_gain_file')
	 	WIDGET_CONTROL, id, Set_Value = PEEM_gain
	ENDIF
end
; ###################################################
pro TXT_gain, Event
@ax_peem_com
on_error,2

	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_gain_file')
	WIDGET_CONTROL, id, Get_Value = test
	PEEM_gain = test(0)
end
; ###################################################
pro TXT_xmin, Event
@ax_peem_com
on_error,2

	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_xmin')
	WIDGET_CONTROL, id, Get_Value = test
 	t = size(region) & if t(n_elements(t)-2) EQ 0 then region = fltarr(4)
	region(0) = float(test(0))
end
; ###################################################
pro TXT_xmax, Event
@ax_peem_com
on_error,2

	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_xmax')
	WIDGET_CONTROL, id, Get_Value = test
 	t = size(region) & if t(n_elements(t)-2) EQ 0 then region = fltarr(4)
	region(1) = float(test(0))
end
; ###################################################
pro TXT_ymin, Event
@ax_peem_com
on_error,2

	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_ymin')
	WIDGET_CONTROL, id, Get_Value = test
 	t = size(region) & if t(n_elements(t)-2) EQ 0 then region = fltarr(4)
	region(2) = float(test(0))
end
; ###################################################
pro TXT_ymax, Event
@ax_peem_com
on_error,2

	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_ymax')
	WIDGET_CONTROL, id, Get_Value = test
 	t = size(region) & if t(n_elements(t)-2) EQ 0 then region = fltarr(4)
	region(3) = float(test(0))
end

; ###################################################
pro BUT_AOI_Browse, Event
@axis_com
@ax_peem_com
on_error,2

; get current value of AOI file
	PEEM_aoi = pickfile2(file=PEEM_aoi, filter='*.aoi',title='PEEM read in: area of interest file')
 	IF PEEM_aoi NE '' then begin
		id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_AOI_file')
	 	WIDGET_CONTROL, id, Set_Value = PEEM_aoi
	ENDIF
end
; ###################################################
pro TXT_aoi, Event
@ax_peem_com
on_error,2

	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_AOI_file')
	WIDGET_CONTROL, id, Get_Value = test
	PEEM_AOI = test(0)
end
; ###################################################
pro BUT_AOI_gain, Event
@ax_peem_com
on_error,2

    if PEEM_aoi_gain EQ 1 then PEEM_aoi_gain = 0 else PEEM_aoi_gain = 1
	if PEEM_aoi_dark EQ 1 OR PEEM_aoi_gain EQ 1 then aoi_sens = 1 else aoi_sens = 0
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_AOI_file')
	WIDGET_CONTROL, id, sensitive = aoi_sens
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_AOI_browse')
	WIDGET_CONTROL, id, sensitive = aoi_sens
end
; ###################################################
pro BUT_AOI_dark, Event
@ax_peem_com
on_error,2

    if PEEM_aoi_dark EQ 1 then PEEM_aoi_dark = 0 else PEEM_aoi_dark = 1
	if PEEM_aoi_dark EQ 1 OR PEEM_aoi_gain EQ 1 then aoi_sens = 1 else aoi_sens = 0
	id = widget_info(Event.top, FIND_BY_UNAME = 'TXT_AOI_file')
	WIDGET_CONTROL, id, sensitive = aoi_sens
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_AOI_browse')
	WIDGET_CONTROL, id, sensitive = aoi_sens
end

; ###################################################
; ---------AX_READ_PEEM GUI processing---------------
; ###################################################
pro BASE_Options_event, Event, _extra=e
@ax_peem_com
on_error,2

  wWidget =  Event.top

  case Event.id of

    Widget_Info(wWidget, FIND_BY_UNAME='BASE_Options'): begin
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_Cancel'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_Cancel, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_Read_data'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_Read_data, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_no_correct'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_no_correct, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_subtract_dark'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_subtract_dark, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_gain'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_gain, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_dark_gain'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_dark_gain, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_median'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_median, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_bits12'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_bits12, Event
    end
 	Widget_Info(wWidget, FIND_BY_UNAME='BUT_region'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_region, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_first_data_browse'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_first_data_Browse, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_last_data_browse'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_last_data_Browse, Event
	end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_dark_browse'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_dark_Browse, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_gain_browse'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_gain_Browse, Event
	end
 	Widget_Info(wWidget, FIND_BY_UNAME='BUT_AOI_browse'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_AOI_browse, Event
    end
   Widget_Info(wWidget, FIND_BY_UNAME='TXT_first_data_file'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then $
        TXT_first, Event
    end
   Widget_Info(wWidget, FIND_BY_UNAME='TXT_last_data_file'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then $
        TXT_last, Event
    end
   Widget_Info(wWidget, FIND_BY_UNAME='TXT_dark_file'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then $
        TXT_dark, Event
    end
   Widget_Info(wWidget, FIND_BY_UNAME='TXT_gain_file'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then $
        TXT_gain, Event
    end
   Widget_Info(wWidget, FIND_BY_UNAME='TXT_AOI_file'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then $
        TXT_AOI, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_AOI_gain'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_AOI_gain, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_AOI_dark'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_AOI_dark, Event
    end

	Widget_Info(wWidget, FIND_BY_UNAME='BUT_dark_avg'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_dark_avg, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='TXT_last_dark'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then $
        TXT_last_dark, Event
    end

    Widget_Info(wWidget, FIND_BY_UNAME='TXT_energy'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then $
        TXT_energy, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='TXT_scale'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then $
        TXT_scale, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='TXT_bin'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then begin
      	print, 'in bin processing'
        TXT_bin, Event
      endif
    end
    Widget_Info(wWidget, FIND_BY_UNAME='TXT_dwell'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then $
        TXT_dwell, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='TXT_xmin'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then begin
        TXT_xmin, Event
      endif
    end
    Widget_Info(wWidget, FIND_BY_UNAME='TXT_xmax'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then begin
        TXT_xmax, Event
      endif
    end
    Widget_Info(wWidget, FIND_BY_UNAME='TXT_ymin'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then begin
        TXT_ymin, Event
      endif
    end
    Widget_Info(wWidget, FIND_BY_UNAME='TXT_ymax'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then begin
        TXT_ymax, Event
      endif
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_AOI_dark'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_AOI_dark, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_AOI_gain'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_AOI_gain, Event
    end

    else: print, 'unknown button pressed'
  endcase

end

; ###################################################
; -------- AX_PEEM_READ ----- Dialog generation -----
; ###################################################

pro BASE_Options, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_

@axis_com
@ax_peem_com
on_error,2

; ----- run modal if called from AXIS2000
 IF axis_on THEN begin
 	modal_val=1
 	wgroup = axis_ID
 ENDIF ELSE begin
 	modal_val=0
 	wgroup = 0
 ENDELSE

  BASE_Options = Widget_Base( GROUP_LEADER=wGroup,  $
      UNAME='BASE_Options' ,SCR_XSIZE=610 ,SCR_YSIZE=380  $
      , NOTIFY_REALIZE='' $
      , modal = modal_val $
      , TITLE='PEEM Read Widget'  $
      ,SPACE=3 ,XPAD=3 ,YPAD=3)

  LBL_X_offset = 80  & LBL_Y_offset = 80 + 8
  LBL_first_data_file = Widget_Label(BASE_Options,  $
      UNAME='LBL_first_data_file', /ALIGN_LEFT  $
      ,XOFFSET=LBL_X_offset ,YOFFSET = 3 $
      ,VALUE='First /single data file')
  LBL_last_data_file = Widget_Label(BASE_Options,  $
      UNAME='LBL_last_data_file',/ALIGN_LEFT  $
     ,XOFFSET=LBL_X_offset ,YOFFSET=LBL_Y_offset + 55 $
      ,VALUE='Last data file')
  LBL_dark_file = Widget_Label(BASE_Options, $
  	   UNAME='LBL_dark_file',/ALIGN_LEFT  $
  	  ,XOFFSET=LBL_X_offset ,YOFFSET=LBL_Y_offset + 112 $
      ,VALUE='PEEM dark file')
  LBL_gain_file = Widget_Label(BASE_Options, $
       UNAME='LBL_gain_file',/ALIGN_LEFT  $
      ,XOFFSET=LBL_X_offset ,YOFFSET=LBL_Y_offset + 160 $
      ,VALUE='PEEM gain file')
  LBL_AOI_file = Widget_Label(BASE_Options, $
      UNAME='LBL_AOI_file',/ALIGN_LEFT  $
      ,XOFFSET=LBL_X_offset ,YOFFSET=LBL_Y_offset + 220 $
      ,VALUE='Area of Interest file')
; select files to apply AOI
  BASE_AOI_options = Widget_Base(BASE_Options, UNAME='BASE_AOI_options'  $
      ,XOFFSET=LBL_X_offset +120,YOFFSET=LBL_Y_offset + 208, /align_center $
      ,SCR_XSIZE=220 ,SCR_YSIZE=25 , ROW = 1,/NONEXCLUSIVE)
  BUT_AOI_dark = Widget_Button(BASE_AOI_options, UNAME='BUT_AOI_dark'  $
      , XOFFSET = 60, /ALIGN_LEFT ,VALUE='AOI*dark')
  BUT_AOI_gain = Widget_Button(BASE_AOI_options, UNAME='BUT_AOI_gain'  $
      , XOFFSET = 150, /ALIGN_LEFT ,VALUE='AOI*gain')

browse_X_off = 5
browse_Y_off = 22 + 80
 BUT_first_data_Browse = Widget_Button(BASE_Options,  $
      UNAME='BUT_first_data_browse' $
      ,XOFFSET=browse_X_off ,YOFFSET=15  $
      ,/ALIGN_CENTER ,VALUE='Browse')
 BUT_last_data_Browse = Widget_Button(BASE_Options,  $
      UNAME='BUT_last_data_browse' $
      ,XOFFSET=browse_X_off, YOFFSET =browse_Y_off + 55 $
      ,/ALIGN_CENTER , VALUE='Browse')
 BUT_dark_Browse = Widget_Button(BASE_Options,  $
      UNAME='BUT_dark_browse' $
      ,XOFFSET=browse_X_off ,YOFFSET= browse_Y_off + 115 $
      ,/ALIGN_CENTER, VALUE='Browse')
  BUT_gain_Browse = Widget_Button(BASE_Options,  $
      UNAME='BUT_gain_browse' $
 	  ,XOFFSET=browse_X_off ,YOFFSET= browse_Y_off + 160 $
      ,/ALIGN_CENTER ,VALUE='Browse')
  BUT_AOI_browse = Widget_Button(BASE_Options,  $
      UNAME='BUT_AOI_browse' $
       ,XOFFSET=browse_X_off ,YOFFSET= browse_Y_off + 220 $
       ,/ALIGN_CENTER ,VALUE='Browse')

filenm_Xsiz = 50
filenm_X_off = 80  & filenm_Y_off = 25 + 80
 TXT_first_data_file = Widget_Text(BASE_Options,  $
      UNAME='TXT_first_data_file' $
      ,XOFFSET=filenm_X_off ,YOFFSET=20  $
      ,/EDITABLE, /all_events, XSIZE=filenm_Xsiz ,YSIZE=1)
 TXT_last_data_file = Widget_Text(BASE_Options,  $
      UNAME='TXT_last_data_file' $
      ,XOFFSET=filenm_X_off , YOFFSET=filenm_Y_off + 55  $
      ,/EDITABLE, /all_events, XSIZE=filenm_Xsiz ,YSIZE=1)
  TXT_dark_file = Widget_Text(BASE_Options, $
  	  UNAME='TXT_dark_file'  $
      ,XOFFSET=filenm_X_off ,YOFFSET=filenm_Y_off + 115 $
      ,/EDITABLE, /all_events, XSIZE=filenm_Xsiz ,YSIZE=1)
  TXT_gain_file = Widget_Text(BASE_Options, $
  	  UNAME='TXT_gain_file'  $
      ,XOFFSET=filenm_X_off ,YOFFSET=filenm_Y_off + 160  $
      ,/EDITABLE, /all_events, XSIZE=filenm_Xsiz ,YSIZE=1)
  TXT_AOI_file = Widget_Text(BASE_Options, $
  	  UNAME='TXT_AOI_file'  $
      ,XOFFSET=filenm_X_off ,YOFFSET=filenm_Y_off +220  $
      ,/EDITABLE, /all_events, XSIZE=filenm_Xsiz ,YSIZE=1)

; ----- option to average multiple dark files
  BASE_dark_options = Widget_Base(BASE_Options, UNAME='BASE_dark_options'  $
      ,FRAME=0 ,XOFFSET=filenm_X_off + 120 , YOFFSET=filenm_Y_off + 85 ,SCR_XSIZE=140 ,SCR_YSIZE=30  $
      ,/NONEXCLUSIVE)
  BUT_dark_avg = Widget_Button(BASE_dark_options, UNAME='BUT_dark_avg'  $
      ,/ALIGN_LEFT ,VALUE='average dark files', frame = 0)
  LBL_last_dark = Widget_Label(BASE_Options, $
       UNAME='LBL_last_dark',/ALIGN_LEFT  $
       ,XOFFSET= 360 , YOFFSET= filenm_Y_off + 92 $
      ,VALUE= 'last dark')
  TXT_last_dark = Widget_Text(BASE_Options, UNAME='TXT_last_dark'  $
  	   ,XOFFSET= 415 ,YOFFSET= filenm_Y_off + 90  $
       ,/EDITABLE, /all_events,  XSIZE=2 ,YSIZE=1)

; -------- energy, scale and bin input boxes
  xnumb_off = 20
  ynumb_off = 50 & ynumb2 = 20
  LBL_energy = Widget_Label(BASE_Options, $
       UNAME='LBL_energy',/ALIGN_LEFT  $
      ,XOFFSET=xnumb_off, YOFFSET=  ynumb_off $
      ,VALUE='Photon Energy (eV)')
  TXT_energy = Widget_Text(BASE_Options, UNAME='TXT_energy'  $
      ,XOFFSET=xnumb_off, YOFFSET=  ynumb_off + ynumb2  $
      ,/EDITABLE, /all_events, XSIZE=10 ,YSIZE=1)
  LBL_scale = Widget_Label(BASE_Options, $
      UNAME='LBL_scale',/ALIGN_LEFT  $
      ,XOFFSET=xnumb_off + 150 ,YOFFSET= ynumb_off $
      ,VALUE='Pixel scale (nm)')
  TXT_scale = Widget_Text(BASE_Options, UNAME='TXT_scale'  $
      ,XOFFSET=xnumb_off +150 ,YOFFSET = ynumb_off + ynumb2  $
      ,/EDITABLE, /all_events,  XSIZE=10 ,YSIZE=1)
  LBL_bin = Widget_Label(BASE_Options, $
       UNAME='LBL_bin',/ALIGN_LEFT  $
      ,XOFFSET=xnumb_off + 260 ,YOFFSET= ynumb_off $
      ,VALUE= '#pixels to bin')
  TXT_bin = Widget_Text(BASE_Options, UNAME='TXT_bin'  $
  	   ,XOFFSET=xnumb_off +280 ,YOFFSET= ynumb_off + ynumb2  $
       ,/EDITABLE, /all_events,  XSIZE=3 ,YSIZE=1)
 LBL_dwell = Widget_Label(BASE_Options, $
       UNAME='LBL_dwell',/ALIGN_LEFT  $
      ,XOFFSET=xnumb_off + 360 ,YOFFSET= ynumb_off $
      ,VALUE= 'dwell (s)')
  TXT_dwell = Widget_Text(BASE_Options, UNAME='TXT_dwell'  $
  	   ,XOFFSET=xnumb_off +360 ,YOFFSET= ynumb_off + ynumb2  $
       ,/EDITABLE, /all_events,  XSIZE=3 ,YSIZE=1)

; ----- non file name buttons ---------

; ------ sub region definition --------
xregion_off = 80
region_lbl_off = 0
region_off = 16
BASE_Region = Widget_Base(BASE_Options, UNAME='BASE_Region' ,FRAME=1  $
      ,XOFFSET=8 ,YOFFSET=98 ,SCR_XSIZE=430 ,SCR_YSIZE=45)
LBL_region =  Widget_Label (BASE_Region, UNAME='LBL_region' $
  	   ,XOFFSET=2, YOFFSET= 20,VALUE = 'REGION:-', /ALIGN_LEFT)
LBL_xmin =  Widget_Label (BASE_Region, UNAME='LBL_xmin' $
  	   ,XOFFSET=xregion_off+10, YOFFSET=  region_lbl_off  $
       ,VALUE = 'X-min', /ALIGN_LEFT)
TXT_xmin =  Widget_Text (BASE_Region, UNAME='TXT_xmin'  $
  	   ,XOFFSET=xregion_off, YOFFSET= region_off  $
       ,/EDITABLE, /all_events, XSIZE=8 ,YSIZE=1)
reg_del = 90
LBL_xmax =  Widget_Label (BASE_Region, UNAME='LBL_xmax' $
  	   ,XOFFSET=xregion_off+reg_del+10, YOFFSET=  region_lbl_off  $
       ,VALUE = 'X-max', /ALIGN_LEFT)
TXT_xmax =  Widget_Text (BASE_Region, UNAME='TXT_xmax'  $
  	   ,XOFFSET=xregion_off+reg_del, YOFFSET= region_off  $
       ,/EDITABLE, /all_events, XSIZE=8 ,YSIZE=1)
LBL_ymin =  Widget_Label (BASE_Region, UNAME='LBL_ymin' $
  	   ,XOFFSET=xregion_off+2*reg_del+10, YOFFSET=  region_lbl_off  $
       ,VALUE = 'Y-min', /ALIGN_LEFT)
TXT_ymin =  Widget_Text (BASE_Region, UNAME='TXT_ymin'  $
  	   ,XOFFSET=xregion_off+2*reg_del, YOFFSET= region_off  $
       ,/EDITABLE, /all_events, XSIZE=8 ,YSIZE=1)
LBL_ymax =  Widget_Label (BASE_Region, UNAME='LBL_ymax' $
  	   ,XOFFSET=xregion_off+3*reg_del+10, YOFFSET=  region_lbl_off  $
       ,VALUE = 'Y-max', /ALIGN_LEFT)
TXT_ymax =  Widget_Text (BASE_Region, UNAME='TXT_ymax'  $
  	   ,XOFFSET=xregion_off+3*reg_del, YOFFSET= region_off  $
       ,/EDITABLE, /all_events, XSIZE=8 ,YSIZE=1)

option_X_off = 450
  LBL_Process_options = Widget_Label(BASE_Options, UNAME='LBL_Process_options'  $
      ,XOFFSET= option_X_off,YOFFSET=0, /ALIGN_CENTER  $
      ,VALUE='Processing Options')
  BASE_Process_options = Widget_Base(BASE_Options, UNAME='BASE_Process_options' ,FRAME=1  $
      ,XOFFSET=option_X_off ,YOFFSET=10 ,SCR_XSIZE=140 ,SCR_YSIZE=120  $
      ,COLUMN=1 ,/EXCLUSIVE)
  BUT_no_correct = Widget_Button(BASE_Process_options, UNAME='BUT_no_correct'  $
      ,/ALIGN_LEFT ,VALUE='No corrections')
  BUT_subtract_dark = Widget_Button(BASE_Process_options, UNAME='BUT_subtract_dark'  $
      ,YOFFSET=25 ,/ALIGN_LEFT ,VALUE='Subtract dark')
  BUT_gain = Widget_Button(BASE_Process_options, UNAME='BUT_gain'  $
      ,YOFFSET=50 ,/ALIGN_LEFT ,VALUE='gain correct')
  BUT_dark_gain = Widget_Button(BASE_Process_options, UNAME='BUT_dark_gain'  $
      ,YOFFSET=75 ,/ALIGN_LEFT ,VALUE='dark, gain correct')

  LBL_other_options = Widget_Label(BASE_Options,  $
      UNAME='LBL_other_options' ,XOFFSET=option_X_off ,YOFFSET=130  $
      ,/ALIGN_CENTER ,VALUE='Other options')
  BASE_other_options = Widget_Base(BASE_Options, UNAME='BASE_other_options'  $
      ,FRAME=1 ,XOFFSET=option_X_off ,YOFFSET=140 ,SCR_XSIZE=140 ,SCR_YSIZE=94  $
      ,COLUMN=1 ,/NONEXCLUSIVE)
  BUT_median = Widget_Button(BASE_other_options, UNAME='BUT_median'  $
      ,/ALIGN_LEFT ,VALUE='median smooth')
  BUT_bits12 = Widget_Button(BASE_other_options, UNAME='BUT_bits12'  $
      ,YOFFSET=25 ,/ALIGN_LEFT ,VALUE='12 bit data')
  BUT_region = Widget_Button(BASE_other_options, UNAME='BUT_region'  $
      ,YOFFSET=50 ,/ALIGN_LEFT ,VALUE='region only')

  BUT_Read_data = Widget_Button(BASE_Options, UNAME='BUT_Read_data'  $
      ,XOFFSET=option_X_off ,YOFFSET=245 ,SCR_XSIZE=140 ,SCR_YSIZE=50  $
      ,/ALIGN_CENTER ,VALUE='Read data')
  BUT_Cancel = Widget_Button(BASE_Options, UNAME='BUT_Cancel'  $
      ,XOFFSET=option_X_off ,YOFFSET=300 ,SCR_XSIZE=140 ,SCR_YSIZE=50  $
      ,/ALIGN_CENTER ,VALUE='Cancel')

; ---- set state of all parameters
  WIDGET_CONTROL, BUT_median, Set_Button = PEEM_median
  t = size(peem_bits12) & if t(n_elements(t)-2) EQ 0 then peem_bits12 = 1  ; set to 1 if undefined (default)
  WIDGET_CONTROL, BUT_bits12, Set_Button = PEEM_bits12
  WIDGET_CONTROL, TXT_first_data_file, Set_Value = PEEM_first
  t = size(peem_region) & if t(n_elements(t)-2) EQ 0 then peem_region = 0 ; set to 0 if undefined (default)
  WIDGET_CONTROL, BUT_region, Set_Button = peem_region
 if peem_region EQ 0 then begin
	WIDGET_CONTROL, TXT_xmin, sensitive = 0
	WIDGET_CONTROL, TXT_xmax, sensitive = 0
	WIDGET_CONTROL, TXT_ymin, sensitive = 0
	WIDGET_CONTROL, TXT_ymax, sensitive = 0
 endif else begin
 	t = size(region) & if t(n_elements(t)-2) EQ 0 then region = fltarr(4)
	WIDGET_CONTROL, TXT_xmin, Set_Value = string(format = '(F6.2)',region(0))
	WIDGET_CONTROL, TXT_xmax, Set_Value = string(format = '(F6.2)',region(1))
	WIDGET_CONTROL, TXT_ymin, Set_Value = string(format = '(F6.2)',region(2))
	WIDGET_CONTROL, TXT_ymax, Set_Value = string(format = '(F6.2)',region(3))
 endelse
 t = size(peem_AOI_dark) & if t(n_elements(t)-2) EQ 0 then peem_AOI_dark = 0  ; set to 0 if undefined (default)
 t = size(peem_AOI_gain) & if t(n_elements(t)-2) EQ 0 then peem_AOI_gain = 0  ; set to 0 if undefined (default)
;  print, ' peem_AOI_gain, peem_AOI_dark', peem_AOI_gain, peem_AOI_dark
 WIDGET_CONTROL, BUT_AOI_dark, Set_Button = PEEM_AOI_dark
 WIDGET_CONTROL, BUT_AOI_gain, Set_Button = PEEM_AOI_gain
 if peem_AOI_gain EQ 0 AND peem_AOI_dark EQ 0 then begin
  	  WIDGET_CONTROL, TXT_AOI_file,sensitive = 0
	  WIDGET_CONTROL, BUT_AOI_browse, sensitive = 0
 endif else begin
  	  WIDGET_CONTROL, TXT_AOI_file,sensitive = 1
	  WIDGET_CONTROL, BUT_AOI_browse, sensitive = 1
 endelse

  if peem_multi EQ 1 then begin
	  WIDGET_CONTROL, TXT_last_data_file, Set_Value = PEEM_last
  	  WIDGET_CONTROL, TXT_last_data_file,sensitive = 1
	  WIDGET_CONTROL, BUT_last_data_browse, sensitive = 1
	  WIDGET_CONTROL, BUT_Read_data, Set_Value = ' convert files'
  endif else begin
  	  WIDGET_CONTROL, TXT_last_data_file,sensitive = 0
	  WIDGET_CONTROL, BUT_last_data_browse, sensitive = 0
  endelse
  WIDGET_CONTROL, TXT_dark_file, Set_Value = PEEM_dark
  WIDGET_CONTROL, TXT_gain_file, Set_Value = PEEM_gain
  WIDGET_CONTROL, TXT_AOI_file, Set_Value = PEEM_aoi

  t = size(peem_e) & if t(n_elements(t)-2) EQ 0 then peem_e = 0.
  WIDGET_CONTROL, TXT_energy, Set_Value = string(format='(F6.2)',PEEM_E)
  t = size(peem_scale) & if t(n_elements(t)-2) EQ 0 then peem_scale = 1.
  WIDGET_CONTROL, TXT_scale, Set_Value = string(format='(I4)',1000.*PEEM_scale)
  t = size(peem_bin) & if t(n_elements(t)-2) EQ 0 then peem_bin = 1
  WIDGET_CONTROL, TXT_bin, Set_Value = string(format='(I2)',PEEM_bin)
  t = size(peem_dwell) & if t(n_elements(t)-2) EQ 0 then peem_dwell = 1
  WIDGET_CONTROL, TXT_dwell, Set_Value = string(format='(I3)',PEEM_dwell)
  t = size(peem_dark_avg) & if t(n_elements(t)-2) EQ 0 then peem_dark_avg = 0
  WIDGET_CONTROL, BUT_dark_AVG, Set_Button = PEEM_dark_avg
   t = size(peem_dark_last) & if t(n_elements(t)-2) EQ 0 then peem_dark_last = 0.
  WIDGET_CONTROL, TXT_last_dark, Set_Value = string(format='(I2)',PEEM_dark_last)
  WIDGET_CONTROL, TXT_last_dark, sensitive = 0

  CASE strtrim(PEEM_process,2) of
	'no correct': BEGIN
	  	WIDGET_CONTROL, BUT_no_correct, Set_Button = 1
	  	WIDGET_CONTROL, TXT_dark_file,sensitive = 0
		WIDGET_CONTROL, BUT_dark_browse, sensitive = 0
	  	WIDGET_CONTROL, TXT_gain_file,sensitive = 0
		WIDGET_CONTROL, BUT_gain_browse, sensitive = 0
	  END
	'subtract dark': BEGIN
		WIDGET_CONTROL, BUT_subtract_dark, Set_Button = 1
	  	WIDGET_CONTROL, TXT_dark_file, sensitive = 1
		WIDGET_CONTROL, BUT_dark_browse, sensitive = 1
	  	WIDGET_CONTROL, TXT_gain_file,sensitive = 0
		WIDGET_CONTROL, BUT_gain_browse, sensitive = 0
		IF peem_dark_avg EQ 0 then $
			WIDGET_CONTROL, TXT_last_dark, sensitive = 0 $
		    ELSE WIDGET_CONTROL, TXT_last_dark, sensitive = 1
	  END
	'gain correct':	BEGIN
		WIDGET_CONTROL, BUT_gain, Set_Button = 1
	  	WIDGET_CONTROL, TXT_dark_file,sensitive = 0
		WIDGET_CONTROL, BUT_dark_browse, sensitive = 0
	  	WIDGET_CONTROL, TXT_gain_file,sensitive = 1
		WIDGET_CONTROL, BUT_gain_browse, sensitive = 1
	  END
	'dark and gain correct': BEGIN
		WIDGET_CONTROL, BUT_dark_gain, Set_Button = 1
	  	WIDGET_CONTROL, TXT_dark_file,sensitive = 1
		WIDGET_CONTROL, BUT_dark_browse, sensitive = 1
	  	WIDGET_CONTROL, TXT_gain_file,sensitive = 1
		WIDGET_CONTROL, BUT_gain_browse, sensitive = 1
		IF peem_dark_avg EQ 0 then $
			WIDGET_CONTROL, TXT_last_dark, sensitive = 0 $
		    ELSE WIDGET_CONTROL, TXT_last_dark, sensitive = 1
	  END
	ELSE: BEGIN
		PEEM_process = 'no correct'
		WIDGET_CONTROL, BUT_no_correct, Set_Button = 1
	  	WIDGET_CONTROL, TXT_dark_file,sensitive = 0
		WIDGET_CONTROL, BUT_dark_browse, sensitive = 0
	  	WIDGET_CONTROL, TXT_gain_file,sensitive = 0
		WIDGET_CONTROL, BUT_gain_browse, sensitive = 0
	END
  ENDCASE

; -------- generate widget -----------------------
  Widget_Control, BASE_Options, /realize
 XManager, 'BASE_Options', BASE_Options
 PEEM_ID = BASE_options    ; define the group leader ID


; this works in VM (tif_convert - 20-nov-04) - but that is a PRO not a function
;
;widget_control, tif_convert_par.main_base, /realize
;xmanager, 'tif_convert', tif_convert_par.main_base


end


; ************************
; ------------------------
; ************************

function ax_peem_read, sphinx=sphinx, ncb=ncb, axis = axis, peem3=peem3, multi=multi, col12=col12  ;, $
;                       GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_
@axis_com
@ax_peem_com
on_error,2

source_path = DefPath
if keyword_set(axis) then axis_on = 1 else axis_on = 0
if keyword_set(col12) then peem_col12 = col12 else peem_col12 = 5
if keyword_set(multi) then peem_multi = 1 else peem_multi = 0
if keyword_set(sphinx) then peem_sphinx = 1 else peem_sphinx = 0
if keyword_set(ncb) then peem_ncb = 1 else peem_ncb=0
if keyword_set(peem3) then peem_3 = 1 else peem_3=0
peem_struct = 0

; implement widget
BASE_Options, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_

return, peem_struct

end
