; Copyright (c) 1998-2014 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		SAV_ALS
;
;LAST CHANGED: ----------------------------------- 	21 Jul 2014
;
; PURPOSE:
;	This function writes the image in the current aXis2000 buffer
; to a Beamline 7 STXM image file in ascii format (*,im*)
; image size, energy, dwell information are stored in (*.dat)
; NB different formats were used for (*.dat) at varous times
; this version matches the Apr-98 format
;
; CATEGORY:
;	Data I/O
;
; CALLING SEQUENCE:
;	Result = SAV_ALS(tmp, scale=scale, file=file, FILTER=fltr, _extra=e )
;
; INPUTS:
;  TMP	aXis2000 structure (internal representation of 2D image)
;
; KEYWORDS:
;	SCALE		scale factor to heep within 32-bit integer range
;	FILE		name of file to write
;	FILTER
;
; OUTPUTS:
;	write 2 files (*.im*, *.dat) to disk
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;
; MODIFICATION HISTORY:
; (13-jun-98 aph) first developed from sav_nsls & read_als; axis_com
; (29-dec-98 aph) added rescale for small dynamic range real data.
; ( 6-jan-99 aph) corrected row, col error !!
; (10-jan-99 aph) BadID check on Widget_control
; ( 7-jul-99 aph) FREE_LUN as well as close !
; (20-may-00 aph) write scale to DAT file; allow fixed scale via keyword
; (21-Jul-14 aph) AXIS standard header added
;-

function sav_als, tmp, scale=scale, file=file, FILTER=fltr, _extra=e
@axis_com
on_error,2

if n_tags(tmp) ne 0 then begin	; check there is an image to write
   if tmp.t EQ '2d' then begin
	       if not keyword_set(file) then begin  ;popup file dialog box if name not given
	          if not keyword_set(fltr) then fltr='*.im*'
	   		  file=PICKFILE2(/Write, FILTER=fltr, /LPATH, DEFPATH=defpath)
		    endif
		if strlen(file) GT 0 THEN BEGIN
	; allow any extension by picking up ext from pickfile choice (aph 14-feb-98)
		   spos=strpos(file,'.')
		   if (spos ne -1) then begin
		      ext = strmid(file,spos+1,3)    ; take extension
		      file=strmid(file,0,spos)		 ; strip extension
	       endif
		   file=strlowcase(file)

			x_start = 1.e-3* tmp.x(0)
			x_stop = 1.e-3*tmp.x(n_elements(tmp.x)-1)
			y_start = 1.e-3*tmp.y(0)
			y_stop = 1.e-3*tmp.y(n_elements(tmp.y)-1)
			x_step = 1.e-3*(tmp.x(1) - tmp.x(0))
			y_step = 1.e-3*(tmp.y(1) - tmp.y(0))
			im_size = size(tmp.d)
			n_cols = im_size(2)  & n_rows = im_size(1)

		; -------- scale tmp.d to avoid losing image quality when rounding to integers
		if not keyword_set(scale) then begin
			scale = 1.
			if(max(tmp.d) - min(tmp.d) LT 20 or max(abs(tmp.d)) GT 32767) then begin
				scale = 32000./max(abs(tmp.d))
				scale = 1000.
			endif
		endif
		tmp.d = tmp.d * scale

		; ------ write paramater file into 'file.dat'
		openw,lun,(file+'.dat'), /get_lun
		delay = 0.
		dwell = 1000.*sd.dwell_time
		energy = 12398./sd.wavelength
		printf, lun, x_start, x_step, n_cols
		printf, lun, y_start, y_step, n_rows
		printf, lun, delay, delay, dwell, energy
		if scale NE 1 then printf, lun, 'Scaled by ', scale
		close,lun & free_lun, lun


		; -------- write tmp.d as 'file.ext' '
		file = file + '.' + ext
		openw,lun,(file), /get_lun, width=n_rows*15
		for i = 0, n_cols-1 do printf, lun, fix(tmp.d(*,i))
		close, lun & free_lun, lun
		if scale NE 1. then begin
			text = STRING(FORMAT ='("scale: ",G10.4,/,"ALS-image wrote to ",A)', $
			   scale, file)
		endif else begin
			text = STRING(FORMAT ='("ALS-image wrote to ",A)', file)
		endelse
		print, text
		WIDGET_CONTROL, Uprompt, Bad_ID = BadID, SET_VALUE = text
	endif else begin 	; go here if user pushes 'cancel'
	   	print, ' file name of zero length. SAV_ALS aborted'
	   	FILE = ' '
	endelse
   endif
endif

return, file
end
