; Copyright (c) 1998-2023 A.P. Hitchcock  All rights reserved
;+
;NAME:  AX_CONVERT_TO_ODC
;
;
;LAST CHANGED: ----------------------------------- 	19 Feb 2023 (aph)

; PURPOSE:
;	This function is designed to be used as a macro ot auto-process STXM chemi-movies
;   It executes several stack transformations that are usually done in an interactive mode; using stack_process.
;  The procedure (i) reads in a STXM transmission stack; (ii) extracts the Io from a pre-defined ROI;
;   (iii) uses that Io to convert the stack to OD; (iv) applies an energy calibration
;
; CATEGORY:
;	    Runs stand alone only
;
; CALLING SEQUENCE:
; for procedures:
;	RESULT = AX_CONVERT_TO_ODC, parameters
;
; INPUTS: all by keywords

; KEYWORDS:
;	STACK	filename od stack to be converted from *j.ncb to *jodc.ncb., where *j.ncd is an aligned stack
;;	ROI:  	ROI for Io ==> MUST be compatible with the size of the stack images
;   ECAL:	energy calibration shift (+ve move E higher)
;
; OUTPUTS: *odc.ncb  in same folder as the input file
;
; COMMON BLOCKS:
; AXIS_COM	standard set of common blocks
; STACK_PROCESS_COM
; BSIF_COM
; volume_data, image_stack

; SPECIAL COMMENTS
; roi, and io_roi are contained within STACK_PROCESS_COM - be careful not to confuse roi_io and io_roi

; MODIFICATION HISTORY:
; (19-Feb-23 aph) first version
;-

FUNCTION ax_convert_to_odc, stack = stack, roi_io = roi_io, ecal = ecal, verbose = verbose
on_error,2
@AXIS_COM
@STACK_PROCESS_COM
@BSIF_COM
COMMON volume_data, image_stack

; ------- read in stack and set-up variables needed for processing

check = findfile(stack)
if strlen(check(0)) EQ 0 then begin
	print, stack, ' not found.  AX_CONVERT_TO_ODC terminated'
	return, 0
endif

t = ax_name(stack)
file_path = t(0)
fileshort = t(1)
stack_rb,stack, /silent
test = size(image_stack)
nimg = test(3) & nx = test(1) & ny = test(2)
if keyword_set(verbose) then print, stack, ' stack read'
if keyword_set(verbose) then print,  string(format='(i4," images. ",/,"   ",i4," x ",i4)', nimg, nx, ny)
;widget_control, Bad_ID=Bad_ID, Uprompt, SET_VALUE=text
x_step = (x_stop - x_start)/n_cols
y_step = (y_stop - y_start)/n_rows
xd = x_start + findgen(n_cols)*x_step
yd = y_start + findgen(n_rows)*x_step
svd_e = ev
od = fltarr(nimg,nx,ny)
for i = 0, nimg-1 do od(i,*,*) = image_stack(*,*,i)
t = ax_name(stack)
data_path = t(0)
stackname_short=t(1)

;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
; ------- extract ROI and save the extracted (Io)  spectrum
; note:*.roi files are binary anf start with 2 long variables which are # of i_roi and # of i0_roi
; while the roi file for the  set up movie from MAXYMUS was a i_roi, the code her should work for an i0_roi as well
if keyword_set(verbose) then print, 'roi_io file is ', roi_io

; IF (t(2) EQ 'roi') then begin			;JSO if filetype is ROI then read as normal
n_i_roi = long(0)
n_i0_roi = long(0)
get_lun,lun
openr,lun,roi_io,/xdr
readu,lun,n_i_roi,n_i0_roi
if keyword_set(verbose) then print, ' n_i_roi,n_i0_roi ' , n_i_roi,n_i0_roi

IF (n_i_roi NE 0) THEN BEGIN
 	i_roi = lonarr(n_i_roi)
 	readu,lun,long(i_roi)
 	i0_roi = i_roi
ENDIF
IF (n_i0_roi NE 0) THEN BEGIN
	i0_roi = lonarr(n_i0_roi)
 	readu,lun,long(i0_roi)
ENDIF

if keyword_set(verbose) then print, i0_roi

close,lun
free_lun,lun


i0_signal = fltarr(n_elements(ev))
FOR i_ev=0,(n_elements(ev)-1) DO BEGIN
	khz = image_stack(*,*,i_ev)
	i0_signal(i_ev) = total(khz(i0_roi))/float(n_elements(i0_roi))
ENDFOR
spectrum = i0_signal

if keyword_set(verbose) then begin
	print,'Io spectrum is '
	print, ev
	print, i0_signal
endif

; ----  have the Io spectrum; now save it; convert transmission stack to OD stack
t=ax_name(roi_io)
tmps = {t:'1d', x:float(ev), d:float(spectrum), dn:float(spectrum), xl:('Io-roi from stack ' + t(1)), dl:('Io for '+ fileshort) }
Io_file = file_path + fileshort + '_io.txt'
t_io = spc_save(tmps, file = Io_file, /noLbl, /silent)
t = ax_name(t_io)
if keyword_set(verbose) then print,'Io saved as ', t(1)

; ----  convert transmission stack to OD stack
for i = 0, n_elements(eV)-1 do image_stack(*,*,i) = -alog(image_stack(*,*,i)/i0_signal(i))


;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
; ------- apply energy calibration

if keyword_set(ecal) then begin
	ev = ev + ecal
; ---- update the energies in filename_ev_msec_list
    FOR i=0,(n_elements(filename_list)-1) DO BEGIN ; update list
    	str = filename_ev_msec_list(i)
    	if fix(strmid(!version.release,0,1)) GE 6 then begin
    		t=strsplit(str,/extract)                  ; updated method 22-Jul-10
    		t(1) = strtrim(string(ev(i),format='(f8.2)'),2)
	    	filename_ev_msec_list(i) = strjoin(t,' ')
    	endif else begin
; --------- STRSPLIT not in IDL5.2  so this is an alternative for IDL5.2
			t0 = ax_strmid(str,'',' ')		;  name of this image
    		t1 = ax_strmid(str,' ',' ')		; old energy
    		t2 = strmid(str,strlen(t0)+strlen(t1)+1)	; rest of string
    		t1 = strtrim(string(ev(i),format='(f8.2)'),2)	; new energy
	    	filename_ev_msec_list(i) = t0 + ' '+ t1 + ' ' + t2
		endelse
	ENDFOR
	if keyword_set(verbose) then print, 'Energy scale shifted by ', string(ecal),' eV'
endif

;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
; ------- write out *jodc.(dat, ncb) files

odc_stack = file_path + fileshort + 'odc.ncb'
stack_wb, odc_stack, /silent, /overwrite
if keyword_set(verbose) then print, fileshort, ' converted to ', fileshort + 'odc'

; return name of stack t0i be used for input to ax_sdf

RETURN,  odc_stack

END
