; Copyright (c) 1998-2015 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_XRF_STACK_HDF5
;
;LAST CHANGED: ----------------------------------- 23-Mar-15 (aph)
;
;PURPOSE:
;	This function reads in a single file containing X-ray fluorescence spectra
; at each pixel, written by XGLabs software
;
;CATEGORY:
;	STAND ALONE: read in utility
;
;CALLING SEQUENCE:
;	Result = READ_XRF_STACK_HDF5 (file=file, NOF=nof, verbose=verbose, group=group, $
;            no_write=no_write, spectr=spectr, silent=silent, energy = energy, one_image=one_image, extra = extra)
;
;CALLED FROM AXIS:
;	Read~Stacks~XRF~XGLabs~hdf5
;
;INPUTS: none. All input parameters are passed as keywords.
;
;KEYWORDS:
;	FILE 	filename to read
;	NOF		no filter
;	GROUP	group ID (aXis_ID)
;	ENERGY	energy of incident excitation (eleectron or X-ray)
;	one_image = index of the image from a stack that is requested to read to define E
;	VERBOSE print log information about processing
;	NO_WRITE  if set, do not write *.ncb (used to load image_stack when reading XAS-XRF 4d stack)
;	SILENT	if set, do not display parameters
;	SPECT - if set, return sum of spectra as axis 1d structure
;	EXTRA 	passed on parameters
;
;OUTPUTS:
;	AXIS (*,dat, *,ncb) stack; spectrum if SPECT keyword is set
;
;COMMON BLOCKS: none
;
;MODIFICATION HISTORY:
; (23-Jul-10 aph) first version - adapted from READ_XRF_STACK.PRO (25-May-10 version)
; (23-Mar-15 aph) corrected error when examing for another purpose
;-

Function read_xrf_stack_hdf5, file=file, NOF=nof,  verbose=verbose, group=group, $
            no_write=no_write, silent=silent, spect=spect, energy = energy, one_image=one_image, extra = extra
@axis_com
@stack_process_com
COMMON volume_data, image_stack
@bsif_com
on_error,2

if NOT keyword_set(group) then group = 0 else begin
	if n_elements(axis_ID) EQ 1 then group = axis_ID
 endelse

IF NOT keyword_set(file) then begin
	if keyword_set(nof) then fltr='*' else  fltr='*.hdf'
	file = pickfile2(/READ, FILTER=fltr, /LPATH, DEFPATH=defpath)
ENDIF
s = ''
if file EQ '' THEN RETURN, s  ; bail-out if no filename
check = findfile(file)		 ; bail-out if non-existent file
if strlen(check(0)) EQ 0 then begin
	axis_log, 'Cannot find ' + file
	return, s
endif

t= ax_name(file)
file_path = t(0)
fileshort = t(1)

widget_control, /hourglass

; -------------- READ in XGLabs HDF5 format xrf stack (XRF at each pixel) ---------------
	a = H5_browser(file, /dialog_read)
	xrf_stack = a._data

	t = size(xrf_stack)
;
	npts = t(1)
	nx = t(2)
	ny = t(3)
	x_start = 0  & x_stop = nx-1  & x_step = 1.
	y_start = 0  & y_stop = ny-1  & y_step = 1.

	IF NOT keyword_set(silent) then  print, 'nx, ny, #E-data points ', nx, ny, npts
  ax_wait
	n_cols=nx & n_rows=ny
	xrf_stack = transpose(xrf_stack)
	image_stack=reform(xrf_stack, n_cols, n_rows, npts)
  FOR i = 0, npts-1 do begin
    t = reform(image_stack(*,*,i))
    t = rotate(t,4)
    image_stack(*,*,i) = t
  ENDFOR
; ----- get incident electron energy (eV) -----
;	energy = 1050.
;	if keyword_set(group) then begin
;		energy = get_num(prompt='Photon energy ', val = energy, group = group)
;	endif else	energy = get_num(prompt='Photon energy ', val = energy)


; ---------------------------------------
; START OF WRITE OUT OF binary stack file
;------------------------------------------

; ------------- prepare & write aXsi2000 stack file
n_cols=nx & n_rows=ny			; needed to pass elsewhere via bsif_com
n_data = npts

; -------- Create aXis2000 stack
ev_max= npts & ev_min = 0
slope = (ev_max - ev_min)/npts
ev = ev_min + slope*findgen(npts)

filename_ev_msec_list = strarr(npts)
for i = 0, npts-1 do filename_ev_msec_list(i) = string(ev(i))

; --------- ask user what file name they want - default name is same as files without counter
if not keyword_set(no_write) then begin
	fileout = file_path + fileshort + '.ncb'
	sname=pickfile2(/write, file= fileout, filter='*.ncb', title = 'name of binary stack file')
	stack_wb, sname, /silent
endif

IF NOT keyword_set(silent) then begin
;	E_string = ' at E = ' + string(energy, format="(F8.2)") + ' eV'
	axis_log, 'XRF stack read from ' + fileshort  ;+ E_string
ENDIF

; -------- launch stack_process ----------------
stack_process, sname, /no_align, fpath=WritePath, /binary, /no_read

s=fileshort
if keyword_set(spect) then begin
  xrf_stack=reform(image_stack, n_cols*n_rows, npts)
  d=total(xrf_stack,1)
  s={t:'1d', d:d, x: ev, xl: 'XRF energy', dn: d, dl: 'Sum of ' + fileshort}
endif


return, s

end

