; Copyright (c) 1998-2000 A.P. Hitchcock  All rights reserved 
;+ 
;NAME: 
;	AX_RD_IM 
; 
;LAST CHANGED: ----------------------------------- 01-jan-00 
; 
;PURPOSE: 
;	This procedure reads ALS BL7 image files (*.im*) and 
; converts all files of same main name to a a single NetCDF format file. 
; used by AXIS to process ALS *.im* to *.nc files for use in stack_analyze 
; designed to process one file at a time 
; e.g. {90123001.im0, 90123001.im1} -> 90123001.nc 
; 
;CATEGORY: 
;	AXIS: stack analysis 
; 
;CALLING SEQUENCE: 
;	AX_RD_IM [, fileheader, bin=bin, img=img, help = help] 
; 
;CALLED FROM AXIS: 
;	->Stacks->bin NSLS->{1,file} 
 
;INPUTS: 
;	FILEHEADER - file with list of filenames 
; 
;KEYWORDS: 
;	BIN - smooth the image by binnning (e.g. bin =2 -> 2x2 -> 1 pixel) 
; 	IMG - sets filter to '*.img' for old-style ALS filenames 
;	HELP - prints a help message 
; 
;OUTPUTS: 
;	No explicit outputs.   A new window is created if necessary 
; 
;COMMON BLOCKS: 
;	@AXIS_COM	standard set of common blocks 
; 
;PROCEDURE: 
;	The filenames in FILEHEADER are read 
; Each file is read in using explicit open, readf, close commands 
; and therefore is dependent on knowing the explicit structure 
; of the ALS BL7.0 data files. Asssumes post Dec-97 filestructure. 
; 
;MODIFICATION HISTORY: 
; (Jacobsen, Jan-98) reads ALS data and structures it as a netCDF file (nsls) 
; (aph 23-Feb-98) added other parameters (SD = scan data) and binning 
; (aph 5-mar-98) set-up as an axis-specific routine to avoid conflict with CJ 
; (aph 16-Aug-98) removed processing of *.lst files written by STXM.vi 
; (aph 7-jan-99) trouble-shooting readin (Common name wrong !) 
; (01-jan-00 aph) AXIS standard documentation 
;- 
 
pro ax_rd_im, fileheader, bin=bin, img=img, help = help 
@axis_com
@bsif_com
on_error,2 
 
IF ((n_params() eq 0) or keyword_set(help)) THEN BEGIN 
    print,'ax_rd_im, fileheader     READ ALS BL7 files for conversion to netCDF' 
    print,'  Looks for files like "fileheader.dat", "fileheader.im0", "fileheader.im1"' 
    return 
ENDIF 
 
filelist = findfile(fileheader+'.*') 
svec = size(filelist) 
IF (svec(0) EQ 0) THEN BEGIN 
    print,'Could not find any files "'+fileheader+'.*"' 
    return 
ENDIF 
 
filelist = filelist(sort(filelist)) 
 
n_files = n_elements(filelist) 
keep_going = 1 
i_dat_file = (-1) 
i_file = 0 
WHILE (keep_going EQ 1) DO BEGIN 
    IF (strupcase(filelist(i_file)) EQ strupcase(fileheader+'.dat')) THEN BEGIN 
        i_dat_file = i_file 
        keep_going = 0 
    ENDIF ELSE BEGIN 
        i_file = i_file + 1 
        IF (i_file EQ n_files) THEN keep_going = 0 
    ENDELSE 
ENDWHILE 
 
IF (i_dat_file EQ (-1)) THEN BEGIN 
    print,'Could not find file "'+fileheader+'.dat"' 
    return 
ENDIF 
 
on_ioerror,bailout 
 
; read ALS *.dat file; put energy and dwell into the nsls SCANDATA (SD) structure 
openr,lun,filelist(i_dat_file), /get_lun 
x_start = float(0)    &   x_step = float(0) 
y_start = float(0)    &   y_step = float(0) 
n_cols = fix(0)       &   n_rows = fix(0) 
input_string = '' 
readf,lun,input_string 
reads,input_string,x_start,x_step,n_cols 
;print, '1:  ', input_string 
x_start = 1.e3*x_start  & xincr = 1.e3*x_step 
x_stop = x_start + xincr*n_cols 
readf,lun,input_string 
reads,input_string,y_start,y_step,n_rows 
;print, '2:  ', input_string 
y_start = 1.e3*y_start  & yincr = 1.e3*y_step 
y_stop = y_start + yincr*n_rows 
readf,lun,input_string 
;print, '3:  ', input_string 
;print, 'x-axis ',x_start,' to ', x_stop,' step ', xincr 
;print, 'y-axis ',y_start,' to ', y_stop,' step ', yincr 
on_ioerror, continue 
energy = 0. 
reads,input_string,delay,delay,dwell,energy 
continue:                ; when reading pre-Mar-98 files without photon energy 
on_ioerror,null 
sd.dwell_time = dwell/1000.    ; report dwell in msec 
close,lun 
if (energy NE 0)then begin 
;	print, format='("Energy (eV) = ",f7.3,". Dwell (ms) = ",f6.2)', energy, sd.dwell_time 
	sd.wavelength=12398./energy    ; save photon energy as wavelength 
endif 
; establish variable set needed for nsls SD 
x_title = 'X' 
y_title = 'Y' 
x_normal=1 
y_normal=1 
rotated=0 
image_title = byte(filelist(i_file)) 
 
image='im1'                        ; default image data in *.im1 (als transmitted signal) 
if keyword_set(img)then image=img 
;print,'  Image size: '+strtrim(string(n_cols),2)+'x'+strtrim(string(n_rows),2) 
 
tab_char = string(byte(9)) 
i_data = 0 
i_col = 0 
i_row = 0 
n_data = n_files-1 
image_data = intarr(n_cols,n_rows,n_data) 
this_int = fix(0) 
input_string = '' 
im_dat = 1 
FOR i_file = 0, (n_files-1) DO BEGIN 
    IF (i_file NE i_dat_file) THEN BEGIN 
   ; force the extension to be ONLY *.im* (aph 16-aug-98) 
        file=filelist(i_file) 
		spos=strpos(file,'.') 
		if (spos ne -1) then ext = strmid(file,spos+1,3)    ; take extension 
		test_ext = strlowcase(strcompress(ext)) 
		if (strmid(test_ext,0,2) NE 'im') then goto, next_file 
		if (ext EQ image) then im_dat=i_data 
;        print,'  Now reading file "'+filelist(i_file)+'"' 
        openr,lun,filelist(i_file) 
        FOR i_row = 0,(n_rows-1) DO BEGIN 
            readf,lun,input_string 
            int_array = intarr(n_cols) 
            reads,input_string,int_array 
            image_data(0:(n_cols-1),i_row,i_data) = int_array 
        ENDFOR 
        close,lun 
        i_col = 0 
        i_row = 0 
        i_data = i_data+1 
    	next_file : 
    ENDIF 
ENDFOR 
 
   IF keyword_set(bin) then begin 
   	IF bin GT 1 THEN BEGIN 
      if (float(n_cols)/float(bin))-fix(n_cols/bin) GT 0  $ 
   	        OR (float(n_rows)/float(bin))-fix(n_rows/bin) then begin 
        nct =fix(n_cols/bin)*bin  &  nrt = fix(n_rows/bin)*bin 
        xtra_c = n_cols - nct + 1 
        xtra_r = n_rows - nrt + 1 
;        print, 'truncate image to ', fix(nct), fix(nrt) 
        x_stop = x_stop*float(nct)/float(n_cols)  ; correct axis length for truncation 
        y_stop = y_stop*float(nrt)/float(n_rows) 
        tmp = image_data                      ;force size to integer mutiple of bin 
        image_data=  intarr(nct,nrt,n_data) 
		image_data=tmp(0:n_cols-xtra_c,0:n_rows-xtra_r,0:n_data-1) 
      endif 
      n_cols=fix(n_cols/bin) 
      n_rows=fix(n_rows/bin) 
;      print, 'bin image ', fix(bin), ' times,  to ',n_cols,'  x', n_rows 
      image_data=rebin(image_data,n_cols,n_rows,n_data) 
    ENDIF 
  ENDIF 
IF (min(image_data) LT 0) THEN BEGIN 
    image_data = long(image_data) 
    negatives = where(image_data LT 0) 
    image_data(negatives) = long(65536)+long(image_data(negatives)) 
ENDIF 
 
; force data from *.im1 to be in image_data(*,*,0) 
image_data(*,*,0) = image_data(*,*,im_dat) 
; make image_data(*,*,1) = 1.0 
if n_data EQ 1 then begin	; make 3d array if necessary 
	tmp = intarr(n_cols,n_rows,2) 
	tmp(*,*,0) = image_data(*,*,0) 
	image_data = tmp 
endif 
image_data(*,*,1)= 1 
sd.clock_frequency = 1. 
free_lun,lun 
return 
 
bailout: 
print,'Bailing out because of an error' 
close,lun 
free_lun,lun 
return 
END 
 
