; Copyright (c) 2005 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	CAPERS-SCALE
;
; last changed ----------------------------- 18-Sep-05 (aph)
;
; PURPOSE
;   This function sets the full size of a CaPeRS image read from a Lox file
; based on values of the Field Lens and Projective 1 Lens currents
;
; CATEGORY:
;	Image processing.
;
; CALLING SEQUENCE:
;	Result = CAPERS-SCALE(P1, F1, HBIN=HBIN, VBIN=VBIN)
;
; INPUTS:
; P1 = Projective 1  lens current (mA)
; FL = Field Lens    lens current (mA)
;
; KEYWORDS:
;  HBIN = integer binning in horizontal direction (on-camera)
;  VBIN = integer binning in vertical   direction (on-camera)
;
; COMMON: none
;
; WARNINGS
;
; Note that different Elmitec microscopes have different lenses
; thus the lens current = pixel size look up table contained in the CAPERS_SCALE.PRO file
; may be have to be parameterized to correctly dal with Elmitec microscopes other than CaPeRS
;
; The {FoV, P1, FL} values and thus limits of the ranges for the preset image modes
; were taken from the values in the LEEM2000 program of CaPeRS in August 2005.
;
; If reduced portions of the camera output were used the image size must be appropriately adapted
; An alternative  would be to return the pixel size, not the full image size
; However, if there is binning done on-camera (most often the case) that information must be taken into account
; to get the correct pixel size; Since the camera binning parameters are stored in the Lox header
; this is possible to implement.
;
; If the image data is binned after read-in, the pixel size must be appropriately modified
;
; MODIFICATION HISTORY:
; (18-Sep-05 aph) first version
;
;-

Function CaPeRS_Scale, P1, FL, hBin=hbin, vBin=vbin
;
;   Field of View (um)     Projective 1    Field Lens            CONDITION
;  -------------------    -----------     -----------    ------------------------
;        150                1240            2950         P1 > 1000 AND  FL >2800
;        100                1240            2500         P1 > 1000 AND  FL >2400
;         75                1240            2300         P1 > 1000 AND  FL >2100
;         50                1240            2050         P1 > 1000 AND  FL >1900
;         40                1240            1850         P1 > 1000 AND  FL >1800
;         30                   0            2775         P1 < 1000 AND  FL >2500
;         20                   0            2300         P1 < 1000 AND  FL >2200
;         15                   0            2150         P1 < 1000 AND  FL >2050
;         10                   0            1950         P1 < 1000 AND  FL >1900
;          8                   0            1800         P1 < 1000 AND  FL >1750
;          4                   0            1700         P1 < 1000 AND  FL >1600
;          1.7                 0            1550         P1 < 1000 AND  FL >1500


; Variables in aXis200
; FL = Field Lens
; P1 = Projective 1
; FoV = Field of View
;
; Should probably put these into an additional file, but for now define in the read-in routine
; C => coarse  (P1 = 1240 mA)    F => fine  (P1 = 0)
  C40 =  1800  & C50 = 1900  & C75 = 2100   &  C100 = 2400  &  C150 = 2800
  F1p7 = 1500  & F4  = 1600  & F8  = 1750   &  F10  = 1900  &  F20  = 2200  &  F30 = 2500
  FoV150 = 150  & FoV100 = 100  &  FoV75 = 75  &  FoV50  = 50  &  FoV40  = 40
  FoV30  = 30   & FoV20  = 20   &  FoV15 = 15  &  FoV10  = 10  &  FoV8   = 8  & FoV4  = 4  & FoV1p7  = 1.7

IF P1 GT 1000 then begin
FoV = 40   ; set a default value so there is a finite size no matter what
;	if FL GT C40  then FoV = FoV40
	if FL GT C50  then FoV = FoV50
	if FL GT C75  then FoV = FoV75
	if FL GT C100 then FoV = FoV100
	if FL GT C150 then FoV = FoV150
endif else begin
FoV = 1.7   ; set a default value so there is a finite size no matter what
;	if FL GT F1p7 then FoV = FoV1p7
	if FL GT F4   then FoV = FoV4
	if FL GT F8   then FoV = FoV8
	if FL GT F10  then FoV = FoV10
	if FL GT F20  then FoV = FoV20
	if FL GT F30  then FoV = FoV30
endelse

; ---- for possible implementation of pixel size rather than imae size
; so far, we have assumed PEEM is always operated with square pixels.
; The LEEM2000 software restricts you to that since the camera output is garbage if not
; What about Lox ?
;
; pixel_size = FoV/1024.   ; pixel size in microns
;
; If keyword_set(HBIN) then pixel_size = pixel_size*HBIN
; If keyword_set(VBIN) then pixel_size = pixel_size*VBIN

Return, FoV

END
