; Copyright (c) 1998-2016 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	AXB_LOAD
;
;LAST CHANGED: ----------------------------------- 26-Apr-16 (aph)
;
;PURPOSE:
;	This function reads in AXIS format image or spectra from
; the XDR transferrable binary format. (Files written using AXB_SAVE).
;
;CATEGORY:
;	STAND ALONE: utility
;
;CALLING SEQUENCE:
;	Result = AXB_LOAD([ NOF=nof, error=err, file=file, title=title, _extra=e])
;
;CALLED FROM AXIS:
;	Read->Images->AXIS (and in many analysis routines)
;
;INPUTS: none. All input parameters are passed as keywords.
;
;KEYWORDS:
;	/NOF 	- no filter (default filter = '*.axb')
;	FILE 	- filename to read
;	/SILENT - do not display feedback on progress
;	_EXTRA 	- passed on parameters
;
;OUTPUTS:
;	RESULT = AXIS 1d or 2d structure
;
;COMMON BLOCKS:
;	@AXIS_COM	standard set of common blocks
;
;PROCEDURE:
;	AXB_LOAD uses the XDR external, machine-independent binary representation.
; Please see the IDL manual (help under 'XDR') for details.
; AXB_LOAD uses information on the first line to determine the structure type.
;
;EXAMPLE:
; Create a sample nxn image, write as binary using AXB_SAVE, read using AXB_LOAD
;	n = 100 & t = findgen(n) & d = dist(n)
;	a = {t:'2d', x:t, y:t, xl:'x (um)    E = 288.601 eV     dwell =  0.20 ms', yl:'y', d: d, dl:'test'}
;	t = AXB_SAVE(a)
;	b = AXB_LOAD(file = t)
;	splot2d, b
;
;MODIFICATION HISTORY:
; (09-aug-97 aph) isolated from AXIS
; (21-jan-98 aph) added s.dn component to 1d structure to be axis-compatible
; (18-jan-99 aph) allow silent readins if provide filename by keyword
; (08-aug-99 aph) image information printed
; (17-oct-99 aph) save energy & dwell in sd.wavelength, sd.dwell  so nsls write works !
; (02-jan-00 aph) AXIS standard documentation
; (26-sep-00 aph) store energy explcitly in 2d structure
; (13-Nov-05 aph) use axis_log for feedback
; (14-jul-07 aph) add /silent keyword
; (02-feb-11 aph) remove path from axis log
; (15-Jul-12 aph) get rid of NaN values when read image data
; (03-Jan-15 aph) add keyword TITLE to allow indication of type of file to load
; (26-Mar-16 aph) problem with doubles from read_nexus fixed in read_nexus (added ax_dts calls to convert to singles)
; (26 Apr-16 aph) strip out tracking print statements
;-

Function axb_load, NOF=nof, file=file, silent=silent, title = title,  _extra=e
@axis_com
@bsif_com
on_error,2

IF NOT keyword_set(file) then begin
	fltr='*.axb'
	if keyword_set(nof) then fltr='*'
	file = pickfile2(/READ, title=title, FILTER=fltr, /LPATH, DEFPATH=defpath)
ENDIF
s = ''
if file EQ '' THEN RETURN, s  ; bail-out if no filename
check = findfile(file)		 ; bail-out if non-existent file
if strlen(check(0)) EQ 0 then begin
	axis_log, 'Cannot find ' + file
	return, s
endif
openr, iunit, file, /get_lun, /XDR
typ = ''
readu, iunit, typ
tmp = ''
readu, iunit, tmp
tmpx = ''
readu, iunit, tmpx
; print, 'x-label = ', tmpx
CASE typ OF
'1d' : begin
	nx = 1
	readu, iunit, nx
    s = {t:'1d', x:fltarr(nx), d:fltarr(nx), dn:fltarr(nx), xl:tmpx, dl: tmp}
	a = fltarr(nx)
	readu, iunit, a   &  s.x = a
	readu, iunit, a   &  s.d = a
	s.dn = s.d
	t=ax_name(file)
	text= strcompress(string(nx)) + ' pt spectrum:  ' + t(1)
	if not keyword_set(silent) then axis_log, text
 end
 '2d' : begin
 	tmpy = ''
 	readu, iunit, tmpy
; 	print, 'y-label = ', tmpy
    nx = 1 &  ny = 1
	readu, iunit, nx
;	print, 'nx = ', nx
	x = fltarr(nx)
	readu, iunit, x
	readu, iunit, ny
;	print, 'ny = ', ny
	y = fltarr(ny)
	readu, iunit, y
	d = fltarr(nx,ny)
	readu, iunit, d
;
; ----- scan for NaN values and replace with minimum
	index_nan = where(~finite(d), count)
	if count GT 0 then d(index_nan)=min(d, /Nan)
;
;  extract energy, dwell of image from X-label
; (assumes it was generated when read from a different format)
	energy = 0. & dwell = 0.
	eq_pos = strpos(tmpx,'=')
	if eq_pos NE -1 then begin
		sp_pos = strpos(tmpx,'e',eq_pos+2)
		energy = float(strmid(tmpx,eq_pos+1,sp_pos-eq_pos-1))
		if energy NE 0. then sd.wavelength = 12398./energy
		eq_pos = strpos(tmpx,'=', sp_pos)
		sp_pos = strpos(tmpx,'m', eq_pos+2)
		dwell = float(strmid(tmpx,eq_pos+1,sp_pos-eq_pos-1))
		if dwell NE 0 then sd.dwell_time = dwell
	endif
	if not keyword_set(silent) then begin
		print, 'AXB_LOAD: read file from ', file
		t = ax_name(file)
		text = t(1)
		axis_log, text
		text = string(nx, ny, energy, dwell, format= $
	  '(i4," x",i4," pixels. E =",f7.2," eV. Dwell =",f5.2," ms.")')
		axis_log, text
	endif
	s = {t:'2d', x:x, y:y, d:d, e: energy, xl:tmpx, yl:tmpy, dl: tmp}	; add E

 end
else :  axis_log, ' Error on AXIS binary format image file read '
endcase
close, iunit
free_lun, iunit
return, s
end