; Copyright (c) 1998-2012 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_XAS_XRF_STACK
;
;LAST CHANGED: ----------------------------------- 01-Sep-12 (aph)
;
;PURPOSE:
;	This function reads in a set of XRF-stack files defined in a *.ssl names file
; = X-ray fluorescence spectra at each pixel of an image, over a set of additional control parameters
;    such as incident X-ray energy (e.g. stacks of stacks written by STXM_control)
;
;CATEGORY:
;	STAND ALONE: read in utility; used in aXis2000
;
;CALLING SEQUENCE:
;	Result = READ_XAS_XRF_STACK([file=file, NOF=nof, verbose=verbose, group=group)
;
;CALLED FROM AXIS:
;	Read->Stacks~XAS_XRF
;
;INPUTS: none. All input parameters are passed as keywords.
;
;KEYWORDS:
;	FILE 	filename to read
;	NOF		no filter
;	GROUP	group ID (aXis_ID)
;	VERBOSE print log information about processing
;	EXTRA 	passed on parameters
;
;OUTPUTS:
;	xas_xrf_stack - 4-D array (x,y,a,b)
;     where a = fluorescence X-ray energy, b = additional control (e.g. incident photon energy) ,
;
;COMMON BLOCKS: none
;
;MODIFICATION HISTORY:
; (20-Feb-10 aph) first version, adapted from read_xrf_stack.pro
; (01-Sep-12 aph) fix error in group name; allow user to generate '*.ssl' file
;-

Function read_xas_xrf_stack, file=file, NOF=nof,verbose=verbose, group=group, extra = extra
@axis_com
@stack_process_com
COMMON volume_data, image_stack
COMMON volume_data2, xas_xrf_stack
@bsif_com
on_error,2

if NOT keyword_set(group) then group = 0 else begin
	if n_elements(axis_ID) EQ 1 then group = axis_ID
 endelse

IF NOT keyword_set(file) then begin
	if keyword_set(nof) then fltr='*' else  fltr='*.ssl'
	file = pickfile2(/READ, FILTER=fltr, /LPATH, DEFPATH=defpath)
ENDIF
s = ''
; if needed, generate stack list from first / last file names
make_ssl=1
if file EQ '' THEN make_ssl=0
if make_ssl EQ 0 then BEGIN
 	fname1 = pickfile2(/READ, PROMPT='Select first XRF file', FILTER='*.xrf',/LPATH, DEFPATH=defpath)
 	fname2 = pickfile2(/READ, PROMPT='Select last  XRF file', FILTER='*.xrf',/LPATH, DEFPATH=defpath)
 	t1 = ax_name(fname1) & fname1s = t1(1)
 	t2 = ax_name(fname2) & fname2s = t2(1)
 	print, 'Combining ', fname1s, ' to ', fname2s
 	fcore = strmid(fname1s, 0, strlen(fname1s)-3)
	first = fix(strmid(fname1s, strlen(fname1s)-3, 3))
 	last =  fix(strmid(fname2s, strlen(fname2s)-3, 3))
	filelist = pickfile2(/WRITE, FILTER='*.ssl', /LPATH, DEFPATH=defpath)
	if filelist EQ '' THEN BEGIN
		RETURN, s  ; bail-out if no filename
	endif else begin
		openw, unit, filelist,/get_lun
		printf, unit, t1(0)
		for i = first, last do begin
			num = strcompress(string(1000 + i))
			num = strmid(num,2,4)
			file_next = fcore + num + '.xrf'
			printf,unit, strcompress(file_next)
		endfor
		close, unit & free_lun, unit
 	endelse
 	t = ax_name(filelist)
 	file = t(0) + t(1) + '.ssl'
 	print, 'SSL file ', file, ' written'
ENDIF

check = findfile(file)		 ; bail-out if non-existent file
if strlen(check(0)) EQ 0 then begin
	axis_log, 'Cannot find ' + file
	return, s
endif

t= ax_name(file)			; this will be name of the stored 4-d binary array
file_path = t(0)
fileshort = t(1)

widget_control, /hourglass

; -------------- READ in first STXM_Control xrf stack (XRF at each pixel) --------------------
; get file name array
stack_readlist, file, filename_list
n_stacks = n_elements(filename_list)
ph_E = fltarr(n_stacks)
; read first file to get final dimensions
tmp = read_XRF_stack(file=filename_list(0), one_image = 1, energy = energy, group=aXis_ID, /no_write)
ph_E(0)= energy
print, 'XAS_XRF stack dimensions (x,y,E) ', strtrim(fix(n_cols)), ' ' , strtrim(fix(n_rows)), ' ' ,  strtrim(fix(n_stacks))
; make xas_xrf_stack big array - WARNING can be many hundred megabytes !!
xas_xrf_stack = intarr(n_cols,n_rows, n_data,n_stacks)

xas_xrf_stack(*,*,*,0) = image_stack

; -------------- READ in sequence of STXM_Control xrf stacks

for i = 1, n_stacks-1 do begin
	widget_control, /hourglass
	tmp = read_XRF_stack(file=filename_list(i), energy = energy, $
	                   one_image = i+1, group=aXisID, /no_write, /silent)
	 ph_E(i)= energy
	E_string = ' at E = ' + string(energy, format="(F8.2)") + ' eV'
	t = ax_name(filename_list(i))
	axis_log, 'XRF stack read from ' + t(1)  + E_string
	xas_xrf_stack(*,*,*,i) = image_stack
endfor

ax_wait, /off

print, 'Size of 4d array'
print, n_cols, ' x ',  n_rows,' x ', n_data,' x ', n_stacks
print, 'Estimated file size  = ', fix(n_cols*n_rows*n_data*n_stacks/500000.), ' Mb'

; --------- ask user what file name they want - default name is same as *.ssl file
fileout = file_path + fileshort + '.ssb'
sname=pickfile2(/write, file= fileout, filter='*.ssb', title = 'name of 4-d stack file')
if strlen(sname) GT 0 then begin
	stack2_wb, sname, ph_E
endif else begin
	print, 'XAS-XRF data read but not saved'
endelse

return, xas_xrf_stack

;return, fileshort

end

