; Copyright (c) 1998-2016 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_XAS
;
;LAST CHANGED: ----------------------------------- 05-Oct-16 (aph)
;
;PURPOSE:
;	This procedure reads a spectrum  from  'XAS' format into NetCDF format (nsls)
;
;CATEGORY:
;	STAND ALONE:  spectral processing
;
;CALLING SEQUENCE:
; 	read_xas,filename,ev,all_data,header,header_only=header_only,help=help
;
;CALLED FROM AXIS:
;	Read-spectrum-NSLS-XAS
;
;INPUTS:
;	filename 	name of file
;	ev			energy array returned to caller
;	all_data	intensity array over a number of channels returned to caller
;	header		header returned to caller
;
;KEYWORDS:
;   hader_only	= read header only
;	help		= help
;
;OUTPUTS:
;	none
;
;COMMON BLOCKS:
;	none
;
;MODIFICATION HISTORY:
; file obtained from Jacobsen Feb-98; developed from Mancini / Hitchcock format
; (26-sep-00 aph) AXIS standard documentation
; (05-Oct-16 aph) adapt to read ISEELS database *.os for print purposes
;	trap out header as it exists in the file & return energy (col(1) & data (col(2,3)
;   NB this will only work for the 3_column format
;-

PRO read_xas,filename,ev,all_data,header,iseels=iseels, header_only=header_only,help=help

IF ((n_params() EQ 0) OR keyword_set(help)) THEN BEGIN
    print,'read_xas,filename,ev,all_data,header,[/header_only]'
    print,'  Header is a struct with members:'
    print,'    formula,common_name,edge,acquisition_mode,source_purity,'
    print,'    comments,delta_ev,min_ev,max_ev,yaxis,contact_person,'+$
      'write_date,'
    print,'    journal,authors,title,volume,issue_number,year,pages,'
    print,'    booktitle,editors,publisher,address'
    return
ENDIF

IF (NOT keyword_set(column)) THEN column=0

max_columns = 6

on_ioerror,nofile
get_lun,lun

sysname = strupcase(strmid(!version.os,0,3))
IF (sysname EQ 'VMS') THEN BEGIN
    dirname = 'public_disk:[x1_lib.xas]'
ENDIF ELSE IF (sysname EQ 'WIN') THEN BEGIN
    dirname = 'c:\xas\'
ENDIF ELSE BEGIN
    dirname = '/usr/local/www/xas/'
ENDELSE

filelist = findfile(filename)
svec = size(filelist)
IF (svec(0) NE 0) THEN BEGIN
    openr,lun,filename
ENDIF ELSE BEGIN
    openr,lun,dirname+filename
ENDELSE

on_ioerror, bailout

header_tags = ['Formula','Common name','Edge',$
               'Acquisition mode','Source and purity','Comments',$
               'Delta eV','Min eV','Max eV','Y axis','Contact person',$
               'Write date','Journal','Authors','Title',$
               'Volume','Issue number','Year','Pages','Booktitle',$
               'Editors','Publisher','Address','No. of Spectra']
header_tags = strupcase(header_tags)
header = {header,$
          formula: '',common_name: '',edge: '',$
          acquisition_mode: '',source_purity: '',comments: '',$
          delta_ev: '',min_ev: '',max_ev: '',yaxis: '',$
          contact_person: '',write_date: '',journal: '',$
          authors: '',title: '',volume: '',$
          issue_number: '',year: '',pages: '',$
          booktitle: '',editors: '',publisher: '',$
          address: ''}
i_header = {i_header,$
            i_formula: fix(0),i_common_name: fix(1),i_edge: fix(2),$
            i_acquisition_mode: fix(3),i_source_purity: fix(4),$
            i_comments: fix(5),$
            i_delta_ev: fix(6),i_min_ev: fix(7),i_max_ev: fix(8),$
            i_yaxis: fix(9),$
            i_contact_person: fix(10),i_write_date: fix(11),$
            i_journal: fix(12),$
            i_authors: fix(13),i_title: fix(14),i_volume: fix(15),$
            i_issue_number: fix(16),i_year: fix(17),i_pages: fix(18),$
            i_booktitle: fix(19),i_editors: fix(20),i_publisher: fix(21),$
            i_address: fix(22)}

i_number_of_spectra = 23
i_last_header = -1

this_line = ''
n_pts = 0
i_last_header = 0
n_is = 0			; counter for ISEELS header
iseels_header=strarr(1) ; array for ISEELS header
WHILE (NOT EOF(lun)) DO BEGIN
    readf,lun,this_line
    this_line = strtrim(this_line,2)

; ------------ trap-out header as-is in the file for ISEELS database files
	if n_pts EQ 0 then begin
		iseels_header(n_is) = this_line
;	   	print, n_is+1, ' ', this_line
	   	n_is = n_is + 1
	  	iseels_header = [iseels_header,' ']
	endif
    IF (strlen(this_line) GE 4) THEN BEGIN
        IF (strmid(this_line,0,2) EQ '*-') THEN BEGIN
            ;; ignore this line
            i_last_header = (-1)
        ENDIF ELSE IF (strmid(this_line,0,2) EQ '**') THEN BEGIN
            ;; ignore this line
            i_last_header = (-1)
        ENDIF ELSE IF (strmid(this_line,0,4) EQ '*   ') THEN BEGIN
            remaining_line = $
              strtrim(strmid(this_line,4,(strlen(this_line)-4)),2)
            CASE i_last_header OF
                i_header.i_formula: $
                  header.formula = header.formula + $
                  remaining_line
                i_header.i_common_name: $
                  header.common_name = header.common_name + $
                  remaining_line
                i_header.i_edge: $
                  header.edge = header.edge + remaining_line
                i_header.i_acquisition_mode: $
                  header.acquisition_mode = header.acquisition_mode + $
                  remaining_line
                i_header.i_source_purity: $
                  header.source_purity = header.source_purity + remaining_line
                i_header.i_comments: $
                  header.comments = header.comments + remaining_line
                i_header.i_delta_ev: $
                  header.delta_ev = header.delta_ev + remaining_line
                i_header.i_min_ev: $
                  header.min_ev = header.min_ev + remaining_line
                i_header.i_max_ev: $
                  header.max_ev = header.max_ev + remaining_line
                i_header.i_yaxis: $
                  header.yaxis = header.yaxis + remaining_line
                i_header.i_contact_person: $
                  header.contact_person = header.contact_person + $
                  remaining_line
                i_header.i_write_date: $
                  header.write_date = header.write_date + remaining_line
                i_header.i_journal: $
                  header.journal = header.journal + remaining_line
                i_header.i_authors: $
                  header.authors = header.authors + remaining_line
                i_header.i_title: $
                  header.title = header.title + remaining_line
                i_header.i_volume: $
                  header.volume = header.volume + remaining_line
                i_header.i_issue_number: $
                  header.issue_number = header.issue_number + remaining_line
                i_header.i_year: $
                  header.year = header.year + remaining_line
                i_header.i_pages: $
                  header.pages = header.pages + remaining_line
                i_header.i_booktitle: $
                  header.booktitle = header.booktitle + remaining_line
                i_header.i_editors: $
                  header.editors = header.editors + remaining_line
                i_header.i_publisher: $
                  header.publisher = header.publisher + remaining_line
                i_header.i_address: $
                  header.address = header.address + remaining_line
                i_number_of_spectra: ; do nothing
                ELSE: print,'  Continuation of unknown field: "'+$
                  remaining_line+'"'
            ENDCASE
        ENDIF ELSE IF (strmid(this_line,0,1) EQ '*') THEN BEGIN
            ;; a new header, not a continuing header
            i_last_header = -1
            colon_pos = strpos(this_line,':')
            IF (colon_pos EQ (-1)) AND NOT keyword_Set(ISEELS) THEN BEGIN
                print,'  Unknown header field "'+this_line+'"'
            ENDIF ELSE BEGIN
                this_tag = strupcase(strmid(this_line,2,(colon_pos-2)))
                remaining_line = $
                  strtrim(strmid(this_line,(colon_pos+1),$
                                 (strlen(this_line)-colon_pos)),2)
                match_index = where(this_tag EQ header_tags,n_matches)
                IF (n_matches NE 1) THEN BEGIN
                    if NOT keyword_Set(iseels) then print,'Unknown header "'+this_tag+'"'
                ENDIF ELSE BEGIN
                    i_last_header = match_index(0)
                    CASE match_index(0) OF
                        i_header.i_formula: $
                          header.formula = header.formula + $
                          remaining_line
                        i_header.i_common_name: $
                          header.common_name = header.common_name + $
                          remaining_line
                        i_header.i_edge: $
                          header.edge = header.edge + remaining_line
                        i_header.i_acquisition_mode: $
                          header.acquisition_mode = header.acquisition_mode + $
                          remaining_line
                        i_header.i_source_purity: $
                          header.source_purity = header.source_purity + $
                          remaining_line
                        i_header.i_comments: $
                          header.comments = header.comments + remaining_line
                        i_header.i_delta_ev: $
                          header.delta_ev = header.delta_ev + remaining_line
                        i_header.i_min_ev: $
                          header.min_ev = header.min_ev + remaining_line
                        i_header.i_max_ev: $
                          header.max_ev = header.max_ev + remaining_line
                        i_header.i_yaxis: $
                          header.yaxis = header.yaxis + remaining_line
                        i_header.i_contact_person: $
                          header.contact_person = header.contact_person + $
                          remaining_line
                        i_header.i_write_date: $
                          header.write_date = header.write_date + $
                          remaining_line
                        i_header.i_journal: $
                          header.journal = header.journal + remaining_line
                        i_header.i_authors: $
                          header.authors = header.authors + remaining_line
                        i_header.i_title: $
                          header.title = header.title + remaining_line
                        i_header.i_volume: $
                          header.volume = header.volume + remaining_line
                        i_header.i_issue_number: $
                          header.issue_number = header.issue_number + $
                          remaining_line
                        i_header.i_year: $
                          header.year = header.year + remaining_line
                        i_header.i_pages: $
                          header.pages = header.pages + remaining_line
                        i_header.i_booktitle: $
                          header.booktitle = header.booktitle + remaining_line
                        i_header.i_editors: $
                          header.editors = header.editors + remaining_line
                        i_header.i_publisher: $
                          header.publisher = header.publisher + remaining_line
                        i_header.i_address: $
                          header.address = header.address + remaining_line
                        i_number_of_spectra: ; do nothing
                        ELSE: print,'  Continuation of unknown field: "'+$
                          remaining_line+'"'
                    ENDCASE
                ENDELSE
            ENDELSE             ; line had a colon in it
        ENDIF ELSE BEGIN
            ;; This is past the header
            i_last_header = -1
            IF (NOT keyword_set(header_only)) THEN BEGIN
                on_ioerror, keep_going
                these_values = fltarr(1,max_columns)
                reads,this_line,these_values
                keep_going:
                IF (n_pts EQ 0) THEN BEGIN
                    data_array = fltarr(1,max_columns)
                    data_array(0,0:(max_columns-1)) = these_values
                    n_pts = 1
                ENDIF ELSE BEGIN
                    data_array = [data_array, these_values]
                    n_pts = n_pts+1
                ENDELSE
            ENDIF               ; if not header_only
        ENDELSE                 ; starts with "*" or not
    ENDIF                       ; strlen not zero
ENDWHILE

IF keyword_set(iseels) then begin
	header = iseels_header(0:n_elements(iseels_header)-3)
	for i = 0, n_elements(header)-1 do print, header(i)
ENDIF

IF keyword_set(header_only) THEN BEGIN
    ;; we're done!
    close,lun
    free_lun,lun
    return
ENDIF

;; well, we wanted more than just the header
ev = data_array(*,0)
n_ev = n_elements(data_array(*,0))
ev = reform(ev,n_ev)

data_array = data_array(*,1:(max_columns-1))
last_nonzero_column = max_columns-2
keep_going = 1
WHILE (keep_going EQ 1) DO BEGIN
    IF (total(abs(data_array(*,last_nonzero_column))) EQ 0.) THEN BEGIN
        last_nonzero_column = last_nonzero_column-1
        IF (last_nonzero_column LT 0) THEN BEGIN
            keep_going = 0
        ENDIF
    ENDIF ELSE BEGIN
        keep_going = 0
    ENDELSE
ENDWHILE

IF (last_nonzero_column EQ 0) THEN BEGIN
    n_data = 1
    all_data = reform(data_array(*,0),n_ev)
ENDIF ELSE BEGIN
    n_data = last_nonzero_column+1
    all_data = fltarr(n_data,n_ev)
    FOR i_data = 0,(n_data-1) DO BEGIN
        all_data(i_data,0:(n_ev-1)) = data_array(0:(n_ev-1),i_data)
    ENDFOR
ENDELSE

close,lun
free_lun,lun
return

bailout:
close,lun
free_lun,lun
print,'Error reading file "'+filename+'"'
return

nofile:
close,lun
free_lun,lun
print,'Could not read file "'+filename+'" (even in "'+dirname+'")'
return

END
