; Copyright (c) 1998-2009 C. Zimba & A.P. Hitchcock  All rights reserved
;+
; NAME:
;	ZSTACK_PLOT.PRO
;
; LAST CHANGED: --------------------------  03-Aug-09
;
; AUTHORS:
;	Carl G. Zimba (Photons UnLimited)
;	Adam Hitchcock (McMaster)
; PURPOSE:
;	Modification of display parameters used for ZSTACK.
;	Called by ZSTACK suite of procedures
; CATEGORY:
;	Data analysis.
; CALLING SEQUENCE:
;	zstack_display, subroutine, new_zoom=new_zoom
; INPUTS:
;	subroutine = name of zstack procedure from which zstack_display was invoked
;	filename_list,ev_list,msec_list,filename_display_list, and image_stack of zstack_common
;		must be previously defined via ZSTACK_ANALYZE.PRO
; KEYWORD PARAMETERS:
;	new_zoom = set to indicate a new image_zoom factor
; OUTPUTS:
;	NONE
;	x_shift, y_shift, corr_dim_stack, corr_stack, shifted_image_stack of zstack_align_common
;		are computed as output
; COMMON BLOCKS:
;	zstack_common	:
;		data_source				:	type of x-ray microscopy data: STXM:1, SXM:2, SM:3, ALS:4, POLY:5
;		data_directory				:	directory of data files
;		image_stack				:	3-D matrix of multiple x-ray microscope images
;		filename_list				:	list of filename corresponding to images in image_stack
;		ev_list					:	list of x-ray energies corresponding to images in image_stack
;		msec_list					:	list of dwell times corresponding to images in image_stack
;		filename_display_list			:	list of filename, x-ray energy, and dwell time corresponding to images in image_stack
;		displayed_file_index			:	index in image_stack, filename_list, msec_list, and filename_display_list currently being displayed or processed
;		n_files					:	number of images
;		list_filename				:	name of file to save or retrieve list of data files
;		shift_filename				:	filename of alignment shifts
;		x_shift					:	array of x-coordinate alignment shifts
;		y_shift					:	array of y-coordinate alignment shifts
;		data_shifted				:	0: data was not shifted and should not be clipped, 1: data was shifted and should be clipped, -1: denotes ZSTACK_PROFILE.PRO is being run as a stand-alone procedure
;		n_clipped_cols				:	number of columns in clipped image
;		n_clipped_rows				:	number of rows in clipped image
;		clipbox					:	vector of four points defining dimensions of region unclipped by alignment: [xleft,xright,ybot,ytop]
;	zstack_analyze_common
;		zstack_analyze_par			:	variables controlling ZSTACK Analyze dialog window
;		do_edge_clip				:	Clip edges of null data created by image alignment:	0: NO, 1: YES
;		do_despike					:	Despike data before alignment:	0: NO, 1: YES
;		do_median_filter			:	Median filter data before alignment:	0: NO, 1: YES
;		do_save_memory				:	Conserve memory by not creating duplicate arrays: 0: NO, 1: YES
;	zstack_build_list_common
;		zstack_build_list_par			:	variables controlling ZSTACK Buildlist dialog window
;		data_filelist				:	List of all data file within data_directory
;		this_filename				:	Name of currently selected data file
;		first_filename				:	Name of data file selected as the first file in sequence
;		last_filename				:	Name of data file selected as the last file in sequence
;		this_ev					:	X-ray energy of currently selected file
;		first_ev					:	X-ray energy of data file selected as the first file in sequence
;		last_ev					:	X-ray energy of data file selected as the last file in sequence
;		got_first					:	Logic control parameter: O: don't have first file, 1: have first file
;		got_last					:	Logic control parameter: O: don't have last file, 1: have last file
;		got_a_file					:	Logic control parameter: O: don't have a file, 1: have a file
;		delete_file_index			:	index of file to be deleted from list
;	zstack_align_common
;		zstack_align_par			:	variables controlling ZSTACK Align dialog window
;		file_num					:	vector of file numbers of image files
;		dragbox					:	vector of four points defining dimensions of region selected for alignment: [xleft,xright,ybot,ytop]
;		edge_enhance				:	edge enhancement parameter: 0: none, 1: sobel, 2: roberts
;		edgegauss_pixels			:	number of pixels used for edgeguass smoothing in ZSTACK_ALIGN_IMAGES.PRO, default = 3
;		edgefill					:	type of filling to occur at edges of shifted images: mean, median, max, min, zero
;		image_match				:	Image to align to: -1: preceding image, 0: constant image, 1: following image
;		constant_file_index			:	image of index to use as reference when aligning to a constant image
;		corr_max					:	0: use 3-pt fitted maximum of correlation function, 1: use center of mass of correlation function
;		maxshift					:	2-point vector of minimum and maximum values of x-shift and y_shift, and temp_x_shift and temp_y_shift within ZSTACK_ALIGN_TUNE
;		doalign_complete			:	-1: alignment in progress, 0: alignment not started, 1: alignment finished, 2	: using alignment from stored file
;		low_memory					:	Reflects keyword LOW_MEM: 0: use full memory allocation, 1: use reduced memory allocation
;		corr_stack					:	3-D matrix of correlation functions corresponding to image_stack
;		shifted_image_stack			:	3-D matrix of shifted images corresponding to image_stack
;		corr_dim_stack				:	3-D matrix (i,j,k) of maxima of correlation functions: i: x maximum, j: y maximum, k: image file
;		shift_threshold				:	mimimum shift required to actually shift the image
;	zstack_tune_common
;		zstack_tune_par				:	variables controlling ZSTACK Tune Alignment dialog window
;		temp_x_shift				:	vector of alignment shifts along x-coordinate for images in image_stack, obtained within ZSTACK_ALIGN_TUNE
;		temp_y_shift				:	vector of alignment shifts along y-coordinate for images in image_stack, obtained within ZSTACK_ALIGN_TUNE
;		init_x_shift				:	vector of initial alignment shifts along x-coordinate for images in image_stack
;		init_y_shift				:	vector of initial alignment shifts along y-coordinate for images in image_stack
;		image_center				:	array of pixel coordinates specifying the center (in pixels) of each image
;		corr_center				:	array of pixel coordinates specifying the center (in pixels) of each correlation function image
;		shifted_center				:	array of pixel coordinates specifying the center (in pixels) of each shifted image
;		subimage_center				:	array of pixel coordinates specifying the center (in pixels) of each subimage
;		subcorr_center				:	array of pixel coordinates specifying the center (in pixels) of each correlation function subimage
;		subshifted_center			:	array of pixel coordinates specifying the center (in pixels) of each shifted subimage
;		subimage_pixels				:	size of subimages in pixels
;		fid_pt					:	pixel coordinates of present fiducial point(s)
;		n_fid_pt					:	number of fiducial points
;		all_fid_pt					:	array of all fiducial points
;		fid_pt_index				:	index of fiducial point(s)
;	zstack_spectra_common
;		zstack_spectra_par			:	variables controlling ZSTACK Spectra dialog window
;		zstack_save_par				:	variables controlling ZSTACK Save dialog window
;		roi_index					:	vector of pixels corresponding to regions of interest for all i spectra
;		i_roi						:	vector of pixels corresponding to region of interest for current i spectrum
;		n_roi						:	number of regions of interest defined for i spectra
;		i0_roi					:	vector of pixels defining region of interest for i0 spectrum
;		is_i0_roi					:	Type of region of interest: 0: i0, 1: i
;		spectrum					:	2-D matrix (j,k) of spectra corresponding to defined regions of interest, j=0: i0 spectrum, j=1-14: i spectra, k corresponds to x-ray energy of image
;		save_spectra_type			:	Type of file to save spectra as:
;		spectra_filename_header		:	base filename of spectra and regions of interest to be saved on ZSTACK Spectra dialog window
;		roi_filename				:	filename for storage and retrieval of regions of interest points
;		save_filename_header			:	base filename of images to be saved on ZSTACK Save dialog window
;		i0_filename				:	base filename of i0 spectrum to be retrieved
;		i0_filetype				:	Type of file of retrieved I0 spectrum:	RAW, XAS, CSV
;		i_roi_color_index			:	vector of indices of colors for display of regions of interest for i spectra
;		i0_roi_color_index			:	color index for display of region of interest for i0 spectrum
;		nonintegral_zoom_data			:	scaled byte image used when image_zoom is non-integral
;	zstack_profile_common
;		zstack_profile_par			:	variables controlling ZSTACK Profile dialog window
;		profile_direction			:	0: along the column direction, 1: along the row direction
;		profile_pixel				:	pixel actively used to generate profile image and plot
;		profile_x_pixel				:	x pixel used to construct profile image
;		profile_y_pixel				:	y pixel used to construct profile image
;		profile_win_pixels			:	number of pixels for profile image window
;		profile_image				:	image of profile intensity and image number
;		profile_spectrum			:	spectrum arising from single pixel at center of cursor crosshairs in profile image
;		profile_y_min				:	mimimum of all profile spectra
;		profile_y_max				:	maximum of all profile spectra
;		profile_filename_header		:	base filename of profile intensity plot to be saved on ZSTACK Profile dialog window
;		profile_filename			:	filename of profile intensity plot to be saved on ZSTACK Profile dialog window
;		spectrum_filename_header		:	base filename of profile spectrum to be saved on ZSTACK Profile dialog window
;		spectrum_filename			:	filename of profile spectrum to be saved on ZSTACK Profile dialog window
;		spectra_filename			:	filename of profile spectra to be saved on ZSTACK Profile dialog window
;		profile_use				:	0: ZSTACK_PROFILE used as a subroutine, 1: ZSTACK_PROFILE used as a stand-alone application
;	zstack_display_common
;		zstack_plot_par			:	variables controlling ZSTACK Display dialog window
;		image_zoom					:	zoom factor for displaying images
;		subimage_zoom				:	zoom factor for subimages
;		movie_delay				:	delay used to display movie images of data stacks, dependent upon machine speed
;		disp_min					:	minimum percentage intensity for display of images
;		disp_max					:	maximum percentage intensity for display of images
;		disp_gamma					:	gamma factor for display of images
;		spectrum_display			:	Display spectra as: 1: Single Beam, 2: Percent Transmittance, 3: Absorbance
;		spectrum_offset				:	Offset used to plot spectra
;		init_zoom					:	initial zoom factor for displaying images (used in ZSTACK_SAVE and ZSTACK_TUNE)
;		movie_active				:	movie of images is active: 0: NO, 1: YES
;		profile_zoom				:	zoom factor for file number axis on profile images
;		image_range				:	scale images using: 0: intensity range of each image, 1: intensity range of full image stack
;		image_ratio				:	Display images normalized by: 0: inv_image_stack, 1: i0 spectrum
;		ratio_image_index			:	index of image to use when ratio images to a constant image
;		image_scale				:	Display images normalized by: 0: inv_image_stack, 1: i0 spectrum
;		image_invert				:	invert color bar: 0: NO, 1: YES
;		temp_old_display			:	initial array of display parameters, set at beginning of ZSTACK_DISPLAY
;		temp_new_display			:	array of display parameters updated as display options are changed
;		zstack_subroutine			:	subroutine from which ZSTACK_DISPLAY was called
;		plot_x_min					:	mimimum x-value to plot shifts or spectra
;		plot_x_max					:	maximum x-value to plot shifts or spectra
;		plot_y_min					:	mimimum y-value to plot shifts or spectra
;		plot_y_max					:	maximum y-value to plot shifts or spectra
;		x_autoscale				:	autoscale x-values on plots of shifts or spectra: 0: NO, 1: YES
;		y_autoscale				:	autoscale y-values on plots of shifts or spectra: 0: NO, 1: YES
;	zstack_color_common
;		bottom_color_index			:	index of lowermost color of gradient colorscale
;		top_color_index				:	index of uppermost color of gradient colorscale
;		black_color_index			:	index of black color
;		white_color_index			:	index of white color
;		plot_bkgd_color_index			:	color index for plot background, either black or white
;		plot_axes_color_index			:	color index for plot axes, either whilte or black
;		image_border_color_index		:	color index for image border in zstack_buildlist and zstack_profile dialog windows
;		dragbox_color_index			:	color index for dragbox used to define subregion for alignment
;		corr_ctr_color_index			:	color index for crosshair showing center of correlation function
;		corr_max_color_index			:	color index for crosshair showing maximum of correlation function
;		x_shift_color_index			:	color index for plotting of x-shift
;		y_shift_color_index			:	color index for plotting of y-shift
;		shift_cursor_color_index		:	color index for cursor in shift plot, indicating file number
;		tune_fiducial_color_index		:	color index for fiducial points in closeup of shifted image in zstack_tune dialog window
;		spectra_color_index			:	color indices (14) for plotting of spectra
;		spectra_cursor_color_index		:	color index for cursor in spectra plot, indicating x-ray energy
;		profile_color_index			:	color index for plotting of intensity profile and cursor in profile image indicating row or column
;		profile_cursor_color_index		:	color index for cursor in profile image, indicating x-ray energy
;		profile_spectrum_color_index	:	color index for plotting of profile spectrum
;		test1_color_index			:	color index #1 for testing
;		test2_color_index			:	color index #2 for testing
;		test3_color_index			:	color index #3 for testing
; SIDE EFFECTS:
;
; RESTRICTIONS:
;	Used as ZSTACK suite of routines
; PROCEDURE:
;	Called by ZSTACK_ALIGN.PRO, ZSTACK_BUILDLIST.PRO, ZSTACK_PROFILE.PRO,
;			ZSTACK_SAVE.PRO, ZSTACK_SPECTRA.PRO, ZSTACK_TUNE.PRO
; EXAMPLE:
;
; MODIFICATION HISTORY:
;
; (13oct00 cgz) altered definition of dragbox so that it is now [xleft,xright,ybot,ytop]
;		this makes it consistent with definition of other cursor boxes
;		and eliminates need for min and max testing
;		Changed logic statement from:	IF ((dragbox(2) NE 0) AND (dragbox(3) NE 0)) THEN BEGIN
;							to:	IF ((dragbox(1) NE 0) AND (dragbox(3) NE 0)) THEN BEGIN
;
; (29oct00 cgz) several modifications to be compatible with new version of zstack_save
; (05nov00 cgz) modified widget to include more choices on control, added image, spectral, and stack subtraction, not fully implemented
;				will be able to use any image, any spectrum for normalization
;			modified names of image normalization variables
;			modified values of image_ratio
; (28nov00 cgz) migrated zstack_display_colorbar routine to zstack_analyze.pro
; (23-may-04 aph) force color scale, 0 when plotting
; (21-mar-08 MJ)  changed to work on unix (Xscroll_size)
; (13-apr-08 aph) turn on pre-set window size (X_Scroll_size, Y_Scroll_size) only for non-Windows OS
; (07-mar-09 aph) added binary_filenmane to zstack_common
; (26-Jul-09 aph) remove all secondary calls to common blocks & replace with @zstack_(common) syntax
; ;               since the Eclipse workspace tags as errors even though it is an allowed IDL syntax
; (03-Aug-09 aph) change from base-10 to natural log for spectra (Larry Nitler Carnegie)
;-
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_plot_prep
;print,'zstack_plot_prep'
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

image_zoom = 1.
subimage_zoom = 1.
movie_delay = 0.02	; This is machine-dependent
disp_min = 0.
disp_max = 100.
disp_gamma = 1.
spectrum_offset = 0.
n_cols = 100
n_rows = 100
profile_win_pixels = 100
plot_x_min = 280
plot_x_max = 310
plot_y_min = 0
plot_y_max = 2
x_autoscale = 1
y_autoscale = 1
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_plot_desensitive
;print,'zstack_plot_desensitive'
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

widget_control, zstack_plot_par.image_zoom_label, sensitive = 0
widget_control, zstack_plot_par.movie_delay_label, sensitive = 0
widget_control, zstack_plot_par.spectrum_offset_label, sensitive = 0
widget_control, zstack_plot_par.xloadct_label, sensitive = 0
widget_control, zstack_plot_par.invertct_label, sensitive = 0
widget_control, zstack_plot_par.loadplotct_label, sensitive = 0
widget_control, zstack_plot_par.spectra_singlebeam_label, sensitive = 0
widget_control, zstack_plot_par.spectra_tranmittance_label, sensitive = 0
widget_control, zstack_plot_par.spectra_absorbance_label, sensitive = 0
widget_control, zstack_plot_par.plot_x_min_label, sensitive = 0
widget_control, zstack_plot_par.plot_x_max_label, sensitive = 0
widget_control, zstack_plot_par.x_autoscale_label, sensitive = 0
widget_control, zstack_plot_par.plot_y_min_label, sensitive = 0
widget_control, zstack_plot_par.plot_y_max_label, sensitive = 0
widget_control, zstack_plot_par.y_autoscale_label, sensitive = 0
widget_control, zstack_plot_par.image_label, sensitive = 0
IF (data_shifted EQ 1) THEN BEGIN
;	widget_control, zstack_plot_par.clipped_image_label, sensitive = 0
ENDIF
widget_control, zstack_plot_par.colorbar_label, sensitive = 0
widget_control, zstack_plot_par.filename_display_list_label, sensitive = 0
widget_control, zstack_plot_par.next_image_label, sensitive = 0
widget_control, zstack_plot_par.prev_image_label, sensitive = 0
widget_control, zstack_plot_par.play_movie_label, sensitive = 0
widget_control, zstack_plot_par.display_parameters_label, sensitive = 0
widget_control, zstack_plot_par.plot_label, sensitive = 0
widget_control, zstack_plot_par.update_label, sensitive = 0
widget_control, zstack_plot_par.close_label, sensitive = 0
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_plot_sensitive
;print,'zstack_plot_sensitive'
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common
;								:
;	image_zoom_label				All
;	disp_min_label					All
;	disp_max_label					All
;	disp_gamma_label				All
;	movie_delay_label				All
;	subimage_zoom_label				Tune
;	profile_zoom_label				Profile
;	spectrum_offset_label				Spectra, Profile, Save
;								:
;	raw_image_label					All
;	ratio_image_label				Spectra, Profile, Save
;	log_ratio_image_label				Spectra, Profile, Save
;	subtract_image_label				Buildlist, Spectra, Profile, Save
;								:
;	ratio_spectrum_label				Spectra, Profile, Save
;	log_ratio_spectrum_label			Spectra, Profile, Save
;	subtract_spectrum_label			Spectra, Profile, Save
;	subtract_stack_label				Spectra, Profile, Save
;								:
;	reference_image_label 			Spectra, Profile, Save
;	reference_image_menu_label 		Spectra, Profile, Save
;	reference_image_filename_label 		Spectra, Profile, Save
;	reference_image_factor_label		Spectra, Profile, Save
;	reference_spectrum_label 			Spectra, Profile, Save
;	reference_spectrum_menu_label 		Spectra, Profile, Save
;	reference_spectrum_filename_label 	Spectra, Profile, Save
;	reference_spectrum_factor_label		Spectra, Profile, Save
;	reference_stack_label 			Spectra, Profile, Save
;	reference_stack_filename_label 		Spectra, Profile, Save
;	reference_stack_factor_label		Spectra, Profile, Save
;								:
;	absolute_scale_label				All
;	relative_scale_label				All
;	image_range_label				All
;	stack_range_label				All
;	image_i0_scale_label				Spectra, Profile, Save
;								:
;	xloadct_label					All
;	invertct_label					All
;	loadplotct_label				All
;								:
;	spectra_singlebeam_label			Spectra, Profile, Save
;	spectra_tranmittance_label			Spectra, Profile, Save
;	spectra_absorbance_label			Spectra, Profile, Save
;								:
;	x_autoscale_label				Align, Tune, Spectra, Profile, Save
;	plot_x_min_label				Align, Tune, Spectra, Profile, Save
;	plot_x_max_label				Align, Tune, Spectra, Profile, Save
;	y_autoscale_label				Align, Tune, Spectra, Profile, Save
;	plot_y_min_label				Align, Tune, Spectra, Profile, Save
;	plot_y_max_label				Align, Tune, Spectra, Profile, Save
;								:
;	image_label					None
;	clipped_image_label				None
;	colorbar_label					None
;	filename_display_list_label		All
;	prev_image_label				All
;	next_image_label				All
;	play_movie_label				All
;	plot_label						Align, Tune, Spectra, Profile, Save
;								:
;	update_label					All
;	close_label					All
;								:
widget_control, zstack_plot_par.image_zoom_label, sensitive = 1
widget_control, zstack_plot_par.movie_delay_label, sensitive = 1
widget_control, zstack_plot_par.xloadct_label, sensitive = 1
widget_control, zstack_plot_par.invertct_label, sensitive = 1
widget_control, zstack_plot_par.loadplotct_label, sensitive = 1	; disabled
widget_control, zstack_plot_par.image_label, sensitive = 0
widget_control, zstack_plot_par.colorbar_label, sensitive = 0
widget_control, zstack_plot_par.filename_display_list_label, sensitive = 1
widget_control, zstack_plot_par.next_image_label, sensitive = 1
widget_control, zstack_plot_par.prev_image_label, sensitive = 1
widget_control, zstack_plot_par.play_movie_label, sensitive = 1
widget_control, zstack_plot_par.display_parameters_label, sensitive = 1
dummy = where(temp_new_display NE temp_old_display, count)
IF (count EQ 0) THEN BEGIN
	widget_control, zstack_plot_par.update_label, sensitive = 0
ENDIF ELSE BEGIN
	widget_control, zstack_plot_par.update_label, sensitive = 1
ENDELSE
widget_control, zstack_plot_par.close_label, sensitive = 1
CASE zstack_subroutine OF
;'buildlist' : BEGIN	; can't ever be here from zstack_buildlist, so delete
;	widget_control, zstack_plot_par.spectrum_offset_label, sensitive = 0
;	widget_control, zstack_plot_par.spectrum_offset_label, sensitive = 0
;	widget_control, zstack_plot_par.spectra_singlebeam_label, sensitive = 0
;	widget_control, zstack_plot_par.spectra_tranmittance_label, sensitive = 0
;	widget_control, zstack_plot_par.spectra_absorbance_label, sensitive = 0
;	widget_control, zstack_plot_par.x_autoscale_label, sensitive = 0
;	widget_control, zstack_plot_par.plot_x_min_label, sensitive = 0
;	widget_control, zstack_plot_par.plot_x_max_label, sensitive = 0
;	widget_control, zstack_plot_par.y_autoscale_label, sensitive = 0
;	widget_control, zstack_plot_par.plot_y_min_label, sensitive = 0
;	widget_control, zstack_plot_par.plot_y_max_label, sensitive = 0
;	IF (data_shifted EQ 1) THEN BEGIN
;		widget_control, zstack_plot_par.clipped_image_label, sensitive = 0
;	ENDIF
;	widget_control, zstack_plot_par.plot_label, sensitive = 0
;END
'align' : BEGIN
	widget_control, zstack_plot_par.spectrum_offset_label, sensitive = 0
	IF (data_shifted EQ 1) THEN BEGIN
		widget_control, zstack_plot_par.clipped_image_label, sensitive = 0
	ENDIF
	IF (doalign_complete GT 0) THEN BEGIN
		widget_control, zstack_plot_par.x_autoscale_label, sensitive = 1
		widget_control, zstack_plot_par.plot_x_min_label, sensitive = 1
		widget_control, zstack_plot_par.plot_x_max_label, sensitive = 1
		widget_control, zstack_plot_par.y_autoscale_label, sensitive = 1
		widget_control, zstack_plot_par.plot_y_min_label, sensitive = 1
		widget_control, zstack_plot_par.plot_y_max_label, sensitive = 1
		widget_control, zstack_plot_par.plot_label, sensitive = 1
	ENDIF ELSE BEGIN
		widget_control, zstack_plot_par.x_autoscale_label, sensitive = 0
		widget_control, zstack_plot_par.plot_x_min_label, sensitive = 0
		widget_control, zstack_plot_par.plot_x_max_label, sensitive = 0
		widget_control, zstack_plot_par.y_autoscale_label, sensitive = 0
		widget_control, zstack_plot_par.plot_y_min_label, sensitive = 0
		widget_control, zstack_plot_par.plot_y_max_label, sensitive = 0
		widget_control, zstack_plot_par.plot_label, sensitive = 0
	ENDELSE
END
'tune' : BEGIN
	widget_control, zstack_plot_par.spectrum_offset_label, sensitive = 0
	widget_control, zstack_plot_par.spectra_singlebeam_label, sensitive = 0
	widget_control, zstack_plot_par.spectra_tranmittance_label, sensitive = 0
	widget_control, zstack_plot_par.spectra_absorbance_label, sensitive = 0
	IF (data_shifted EQ 1) THEN BEGIN
		widget_control, zstack_plot_par.clipped_image_label, sensitive = 0
	ENDIF
	IF (doalign_complete GT 0) THEN BEGIN
		widget_control, zstack_plot_par.x_autoscale_label, sensitive = 1
		widget_control, zstack_plot_par.plot_x_min_label, sensitive = 1
		widget_control, zstack_plot_par.plot_x_max_label, sensitive = 1
		widget_control, zstack_plot_par.y_autoscale_label, sensitive = 1
		widget_control, zstack_plot_par.plot_y_min_label, sensitive = 1
		widget_control, zstack_plot_par.plot_y_max_label, sensitive = 1
		widget_control, zstack_plot_par.plot_label, sensitive = 1
	ENDIF ELSE BEGIN
		widget_control, zstack_plot_par.x_autoscale_label, sensitive = 0
		widget_control, zstack_plot_par.plot_x_min_label, sensitive = 0
		widget_control, zstack_plot_par.plot_x_max_label, sensitive = 0
		widget_control, zstack_plot_par.y_autoscale_label, sensitive = 0
		widget_control, zstack_plot_par.plot_y_min_label, sensitive = 0
		widget_control, zstack_plot_par.plot_y_max_label, sensitive = 0
		widget_control, zstack_plot_par.plot_label, sensitive = 0
	ENDELSE
END
'spectra' : BEGIN
	IF (n_elements(n_roi) NE 0) THEN BEGIN
		IF (n_roi GE 1) THEN BEGIN
			widget_control, zstack_plot_par.spectrum_offset_label, sensitive = 1
			widget_control, zstack_plot_par.spectra_singlebeam_label, sensitive = 1
			dummy = where(spectrum[0,*] NE 0,count)
			IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
				widget_control, zstack_plot_par.spectra_tranmittance_label, sensitive = 1
				widget_control, zstack_plot_par.spectra_absorbance_label, sensitive = 1
			ENDIF ELSE BEGIN
				widget_control, zstack_plot_par.spectra_tranmittance_label, sensitive = 0
				widget_control, zstack_plot_par.spectra_absorbance_label, sensitive = 0
			ENDELSE
		ENDIF
	ENDIF ELSE BEGIN
		widget_control, zstack_plot_par.spectrum_offset_label, sensitive = 0
		widget_control, zstack_plot_par.spectra_singlebeam_label, sensitive = 0
		widget_control, zstack_plot_par.spectra_tranmittance_label, sensitive = 0
		widget_control, zstack_plot_par.spectra_absorbance_label, sensitive = 0
	ENDELSE
	IF (data_shifted EQ 1) THEN BEGIN
		widget_control, zstack_plot_par.clipped_image_label, sensitive = 0
	ENDIF
	IF (total(spectrum NE 0) EQ 0) THEN BEGIN	; no spectral regions are defined
		widget_control, zstack_plot_par.y_autoscale_label, sensitive = 0
		widget_control, zstack_plot_par.plot_y_min_label, sensitive = 0
		widget_control, zstack_plot_par.plot_y_max_label, sensitive = 0
		widget_control, zstack_plot_par.x_autoscale_label, sensitive = 0
		widget_control, zstack_plot_par.plot_x_min_label, sensitive = 0
		widget_control, zstack_plot_par.plot_x_max_label, sensitive = 0
		widget_control, zstack_plot_par.plot_label, sensitive = 0
	ENDIF ELSE BEGIN	; some spectral regions exist
		widget_control, zstack_plot_par.y_autoscale_label, sensitive = 1
		widget_control, zstack_plot_par.plot_y_min_label, sensitive = 1
		widget_control, zstack_plot_par.plot_y_max_label, sensitive = 1
		widget_control, zstack_plot_par.x_autoscale_label, sensitive = 1
		widget_control, zstack_plot_par.plot_x_min_label, sensitive = 1
		widget_control, zstack_plot_par.plot_x_max_label, sensitive = 1
		widget_control, zstack_plot_par.plot_label, sensitive = 1
	ENDELSE
END
'profile' : BEGIN
	IF (n_elements(n_roi) NE 0) THEN BEGIN
		IF (n_roi GE 1) THEN BEGIN
			widget_control, zstack_plot_par.spectrum_offset_label, sensitive = 1
			widget_control, zstack_plot_par.spectra_singlebeam_label, sensitive = 1
			dummy = where(spectrum[0,*] NE 0,count)
			IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
				widget_control, zstack_plot_par.spectra_tranmittance_label, sensitive = 1
				widget_control, zstack_plot_par.spectra_absorbance_label, sensitive = 1
			ENDIF ELSE BEGIN
				widget_control, zstack_plot_par.spectra_tranmittance_label, sensitive = 0
				widget_control, zstack_plot_par.spectra_absorbance_label, sensitive = 0
			ENDELSE
		ENDIF
	ENDIF ELSE BEGIN
		widget_control, zstack_plot_par.spectrum_offset_label, sensitive = 0
		widget_control, zstack_plot_par.spectra_singlebeam_label, sensitive = 0
		widget_control, zstack_plot_par.spectra_tranmittance_label, sensitive = 0
		widget_control, zstack_plot_par.spectra_absorbance_label, sensitive = 0
	ENDELSE
		widget_control, zstack_plot_par.y_autoscale_label, sensitive = 1
		widget_control, zstack_plot_par.plot_y_min_label, sensitive = 1
		widget_control, zstack_plot_par.plot_y_max_label, sensitive = 1
		widget_control, zstack_plot_par.x_autoscale_label, sensitive = 1
		widget_control, zstack_plot_par.plot_x_min_label, sensitive = 1
		widget_control, zstack_plot_par.plot_x_max_label, sensitive = 1
		widget_control, zstack_plot_par.plot_label, sensitive = 1
END
'save' : BEGIN
	IF (n_elements(n_roi) NE 0) THEN BEGIN
		IF (n_roi GE 1) THEN BEGIN
			widget_control, zstack_plot_par.spectrum_offset_label, sensitive = 1
			widget_control, zstack_plot_par.spectra_singlebeam_label, sensitive = 1
			dummy = where(spectrum[0,*] NE 0,count)
			IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
				widget_control, zstack_plot_par.spectra_tranmittance_label, sensitive = 1
				widget_control, zstack_plot_par.spectra_absorbance_label, sensitive = 1
			ENDIF ELSE BEGIN
				widget_control, zstack_plot_par.spectra_tranmittance_label, sensitive = 0
				widget_control, zstack_plot_par.spectra_absorbance_label, sensitive = 0
			ENDELSE
		ENDIF
	ENDIF ELSE BEGIN
		widget_control, zstack_plot_par.spectrum_offset_label, sensitive = 0
		widget_control, zstack_plot_par.spectra_singlebeam_label, sensitive = 0
		widget_control, zstack_plot_par.spectra_tranmittance_label, sensitive = 0
		widget_control, zstack_plot_par.spectra_absorbance_label, sensitive = 0
	ENDELSE
	IF (data_shifted EQ 1) THEN BEGIN
		widget_control, zstack_plot_par.clipped_image_label, sensitive = 0
	ENDIF
	IF (total(spectrum NE 0) EQ 0) THEN BEGIN	; no spectral regions are defined
		widget_control, zstack_plot_par.y_autoscale_label, sensitive = 0
		widget_control, zstack_plot_par.plot_y_min_label, sensitive = 0
		widget_control, zstack_plot_par.plot_y_max_label, sensitive = 0
		widget_control, zstack_plot_par.x_autoscale_label, sensitive = 0
		widget_control, zstack_plot_par.plot_x_min_label, sensitive = 0
		widget_control, zstack_plot_par.plot_x_max_label, sensitive = 0
		widget_control, zstack_plot_par.plot_label, sensitive = 0
	ENDIF ELSE BEGIN	; some spectral regions exist
		widget_control, zstack_plot_par.y_autoscale_label, sensitive = 1
		widget_control, zstack_plot_par.plot_y_min_label, sensitive = 1
		widget_control, zstack_plot_par.plot_y_max_label, sensitive = 1
		widget_control, zstack_plot_par.x_autoscale_label, sensitive = 1
		widget_control, zstack_plot_par.plot_x_min_label, sensitive = 1
		widget_control, zstack_plot_par.plot_x_max_label, sensitive = 1
		widget_control, zstack_plot_par.plot_label, sensitive = 1
	ENDELSE
END
ELSE : BEGIN
	print,'Unrecognized zstack_subroutine : zstack_plot_sensitive'
END
ENDCASE
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_plot_loadplotct
;print,'zstack_plot_loadplotct'
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common


return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_plot_imgdisp,i_file
;print,'zstack_plot_imgdisp'
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

zstack_analyze_imgprep,i_file,image
zstack_analyze_bytescale, image, byte_image
; Display image, applying zoom factor
wset,zstack_plot_par.image_win
IF (image_zoom GE 1.0) THEN BEGIN
	IF ((image_zoom) EQ fix(image_zoom)) THEN BEGIN
		tv,rebin(byte_image,n_cols*image_zoom,n_rows*image_zoom,/sample),0,0
	ENDIF ELSE BEGIN
		tv,congrid(byte_image,n_cols*image_zoom,n_rows*image_zoom),0,0
	ENDELSE
ENDIF ELSE BEGIN
	IF ( ((1./image_zoom) EQ fix(1./image_zoom)) AND $
			((image_zoom*n_cols) EQ fix(image_zoom*n_cols)) AND $
			((image_zoom*n_rows) EQ fix(image_zoom*n_rows)) ) THEN BEGIN
		tv,rebin(byte_image,n_cols*image_zoom,n_rows*image_zoom,/sample),0,0
	ENDIF ELSE BEGIN
		tv,congrid(byte_image,n_cols*image_zoom,n_rows*image_zoom),0,0
	ENDELSE
ENDELSE
;  Plot dragbox (if it exists)
IF (zstack_subroutine EQ 'align') OR (zstack_subroutine EQ 'tune') OR $
			(zstack_subroutine EQ 'save') THEN BEGIN	; plot dragbox over image
	IF ((dragbox(1) NE 0) AND (dragbox(3) NE 0)) THEN BEGIN
		;  Determine dimensions of dragbox
		xleft = dragbox(0)>0
		xright = dragbox(1)<(n_cols-1)
		ybot = dragbox(2)>0
		ytop = dragbox(3)<(n_rows-1)
	ENDIF ELSE BEGIN  ; set dimensions of dragbox to entire image
		xleft = 0
		xright = n_cols-1
		ybot = 0
		ytop = n_rows-1
	ENDELSE
	;  Plot dragbox over image
	IF ((dragbox(1) NE 0) AND (dragbox(3) NE 0)) THEN BEGIN
		px = image_zoom*[xleft, xright, xright, xleft, xleft] ;X points
		py = image_zoom*[ybot, ybot, ytop, ytop, ybot] ;Y values
		plots,px,py,/device,color=dragbox_color_index
	ENDIF
ENDIF
; Display clipped shifted image in clipped_image_win
; 	if there is a non-zero shift, i.e., alignment was kept
; 	if shift is discarded, i.e., all zero, don't display clipped_win_image
IF (zstack_subroutine EQ 'align') OR (zstack_subroutine EQ 'tune') THEN BEGIN
	; Display shift images if alignment is complete
	IF (doalign_complete NE 0) THEN BEGIN
		; Display shifted image with dragbox
		IF (low_memory EQ 1) THEN BEGIN
			; Shift image here to form shifted image for display
			; Conserves memory since shifted_image_stack is not formed and used
			IF ((abs(x_shift(i_file)) GT shift_threshold) OR (abs(y_shift(i_file)) GT shift_threshold)) THEN BEGIN
				; Shift the image to form this_image
				zstack_shift,image_stack(*,*,i_file),x_shift(i_file),y_shift(i_file),this_image,edgefill=edgefill
			ENDIF ELSE BEGIN
				; Shift smaller than threshold
				this_image = image_stack(*,*,i_file)
			ENDELSE
		ENDIF ELSE BEGIN
			; Get shifted image from stored array
			this_image = shifted_image_stack(*,*,i_file)
		ENDELSE

		; Scale image intensity to minimum and maximum values
		IF (image_range EQ 0) THEN BEGIN
			min_image = min(image_stack(*,*,i_file))
			this_image = temporary(this_image) - min_image
			max_image = max(this_image)
		ENDIF ELSE BEGIN
			min_image = min(image_stack)
			this_image = temporary(this_image) - min_image
			max_image = max(image_stack)
		ENDELSE
		percent_image = 100. * this_image / max_image

		; Scale shifted image to fill gray scale range (same scale as raw image)
		display_image = (( ((percent_image - disp_min) $
					/ ((disp_max - disp_min)^disp_gamma)) >0.)<1.)
		byte_image = byte( float(bottom_color_index) + $
					float(top_color_index - bottom_color_index) * $
					display_image)

		; Display shifted image, applying zoom factor
		wset,zstack_plot_par.shifted_image_win
		IF (image_zoom GE 1.0) THEN BEGIN
			IF ((image_zoom) EQ fix(image_zoom)) THEN BEGIN
				tv,rebin(byte_image,n_cols*image_zoom,n_rows*image_zoom,/sample),0,0
		ENDIF ELSE BEGIN
				tv,congrid(byte_image,n_cols*image_zoom,n_rows*image_zoom),0,0
			ENDELSE
		ENDIF ELSE BEGIN
			IF ( ((1./image_zoom) EQ fix(1./image_zoom)) AND $
					((image_zoom*n_cols) EQ fix(image_zoom*n_cols)) AND $
					((image_zoom*n_rows) EQ fix(image_zoom*n_rows)) ) THEN BEGIN
				tv,rebin(byte_image,n_cols*image_zoom,n_rows*image_zoom,/sample),0,0
			ENDIF ELSE BEGIN
				tv,congrid(byte_image,n_cols*image_zoom,n_rows*image_zoom),0,0
			ENDELSE
		ENDELSE

		;  Plot dragbox (if it exists) over shifted image
		IF ((dragbox(1) NE 0) AND (dragbox(3) NE 0)) THEN BEGIN
			px = image_zoom*[xleft-x_shift(i_file), xright-x_shift(i_file), $
						xright-x_shift(i_file), xleft-x_shift(i_file), $
						xleft-x_shift(i_file)] ;X points
			py = image_zoom*[ybot-y_shift(i_file), ybot-y_shift(i_file), $
						ytop-y_shift(i_file), ytop-y_shift(i_file), $
						ybot-y_shift(i_file)] ;Y values
			plots,px,py,/device,color=dragbox_color_index
		ENDIF


	ENDIF
ENDIF
; Display clipped shifted image in clipped_image_win
; 	if there is a non-zero shift, i.e., alignment was kept
; 	if shift is discarded, i.e., all zero, don't display clipped_win_image
IF (zstack_subroutine EQ 'spectra') OR (zstack_subroutine EQ 'save') THEN BEGIN
	IF (data_shifted EQ 1) THEN BEGIN
		; Clip edges of image stack where no data is present due to alignment shifts
		byte_image = byte_image(clipbox[0]:clipbox[1],clipbox[2]:clipbox[3])
		wset,zstack_plot_par.clipped_image_win
		IF (image_zoom GE 1.0) THEN BEGIN
			IF ((image_zoom) EQ fix(image_zoom)) THEN BEGIN
				tv,rebin(byte_image,n_clipped_cols*image_zoom,n_clipped_rows*image_zoom,/sample),0,0
			ENDIF ELSE BEGIN
				tv,congrid(byte_image,n_clipped_cols*image_zoom,n_clipped_rows*image_zoom),0,0
			ENDELSE
		ENDIF ELSE BEGIN
			IF ( ((1./image_zoom) EQ fix(1./image_zoom)) AND $
					((image_zoom*n_clipped_cols) EQ fix(image_zoom*n_clipped_cols)) AND $
					((image_zoom*n_clipped_rows) EQ fix(image_zoom*n_clipped_rows)) ) THEN BEGIN
				tv,rebin(byte_image,n_clipped_cols*image_zoom,n_clipped_rows*image_zoom,/sample),0,0
			ENDIF ELSE BEGIN
				tv,congrid(byte_image,n_clipped_cols*image_zoom,n_clipped_rows*image_zoom),0,0
			ENDELSE
		ENDELSE
	ENDIF
ENDIF
; Display image description info
widget_control, zstack_plot_par.filename_display_list_label,$
		set_droplist_select=i_file
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_plot_plotshifts,i_file
;print,'zstack_plot_plotshifts'
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

; Plot alignment shifts with cursor bar
wset,zstack_plot_par.plot_win
CASE zstack_subroutine OF
	'align' : BEGIN
		; Determine range of x-values in plot
		IF (x_autoscale EQ 1)	 THEN BEGIN	; autoscale x-axis
			!x.range = [min(file_num),max(file_num)]
		ENDIF ELSE BEGIN
			!x.range = [plot_x_min, plot_x_max]
		ENDELSE
		dummy = where(file_num GE !x.range[0] AND file_num LE !x.range[1], count)
		; Determine range of y-values in plot
		IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
;			!y.range = [min([min(x_shift),min(y_shift)]),max([max(x_shift),max(y_shift)])]
			!y.range = [min([min(x_shift[dummy]),min(y_shift[dummy])]), $
					max([max(x_shift[dummy]),max(y_shift[dummy])])]
			!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
			!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
		ENDIF ELSE BEGIN
			!y.range = [plot_y_min, plot_y_max]
		ENDELSE
		; Plot alignment shift vs file #
		plot,file_num,x_shift,/nodata, $
				xtitle='File #', ytitle='Shift (pixels)',$
				xtick_get = xticks, ytick_get = yticks, $
				color=plot_axes_color_index, background=plot_bkgd_color_index
		oplot,file_num,x_shift,color=x_shift_color_index
		oplot,file_num,y_shift,color=y_shift_color_index
		; Plot X and Y labels on alignment shift plot
		xyouts,3+!x.range[0],(!y.range[0]+0.8*(!y.range[1] - !y.range[0])),/data, $
				color=x_shift_color_index,charthick=2,'X'
		xyouts,3+!x.range[0],(!y.range[0]+0.2*(!y.range[1] - !y.range[0])),/data, $
				color=y_shift_color_index,charthick=2,'Y'
		; Plot a indicator bar marking shift for displayed image
		oplot,file_num(i_file)+[0.,0.],!y.crange,color=shift_cursor_color_index
		; Set variables indicating limits of plot
		!x.range = [min(xticks),max(xticks)]
		!y.range = [min(yticks),max(yticks)]
		plot_x_min = !x.range[0]
		plot_x_max = !x.range[1]
		plot_y_min = !y.range[0]
		plot_y_max = !y.range[1]
		widget_control, zstack_plot_par.plot_y_min_label, $
				set_value = ' '+strtrim(string(!y.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_y_max_label, $
				set_value = ' '+strtrim(string(!y.range[1],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_min_label, $
				set_value = ' '+strtrim(string(!x.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_max_label, $
				set_value = ' '+strtrim(string(!x.range[1],format='(f10.3)'),2)
	END
	'tune' : BEGIN
		; Determine range of x-values in plot
		IF (x_autoscale EQ 1)	 THEN BEGIN	; autoscale x-axis
			!x.range = [min(file_num),max(file_num)]
		ENDIF ELSE BEGIN
			!x.range = [plot_x_min, plot_x_max]
		ENDELSE
		dummy = where(file_num GE !x.range[0] AND file_num LE !x.range[1], count)
		; Determine range of y-values in plot
		IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
;			!y.range = [min([min(x_shift),min(y_shift),min(temp_x_shift),min(temp_y_shift)]), $
;					max([max(x_shift),max(y_shift), max(temp_x_shift),max(temp_y_shift)])]
			!y.range = [min([min(     x_shift[dummy]),min(     y_shift[dummy]), $
						 min(temp_x_shift[dummy]),min(temp_y_shift[dummy])]), $
			            max([max(     x_shift[dummy]),max(     y_shift[dummy]), $
					 	 max(temp_x_shift[dummy]),max(temp_y_shift[dummy])])]
			!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
			!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
		ENDIF ELSE BEGIN
			!y.range = [plot_y_min, plot_y_max]
		ENDELSE
		; Plot alignment shift vs file #
		plot,file_num,x_shift,/nodata, $
				xtitle='File #', ytitle='Shift (pixels)',$
				xtick_get = xticks, ytick_get = yticks, $
				color=plot_axes_color_index, background=plot_bkgd_color_index
		oplot,file_num,x_shift,color=x_shift_color_index
		oplot,file_num,y_shift,color=y_shift_color_index
		oplot,file_num,temp_x_shift,color=x_shift_color_index,linestyle=1
		oplot,file_num,temp_y_shift,color=y_shift_color_index,linestyle=1
		; Plot X and Y labels on alignment shift plot
		xyouts,3+!x.range[0],(!y.range[0]+0.8*(!y.range[1] - !y.range[0])),/data, $
				color=x_shift_color_index,charthick=2,'X'
		xyouts,3+!x.range[0],(!y.range[0]+0.2*(!y.range[1] - !y.range[0])),/data, $
				color=y_shift_color_index,charthick=2,'Y'
		; Plot a indicator bar marking shift for displayed image
		oplot,file_num(i_file)+[0.,0.],!y.crange,color=shift_cursor_color_index
		; Set variables indicating limits of plot
		!x.range = [min(xticks),max(xticks)]
		!y.range = [min(yticks),max(yticks)]
		plot_x_min = !x.range[0]
		plot_x_max = !x.range[1]
		plot_y_min = !y.range[0]
		plot_y_max = !y.range[1]
		widget_control, zstack_plot_par.plot_y_min_label, $
				set_value = ' '+strtrim(string(!y.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_y_max_label, $
				set_value = ' '+strtrim(string(!y.range[1],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_min_label, $
				set_value = ' '+strtrim(string(!x.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_max_label, $
				set_value = ' '+strtrim(string(!x.range[1],format='(f10.3)'),2)
	END
	ELSE : BEGIN
		; don't plot alignment shifts
	END
ENDCASE
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_plot_plotspectrum, i_file
;print,'zstack_plot_plotspectrum'
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

; ----------------- force color scale to 0  ------ (APH 23-may-04) ----------
R = indgen(256) & G = R & B = R
TVLCT, R, B, G	; load grayscale color table


IF (strupcase(!d.name) NE 'Z') THEN BEGIN
	wset,zstack_plot_par.plot_win
ENDIF ELSE BEGIN
	print,'Z spectra'
ENDELSE
erase
IF (n_elements(spectrum) EQ 0) THEN BEGIN	; if no spectra exist, leave window erased
	return
ENDIF ELSE BEGIN
	dummy = where(spectrum NE 0,count)	; if no spectra exist, leave window erased
	IF (count EQ 0) THEN BEGIN
		return
	ENDIF
ENDELSE
; Determine range of x-values in plot
IF (x_autoscale EQ 1)	 THEN BEGIN	; autoscale x-axis
	!x.range = [min(ev_list),max(ev_list)]
ENDIF ELSE BEGIN
	!x.range = [plot_x_min, plot_x_max]
ENDELSE
x_index = where(ev_list GE !x.range[0] AND ev_list LE !x.range[1], count)
; Calculate spectra (Single Beam vs Percent Transmittance vs Absorbance)
; Add in offset
; Determine range of y-values in plot
IF (zstack_subroutine NE 'profile') THEN BEGIN
	; if zstack_display is called from any zstack routine except zstack_profile zzz
CASE spectrum_display OF	; Single Beam vs Percent Transmittance vs Absorbance
	1 : BEGIN		; plot single beam spectra
		dummy = where(spectrum[0,*] NE 0,count)
		IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
			dummy = where(spectrum[1:14,*] NE 0,count)
			IF (count GT 0) THEN BEGIN	; if both i0 and some i spectra exist
;				print,'SB :  SB, i0, i, profile'
				spectra = spectrum[0:n_roi,*]				; [i0,i1,i2,i3,...]
				IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
					FOR i=1,n_roi DO BEGIN
						spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
					ENDFOR
				ENDIF
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [min(spectra[0:n_roi,x_index]),max(spectra[0:n_roi,x_index])]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectra,/nodata, $
						xtitle='eV',ytitle='Single Beam Intensity', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				FOR i=0,n_roi DO BEGIN
					oplot,ev_list,spectra[i,*],color=spectra_color_index(i)
				ENDFOR
				IF (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					label = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
				ENDIF
			ENDIF ELSE BEGIN	; if only i0 spectrum exists
;				print,'SB :  SB, i0,  , profile'
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [min(spectrum[0,x_index]),max(spectrum[0,x_index])]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectrum[0,*],/nodata,xtitle='eV',ytitle='I0', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				oplot,ev_list,spectrum[0,*],color=spectra_color_index(0)
			ENDELSE
		ENDIF ELSE BEGIN	; if i0 spectrum does not exist
;			print,'SB : SB,   ,  , profile'
			spectra = spectrum[1:n_roi,*]				; [i1,i2,i3,...]
			IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
				FOR i=1,n_roi-1 DO BEGIN
					spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
				ENDFOR
			ENDIF
			IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
				!y.range = [min(spectra[0:n_roi-1,x_index]),max(spectra[0:n_roi-1,x_index])]
				!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
				!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
			ENDIF ELSE BEGIN
				!y.range = [plot_y_min, plot_y_max]
			ENDELSE
			plot,ev_list,spectra,/nodata, $
					xtitle='eV',ytitle='Single Beam Intensity', $
					xtick_get = xticks, ytick_get = yticks, $
					color=plot_axes_color_index, background=plot_bkgd_color_index
			FOR i=0,n_roi-1 DO BEGIN
				oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
			ENDFOR
			IF (n_roi GT 1) AND (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					label = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
			ENDIF
		ENDELSE
	END
	2 : BEGIN		; plot percent transmittance spectra
		dummy = where(spectrum[0,*] NE 0,count)
		IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
			dummy = where(spectrum[1:14,*] NE 0,count)
			IF (count GT 0) THEN BEGIN	; if both i0 and some i spectra exist
;				print,'TXM: TXM,   , i, profile'
				spectra = fltarr(n_roi,n_files)
				FOR i=1,n_roi DO BEGIN	; calculate percent transmittance spectra
					spectra[i-1,*] = 100.*spectrum[i,*]/spectrum[0,*]	; [i1/i0,i2/i0,i3/i0,...]
				ENDFOR
				IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
					FOR i=1,n_roi-1 DO BEGIN
						spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
					ENDFOR
				ENDIF
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [min(spectra[0:n_roi-1,x_index]),max(spectra[0:n_roi-1,x_index])]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectra,/nodata, $
						xtitle='eV',ytitle='Percent Tranmittance', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				FOR i=0,n_roi-1 DO BEGIN
					oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
				ENDFOR
				IF (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					label = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
				ENDIF
			ENDIF ELSE BEGIN	; if only i0 spectrum exists
;				print,'TXM: TXM,   ,  , profile'
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [min(spectrum[0,x_index]),max(spectrum[0,x_index])]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectrum[0,*],/nodata,xtitle='eV',ytitle='I0', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				oplot,ev_list,spectrum[0,*],color=spectra_color_index(0)
			ENDELSE
		ENDIF ELSE BEGIN	; if i0 spectrum does not exist
;			print,'TXM: SB,   ,  , profile'
			spectra = spectrum[1:n_roi,*]				; [i1,i2,i3,...]
			IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
				FOR i=1,n_roi-1 DO BEGIN
					spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
				ENDFOR
			ENDIF
			IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
				!y.range = [min(spectra[0:n_roi-1,x_index]),max(spectra[0:n_roi-1,x_index])]
				!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
				!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
			ENDIF ELSE BEGIN
				!y.range = [plot_y_min, plot_y_max]
			ENDELSE
			plot,ev_list,spectra,/nodata, $
					xtitle='eV',ytitle='Single Beam Intensity', $
					xtick_get = xticks, ytick_get = yticks, $
					color=plot_axes_color_index, background=plot_bkgd_color_index
			FOR i=0,n_roi-1 DO BEGIN
				oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
			ENDFOR
			IF (n_roi GT 1) AND (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					label = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
			ENDIF
		ENDELSE
	END
	3 : BEGIN		; plot absorbance spectra
		dummy = where(spectrum[0,*] NE 0,count)
		IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
			dummy = where(spectrum[1:14,*] NE 0,count)
			IF (count GT 0) THEN BEGIN	; if both i0 and some i spectra exist
;				print,'ABS: ABS,   , i, profile'
				spectra = fltarr(n_roi,n_files)
				FOR i=1,n_roi DO BEGIN	; calculate absorbance spectra - CHANGE to natural log 03-Aug-09
					spectra[i-1,*] = -alog(spectrum[i,*]/spectrum[0,*])	; [i1/i0,i2/i0,i3/i0,...]
				ENDFOR
				IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
					FOR i=1,n_roi-1 DO BEGIN
						spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
					ENDFOR
				ENDIF
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [min(spectra[0:n_roi-1,x_index]),max(spectra[0:n_roi-1,x_index])]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectra,/nodata, $
						xtitle='eV',ytitle='Absorbance', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				FOR i=0,n_roi-1 DO BEGIN
					oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
				ENDFOR
				IF (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					label = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
				ENDIF
			ENDIF ELSE BEGIN	; if only i0 spectrum exists
;				print,'ABS: ABS,   ,  , profile'
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [min(spectrum[0,x_index]),max(spectrum[0,x_index])]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectrum[0,*],/nodata,xtitle='eV',ytitle='I0', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				oplot,ev_list,spectrum[0,*],color=spectra_color_index(0)
			ENDELSE
		ENDIF ELSE BEGIN	; if i0 spectrum does not exist
;			print,'ABS: SB,   ,  , profile'
			spectra = spectrum[1:n_roi,*]				; [i1,i2,i3,...]
			IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
				FOR i=1,n_roi-1 DO BEGIN
					spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
				ENDFOR
			ENDIF
			IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
				!y.range = [min(spectra[0:n_roi-1,x_index]),max(spectra[0:n_roi-1,x_index])]
				!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
				!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
			ENDIF ELSE BEGIN
				!y.range = [plot_y_min, plot_y_max]
			ENDELSE
			plot,ev_list,spectra,/nodata, $
					xtitle='eV',ytitle='Single Beam Intensity', $
					xtick_get = xticks, ytick_get = yticks, $
					color=plot_axes_color_index, background=plot_bkgd_color_index
			FOR i=0,n_roi-1 DO BEGIN
				oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
			ENDFOR
			IF (n_roi GT 1) AND (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					label = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
			ENDIF
		ENDELSE
	END
	ELSE : BEGIN
		print,'spectrum_display not specified'
	END
ENDCASE
ENDIF ELSE BEGIN
	; if zstack_display is called from zstack_profile   zzz
; Find y limits of all potential profile spectra
;IF (profile_y_max EQ 0) THEN BEGIN
		CASE spectrum_display OF	; Single Beam vs Percent Transmittance vs Absorbance
			1 : BEGIN		; single beam spectra
;				profile_y_min = min(image_stack, max=profile_y_max,subscript)
;				p_spectrum_y_min = min(spectrum[0,*], max=p_spectrum_y_max)
				profile_y_min = min(image_stack(*,*,x_index), max=profile_y_max, subscript)
				p_spectrum_y_min = min(spectrum[0,x_index], max=p_spectrum_y_max)
				y_min = profile_y_min
				y_max = profile_y_max
			END
			2 : BEGIN		; percent transmittance spectra
				dummy = where(spectrum[0,*] NE 0,count)
				IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
					FOR i=0,n_elements(x_index)-1 DO BEGIN
						IF (finite(spectrum[0,i]) EQ 1) THEN BEGIN
							profile_y_min = min(100.*(image_stack(*,*,x_index(i))/spectrum[0,x_index(i)]), $
									max=profile_y_max, subscript)
							IF (i EQ 0) THEN BEGIN
								y_min = profile_y_min
								y_max = profile_y_max
							ENDIF
							IF (profile_y_min LT y_min) THEN y_min = profile_y_min
							IF (profile_y_max GT y_max) THEN y_max = profile_y_max
						ENDIF
					ENDFOR
				ENDIF ELSE BEGIN	; if i0 spectrum does not exist
;					profile_y_min = min(image_stack, max=profile_y_max)
					profile_y_min = min(image_stack(*,*,x_index), max=profile_y_max, subscript)
					y_min = profile_y_min
					y_max = profile_y_max
				ENDELSE
			END
			3 : BEGIN		; absorbance spectra
				dummy = where(spectrum[0,*] NE 0,count)
				IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
					FOR i=0,n_elements(x_index)-1 DO BEGIN
						IF (finite(spectrum[0,i]) EQ 1) THEN BEGIN
							profile_y_max = max(-alog(image_stack(*,*,x_index(i))/spectrum[0,x_index(i)]), $
									min=profile_y_min,  subscript)
							IF (i EQ 0) THEN BEGIN
								y_min = profile_y_min
								y_max = profile_y_max
							ENDIF
							IF (profile_y_min LT y_min) THEN y_min = profile_y_min
							IF (profile_y_max GT y_max) THEN y_max = profile_y_max
						ENDIF
					ENDFOR
				ENDIF ELSE BEGIN	; if i0 spectrum does not exist
;					profile_y_min = min(image_stack, max=profile_y_max)
					profile_y_min = min(image_stack(*,*,x_index), max=profile_y_max, subscript)
					y_min = profile_y_min
					y_max = profile_y_max
				ENDELSE
			END
			ELSE : BEGIN
				print,'spectrum_display not specified'
			END
		ENDCASE
;ENDIF
;print,'x_index, subscript : ',x_index, subscript
;print,'profile_y_min, profile_y_max : ', profile_y_min, profile_y_max
; profile_y_min and profile_y_max are the minimum and maximum values of all potential profile spectra
; i.e., the range in intensity of all profile spectra
; Calculate spectra (Single Beam vs Percent Transmittance vs Absorbance)
; Add in offset
CASE spectrum_display OF	; Single Beam vs Percent Transmittance vs Absorbance
	1 : BEGIN		; plot single beam spectra
		dummy = where(spectrum[0,*] NE 0,count)
		IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
			dummy = where(spectrum[1:14,*] NE 0,count)
			IF (count GT 0) THEN BEGIN	; if both i0 and some i spectra exist
;				print,'SB :  SB, i0, i, profile'
				spectra = spectrum[0:n_roi,*]				; [i0,i1,i2,i3,...]
				IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
					FOR i=0,n_roi DO BEGIN
						spectra[i,*] = (spectrum_offset * (i+1)) + spectra[i,*]
					ENDFOR
					dummy = spectra[0:n_roi,*]
					y_min = min(dummy[where(finite(dummy))],max=y_max)
					y_min = (1.05 * y_max) < (0.95 * y_min)
					y_max = (1.05 * y_max) > (0.95 * y_min)
					IF (y_min LT profile_y_min) THEN  profile_y_min = y_min
					IF (y_max GT profile_y_max) THEN  profile_y_max = y_max
				ENDIF
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [profile_y_min,profile_y_max]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectra,/nodata, $
						xtitle='eV',ytitle='Single Beam Intensity', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				FOR i=0,n_roi DO BEGIN
					oplot,ev_list,spectra[i,*],color=spectra_color_index(i)
				ENDFOR
				oplot,ev_list,profile_spectrum,color=plot_axes_color_index
				IF (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					label = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
				ENDIF
			ENDIF ELSE BEGIN	; if only i0 spectrum exists
;				print,'SB :  SB, i0,  , profile'
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [profile_y_min,profile_y_max]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectrum[0,*],/nodata,xtitle='eV',ytitle='Single Beam Intensity', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				oplot,ev_list,spectrum[0,*]+spectrum_offset,color=spectra_color_index(0)
				oplot,ev_list,profile_spectrum,color=plot_axes_color_index
			ENDELSE
		ENDIF ELSE BEGIN	; if i0 spectrum does not exist
			dummy = where(spectrum[1:14,*] NE 0,count)
			IF (count EQ 0) THEN BEGIN	; if no i spectrum exist
;				print,'SB : SB,   ,  , profile'
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [profile_y_min,profile_y_max]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,profile_spectrum,/nodata,xtitle='eV',ytitle='Single Beam Intensity', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				oplot,ev_list,profile_spectrum,color=plot_axes_color_index
			ENDIF ELSE BEGIN; if some i spectra exist
;				print,'SB : SB,   , i, profile'
				spectra = spectrum[1:n_roi,*]				; [i1,i2,i3,...]
				IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
					FOR i=1,n_roi-1 DO BEGIN
						spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
					ENDFOR
					dummy = spectra[1:n_roi-1,*]
					y_min = min(dummy[where(finite(dummy))],max=y_max)
					y_min = (1.05 * y_max) < (0.95 * y_min)
					y_max = (1.05 * y_max) > (0.95 * y_min)
					IF (y_min LT profile_y_min) THEN  profile_y_min = y_min
					IF (y_max GT profile_y_max) THEN  profile_y_max = y_max
				ENDIF
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [profile_y_min,profile_y_max]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectra,/nodata, $
						xtitle='eV',ytitle='Single Beam Intensity', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				FOR i=0,n_roi-1 DO BEGIN
					oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
				ENDFOR
				oplot,ev_list,profile_spectrum,color=plot_axes_color_index
				IF (n_roi GT 1) AND (spectrum_offset NE 0) THEN BEGIN	; plot offset label
						label = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
						xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
								alignment=0.0,label
				ENDIF
			ENDELSE
		ENDELSE
	END
	2 : BEGIN		; plot percent transmittance spectra
		dummy = where(spectrum[0,*] NE 0,count)
		IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
			temp_profile_spectrum = 100.*(profile_spectrum / spectrum[0,*])
			dummy = where(spectrum[1:14,*] NE 0,count)
			IF (count GT 0) THEN BEGIN	; if both i0 and some i spectra exist
;				print,'TXM: TXM,   , i, profile'
				spectra = fltarr(n_roi,n_files)
				FOR i=1,n_roi DO BEGIN	; calculate percent transmittance spectra
					spectra[i-1,*] = 100.*spectrum[i,*]/spectrum[0,*]	; [i1/i0,i2/i0,i3/i0,...]
				ENDFOR
				IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
					FOR i=0,n_roi-1 DO BEGIN
						spectra[i,*] = (spectrum_offset * (i+1)) + spectra[i,*]
					ENDFOR
					dummy = spectra[0:n_roi-1,*]
					y_min = min(dummy[where(finite(dummy))],max=y_max)
					y_min = (1.05 * y_max) < (0.95 * y_min)
					y_max = (1.05 * y_max) > (0.95 * y_min)
					IF (y_min LT profile_y_min) THEN  profile_y_min = y_min
					IF (y_max GT profile_y_max) THEN  profile_y_max = y_max
				ENDIF
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [profile_y_min,profile_y_max]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectra,/nodata, $
						xtitle='eV',ytitle='Percent Tranmittance', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				FOR i=0,n_roi-1 DO BEGIN
					oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
				ENDFOR
				oplot,ev_list,temp_profile_spectrum,color=plot_axes_color_index
				IF (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					label = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
				ENDIF
			ENDIF ELSE BEGIN	; if only i0 spectrum exists
;				print,'TXM: TXM,   ,  , profile'
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [profile_y_min,profile_y_max]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectrum[0,*],/nodata,xtitle='eV',ytitle='Percent Transmittance', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				oplot,ev_list,spectrum[0,*]+spectrum_offset,color=spectra_color_index(0)
				oplot,ev_list,temp_profile_spectrum,color=plot_axes_color_index
			ENDELSE
		ENDIF ELSE BEGIN	; if i0 spectrum does not exist
			dummy = where(spectrum[1:14,*] NE 0,count)
			IF (count EQ 0) THEN BEGIN	; if no i spectrum exist
;				print,'TXM: SB,   ,  , profile'
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
					!y.range = [profile_y_min,profile_y_max]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,profile_spectrum,/nodata,xtitle='eV',ytitle='Single Beam Intensity', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				oplot,ev_list,profile_spectrum,color=plot_axes_color_index
			ENDIF ELSE BEGIN; if some i spectra exist
;				print,'TXM: SB,   , i, profile'
				spectra = spectrum[1:n_roi,*]				; [i1,i2,i3,...]
				IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
					FOR i=1,n_roi-1 DO BEGIN
						spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
					ENDFOR
					dummy = spectra[0:n_roi-1,*]
					y_min = min(dummy[where(finite(dummy))],max=y_max)
					y_min = (1.05 * y_max) < (0.95 * y_min)
					y_max = (1.05 * y_max) > (0.95 * y_min)
					IF (y_min LT profile_y_min) THEN  profile_y_min = y_min
					IF (y_max GT profile_y_max) THEN  profile_y_max = y_max
				ENDIF
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
dummy = where(ev_list GE !x.range[0] AND ev_list LE !x.range[1], count)
					!y.range = [profile_y_min,profile_y_max]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectra,/nodata, $
						xtitle='eV',ytitle='Single Beam Intensity', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				FOR i=0,n_roi-1 DO BEGIN
					oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
				ENDFOR
				oplot,ev_list,profile_spectrum,color=plot_axes_color_index
				IF (n_roi GT 1) AND (spectrum_offset NE 0) THEN BEGIN	; plot offset label
						label = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
						xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
								alignment=0.0,label
				ENDIF
			ENDELSE
		ENDELSE
	END
	3 : BEGIN		; plot absorbance spectra
		dummy = where(spectrum[0,*] NE 0,count)
		IF (count NE 0) THEN BEGIN	; if i0 spectrum exists
			temp_profile_spectrum = -alog(profile_spectrum / spectrum[0,*])
			dummy = where(spectrum[1:14,*] NE 0,count)
			IF (count GT 0) THEN BEGIN	; if both i0 and some i spectra exist
;				print,'ABS: ABS,   , i, profile'
				spectra = fltarr(n_roi,n_files)
				FOR i=1,n_roi DO BEGIN	; calculate absorbance spectra
					spectra[i-1,*] = -alog(spectrum[i,*]/spectrum[0,*])	; [i1/i0,i2/i0,i3/i0,...]
				ENDFOR
				IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
					FOR i=0,n_roi-1 DO BEGIN
						spectra[i,*] = (spectrum_offset * (i+1)) + spectra[i,*]
					ENDFOR
					dummy = spectra[0:n_roi-1,*]
					y_min = min(dummy[where(finite(dummy))],max=y_max)
					y_min = (1.05 * y_max) < (0.95 * y_min)
					y_max = (1.05 * y_max) > (0.95 * y_min)
					IF (y_min LT profile_y_min) THEN  profile_y_min = y_min
					IF (y_max GT profile_y_max) THEN  profile_y_max = y_max
				ENDIF
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
dummy = where(ev_list GE !x.range[0] AND ev_list LE !x.range[1], count)
					!y.range = [profile_y_min,profile_y_max]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectra,/nodata, $
						xtitle='eV',ytitle='Absorbance', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				FOR i=0,n_roi-1 DO BEGIN
					oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
				ENDFOR
				oplot,ev_list,temp_profile_spectrum,color=plot_axes_color_index
				IF (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					label = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
				ENDIF
			ENDIF ELSE BEGIN	; if only i0 spectrum exists
;				print,'ABS: ABS,   ,  , profile'
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
dummy = where(ev_list GE !x.range[0] AND ev_list LE !x.range[1], count)
					!y.range = [profile_y_min,profile_y_max]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectrum[0,*],/nodata,xtitle='eV',ytitle='Absorbance', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				oplot,ev_list,spectrum[0,*]+spectrum_offset,color=spectra_color_index(0)
				oplot,ev_list,temp_profile_spectrum,color=plot_axes_color_index
			ENDELSE
		ENDIF ELSE BEGIN	; if i0 spectrum does not exist
			dummy = where(spectrum[1:14,*] NE 0,count)
			IF (count EQ 0) THEN BEGIN	; if no i spectrum exist
;				print,'ABS: SB,   ,  , profile'
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
dummy = where(ev_list GE !x.range[0] AND ev_list LE !x.range[1], count)
					!y.range = [profile_y_min,profile_y_max]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,profile_spectrum,/nodata,xtitle='eV',ytitle='Single Beam Intensity', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				oplot,ev_list,profile_spectrum,color=plot_axes_color_index
			ENDIF ELSE BEGIN; if some i spectra exist
;				print,'ABS: SB,   , i, profile'
				spectra = spectrum[1:n_roi,*]				; [i1,i2,i3,...]
				IF (spectrum_offset NE 0) THEN BEGIN	; add in offset to spectra
					FOR i=1,n_roi-1 DO BEGIN
						spectra[i,*] = (spectrum_offset * (i)) + spectra[i,*]
					ENDFOR
					dummy = spectra[0:n_roi-1,*]
					y_min = min(dummy[where(finite(dummy))],max=y_max)
					y_min = (1.05 * y_max) < (0.95 * y_min)
					y_max = (1.05 * y_max) > (0.95 * y_min)
					IF (y_min LT profile_y_min) THEN  profile_y_min = y_min
					IF (y_max GT profile_y_max) THEN  profile_y_max = y_max
				ENDIF
				IF (y_autoscale EQ 1) THEN BEGIN	; autoscale y-axis
dummy = where(ev_list GE !x.range[0] AND ev_list LE !x.range[1], count)
					!y.range = [profile_y_min,profile_y_max]
					!y.range[0] = (1.05 * !y.range[0]) < (0.95 * !y.range[0])
					!y.range[1] = (1.05 * !y.range[1]) > (0.95 * !y.range[1])
				ENDIF ELSE BEGIN
					!y.range = [plot_y_min, plot_y_max]
				ENDELSE
				plot,ev_list,spectra,/nodata, $
						xtitle='eV',ytitle='Single Beam Intensity', $
						xtick_get = xticks, ytick_get = yticks, $
						color=plot_axes_color_index, background=plot_bkgd_color_index
				FOR i=0,n_roi-1 DO BEGIN
					oplot,ev_list,spectra[i,*],color=spectra_color_index(i+1)
				ENDFOR
				oplot,ev_list,profile_spectrum,color=plot_axes_color_index
				IF (n_roi GT 1) AND (spectrum_offset NE 0) THEN BEGIN	; plot offset label
					label = 'Offset = '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
					xyouts,5,5, /device, color=plot_axes_color_index, charthick=1, $
							alignment=0.0,label
				ENDIF
			ENDELSE
		ENDELSE
	END
	ELSE : BEGIN
		print,'spectrum_display not specified'
	END
ENDCASE
ENDELSE
;plot_y_min = !y.range[0]
;plot_y_max = !y.range[1]
!x.range = [min(xticks),max(xticks)]
!y.range = [min(yticks),max(yticks)]
IF (ev_list(i_file) GT !x.range[0]) AND (ev_list(i_file) LT !x.range[1]) THEN BEGIN
	plots,ev_list(i_file)+[0.,0.],!y.crange,color=spectra_cursor_color_index
ENDIF
;print,'x y autoscale :',x_autoscale,y_autoscale
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_plot_event,event
;print,'zstack_plot_event'
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

CASE event.id OF
	zstack_plot_par.image_zoom_label : BEGIN
		init_value = image_zoom
		temp_string = ''
		widget_control, zstack_plot_par.image_zoom_label, $
				get_value = temp_string
		on_ioerror, image_zoom_label_oops
;		IF (strlen(temp_string(0)) EQ 0) THEN goto, image_zoom_label_oops
;		IF (strlen(temp_string) EQ 0) THEN goto, image_zoom_label_oops
		reads,temp_string(0),image_zoom
		image_zoom_label_oops:
		widget_control, zstack_plot_par.image_zoom_label, $
				set_value = ' '+strtrim(string(image_zoom,format='(f10.2)'),2)
 		IF (image_zoom NE init_value) THEN BEGIN	; Get nearest integral value for image_zoom
			IF (image_zoom GE 1) THEN BEGIN
				IF (image_zoom GT 5) THEN image_zoom = 5		; maximum zoom is 5
				widget_control, zstack_plot_par.image_zoom_label, $
						set_value = ' '+strtrim(string(image_zoom,format='(f10.2)'),2)
			ENDIF
			IF (image_zoom LT 1) THEN BEGIN
				IF (image_zoom LT 0.2) THEN image_zoom = 0.2		; minimum zoom is 0.2
				widget_control, zstack_plot_par.image_zoom_label, $
						set_value = ' '+strtrim(string(image_zoom,format='(f10.2)'),2)
			ENDIF
		ENDIF
		IF (image_zoom NE init_value) THEN BEGIN
			temp_new_display[0] = image_zoom
			widget_control, zstack_plot_par.main_base, /destroy
			zstack_plot, zstack_subroutine,/new_zoom
		ENDIF
		dummy = where(temp_new_display NE temp_old_display, count)
		IF (count EQ 0) THEN BEGIN
			widget_control, zstack_plot_par.update_label, sensitive = 0
		ENDIF ELSE BEGIN
			widget_control, zstack_plot_par.update_label, sensitive = 1
		ENDELSE
	END
	zstack_plot_par.movie_delay_label : BEGIN
		init_value = movie_delay
		temp_string = ''
		widget_control, zstack_plot_par.movie_delay_label, $
				get_value = temp_string
		on_ioerror, movie_delay_label_oops
		IF (strlen(temp_string(0)) EQ 0) THEN GOTO, movie_delay_label_oops
		reads,temp_string(0),movie_delay
		movie_delay_label_oops:
		IF (movie_delay NE init_value) THEN BEGIN
			temp_new_display[4] = movie_delay
			widget_control, zstack_plot_par.movie_delay_label, $
					set_value = ' '+strtrim(string(movie_delay,format='(f10.3)'),2)
			zstack_plot_imgdisp,displayed_file_index
		ENDIF
		dummy = where(temp_new_display NE temp_old_display, count)
		IF (count EQ 0) THEN BEGIN
			widget_control, zstack_plot_par.update_label, sensitive = 0
		ENDIF ELSE BEGIN
			widget_control, zstack_plot_par.update_label, sensitive = 1
		ENDELSE
	END
	zstack_plot_par.spectrum_offset_label : BEGIN
		init_value = spectrum_offset
		temp_string = ''
		widget_control, zstack_plot_par.spectrum_offset_label,$
				get_value = temp_string
		on_ioerror, spectrum_offset_label_oops
		reads,temp_string(0),spectrum_offset
		spectrum_offset_label_oops:
		IF (spectrum_offset NE init_value) THEN BEGIN
			temp_new_display[7] = spectrum_offset
			widget_control, zstack_plot_par.spectrum_offset_label, $
					set_value = ' '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
			zstack_plot_plotspectrum, displayed_file_index
		ENDIF
		dummy = where(temp_new_display NE temp_old_display, count)
		IF (count EQ 0) THEN BEGIN
			widget_control, zstack_plot_par.update_label, sensitive = 0
		ENDIF ELSE BEGIN
			widget_control, zstack_plot_par.update_label, sensitive = 1
		ENDELSE
		profile_y_min = 0
		profile_y_max = 0
	END
	zstack_plot_par.xloadct_label : BEGIN
		zstack_plot_desensitive
		temp_new_display[8] = 1
		xloadct, bottom=bottom_color_index, ncolors=top_color_index - bottom_color_index,/modal
		IF (image_invert EQ 1) THEN BEGIN
			image_invert = 0
			widget_control, zstack_plot_par.invertct_label, set_value = 'Invert Color Table'
		ENDIF

		wset,zstack_plot_par.image_win
		zstack_plot_imgdisp, displayed_file_index
		zstack_analyze_colorbar, zstack_plot_par.colorbar_win
;		widget_control, zstack_plot_par.update_label, sensitive = 1
;		widget_control, zstack_plot_par.close_label, sensitive = 0
		zstack_plot_sensitive
	END
	zstack_plot_par.invertct_label : BEGIN
		init_value = image_invert
		IF (image_invert EQ 0) THEN BEGIN
			image_invert = 1
			widget_control, zstack_plot_par.invertct_label, set_value = 'Re-invert Color Table'
		ENDIF ELSE BEGIN
			image_invert = 0
			widget_control, zstack_plot_par.invertct_label, set_value = 'Invert Color Table'
		ENDELSE
		IF (image_invert NE init_value) THEN BEGIN
			zstack_plot_desensitive
			temp_new_display[9] = image_invert
			; reverse grey-scale portion of color table and redisplay image
			tvlct, r, g, b, bottom_color_index, /get
			tvlct, reverse(r), reverse(g), reverse(b), bottom_color_index
			zstack_plot_imgdisp,displayed_file_index
			zstack_analyze_colorbar, zstack_profile_par.colorbar_win
			zstack_plot_sensitive
		ENDIF
;		dummy = where(temp_new_display NE temp_old_display, count)
;		IF (count EQ 0) THEN BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 0
;		ENDIF ELSE BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 1
;		ENDELSE
	END

	zstack_plot_par.loadplotct_label : BEGIN
		temp_new_display[10] = 1
		zstack_pickcolor,group_leader_ID=zstack_plot_par.main_base
;		wset,zstack_plot_par.image_win
		zstack_plot_imgdisp, displayed_file_index
		zstack_analyze_colorbar, zstack_plot_par.colorbar_win
		CASE zstack_subroutine OF
			'align' : BEGIN
				zstack_plot_plotshifts, displayed_file_index
			END
			'tune' : BEGIN
				zstack_plot_plotshifts, displayed_file_index
			END
			'spectra' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			'profile' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			'save' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			ELSE : BEGIN
			print,'Unrecognized zstack_subroutine : zstack_plot_par.x_autoscale_label'
			END
		ENDCASE
		zstack_plot_sensitive
	END

	zstack_plot_par.spectra_singlebeam_label : BEGIN
		zstack_plot_desensitive
		spectrum_display = 1
		temp_new_display[13] = spectrum_display
		y_autoscale = 1
		temp_new_display[18] = y_autoscale
		profile_y_min = 0
		profile_y_max = 0
		widget_control, zstack_plot_par.y_autoscale_label, set_button = 1
		CASE zstack_subroutine OF
			'align' : BEGIN
				zstack_plot_plotshifts, displayed_file_index
			END
			'tune' : BEGIN
				zstack_plot_plotshifts, displayed_file_index
			END
			'spectra' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			'profile' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			'save' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			ELSE : BEGIN
			print,'Unrecognized zstack_subroutine : zstack_plot_par.x_autoscale_label'
			END
		ENDCASE
;		dummy = where(temp_new_display NE temp_old_display, count)
;		IF (count EQ 0) THEN BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 0
;		ENDIF ELSE BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 1
;		ENDELSE
		spectrum_offset = 0.0
		widget_control, zstack_plot_par.spectrum_offset_label, $
				set_value = ' '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
		zstack_plot_plotspectrum, displayed_file_index
		widget_control, zstack_plot_par.plot_y_min_label, $
				set_value = ' '+strtrim(string(!y.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_y_max_label, $
				set_value = ' '+strtrim(string(!y.range[1],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_min_label, $
				set_value = ' '+strtrim(string(!x.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_max_label, $
				set_value = ' '+strtrim(string(!x.range[1],format='(f10.3)'),2)
		zstack_plot_sensitive
	END
	zstack_plot_par.spectra_tranmittance_label : BEGIN
		zstack_plot_desensitive
		spectrum_display = 2
		temp_new_display[13] = spectrum_display
		y_autoscale = 1
		temp_new_display[18] = y_autoscale
		profile_y_min = 0
		profile_y_max = 0
		widget_control, zstack_plot_par.y_autoscale_label, set_button = 1
		CASE zstack_subroutine OF
			'align' : BEGIN
				zstack_plot_plotshifts, displayed_file_index
			END
			'tune' : BEGIN
				zstack_plot_plotshifts, displayed_file_index
			END
			'spectra' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			'profile' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			'save' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			ELSE : BEGIN
			print,'Unrecognized zstack_subroutine : zstack_plot_par.x_autoscale_label'
			END
		ENDCASE
;		dummy = where(temp_new_display NE temp_old_display, count)
;		IF (count EQ 0) THEN BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 0
;		ENDIF ELSE BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 1
;		ENDELSE
		spectrum_offset = 0.0
		widget_control, zstack_plot_par.spectrum_offset_label, $
				set_value = ' '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
		zstack_plot_plotspectrum, displayed_file_index
		widget_control, zstack_plot_par.plot_y_min_label, $
				set_value = ' '+strtrim(string(!y.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_y_max_label, $
				set_value = ' '+strtrim(string(!y.range[1],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_min_label, $
				set_value = ' '+strtrim(string(!x.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_max_label, $
				set_value = ' '+strtrim(string(!x.range[1],format='(f10.3)'),2)
		zstack_plot_sensitive
	END
	zstack_plot_par.spectra_absorbance_label : BEGIN
		zstack_plot_desensitive
		spectrum_display = 3
		temp_new_display[13] = spectrum_display
		y_autoscale = 1
		temp_new_display[18] = y_autoscale
		profile_y_min = 0
		profile_y_max = 0
		widget_control, zstack_plot_par.y_autoscale_label, set_button = 1
		CASE zstack_subroutine OF
			'align' : BEGIN
				zstack_plot_plotshifts, displayed_file_index
			END
			'tune' : BEGIN
				zstack_plot_plotshifts, displayed_file_index
			END
			'spectra' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			'profile' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			'save' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			ELSE : BEGIN
			print,'Unrecognized zstack_subroutine : zstack_plot_par.x_autoscale_label'
			END
		ENDCASE
;		dummy = where(temp_new_display NE temp_old_display, count)
;		IF (count EQ 0) THEN BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 0
;		ENDIF ELSE BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 1
;		ENDELSE
		spectrum_offset = 0.0
		widget_control, zstack_plot_par.spectrum_offset_label, $
				set_value = ' '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
		zstack_plot_plotspectrum, displayed_file_index
		widget_control, zstack_plot_par.plot_y_min_label, $
				set_value = ' '+strtrim(string(!y.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_y_max_label, $
				set_value = ' '+strtrim(string(!y.range[1],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_min_label, $
				set_value = ' '+strtrim(string(!x.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_max_label, $
				set_value = ' '+strtrim(string(!x.range[1],format='(f10.3)'),2)
		zstack_plot_sensitive
	END
	zstack_plot_par.x_autoscale_label : BEGIN
		zstack_plot_desensitive
		IF (x_autoscale EQ 0) THEN BEGIN
			x_autoscale = 1
		ENDIF ELSE BEGIN
			x_autoscale = 0
		ENDELSE
		temp_new_display[15] = x_autoscale
		CASE zstack_subroutine OF
			'align' : BEGIN
				zstack_plot_plotshifts, displayed_file_index
			END
			'tune' : BEGIN
				zstack_plot_plotshifts, displayed_file_index
			END
			'spectra' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			'profile' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			'save' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			ELSE : BEGIN
			print,'Unrecognized zstack_subroutine : zstack_plot_par.x_autoscale_label'
			END
		ENDCASE
;		zstack_plot_plotspectrum, displayed_file_index
;		dummy = where(temp_new_display NE temp_old_display, count)
;		IF (count EQ 0) THEN BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 0
;		ENDIF ELSE BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 1
;		ENDELSE
		widget_control, zstack_plot_par.plot_y_min_label, $
				set_value = ' '+strtrim(string(!y.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_y_max_label, $
				set_value = ' '+strtrim(string(!y.range[1],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_min_label, $
				set_value = ' '+strtrim(string(!x.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_max_label, $
				set_value = ' '+strtrim(string(!x.range[1],format='(f10.3)'),2)
		zstack_plot_sensitive
	END
	zstack_plot_par.plot_x_min_label : BEGIN
		on_ioerror, plot_x_min_label_oops
		init_value = [plot_x_min,plot_x_max]
		temp_string = ''
		widget_control, zstack_plot_par.plot_x_min_label,$
				get_value = temp_string
		reads,temp_string(0),plot_x_min
		plot_x_min_label_oops:
		IF (plot_x_min NE init_value[0]) THEN BEGIN
			zstack_plot_desensitive
			x_autoscale = 0
			temp_new_display[15] = x_autoscale
			widget_control, zstack_plot_par.x_autoscale_label, set_button = 0
			CASE zstack_subroutine OF
				'align' : BEGIN
					zstack_plot_plotshifts, displayed_file_index
				END
				'tune' : BEGIN
					zstack_plot_plotshifts, displayed_file_index
				END
				'spectra' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				'profile' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				'save' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				ELSE : BEGIN
					print,'Unrecognized zstack_subroutine : zstack_plot_par.plot_x_min_label'
				END
			ENDCASE
;			zstack_plot_plotspectrum, displayed_file_index
			IF (!x.range[0] NE init_value[0]) THEN BEGIN
				plot_x_min = !x.range[0]
				temp_new_display[16] = !x.range[0]
				widget_control, zstack_plot_par.plot_x_min_label, $
						set_value = ' '+strtrim(string(!x.range[0],format='(f10.3)'),2)
			ENDIF
			IF (!x.range[1] NE init_value[1]) THEN BEGIN
				plot_x_max = !x.range[1]
				temp_new_display[17] = !x.range[1]
				widget_control, zstack_plot_par.plot_x_max_label, $
						set_value = ' '+strtrim(string(!x.range[1],format='(f10.3)'),2)
			ENDIF
			zstack_plot_sensitive
		ENDIF
;		dummy = where(temp_new_display NE temp_old_display, count)
;		IF (count EQ 0) THEN BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 0
;		ENDIF ELSE BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 1
;		ENDELSE
		widget_control, zstack_plot_par.plot_y_min_label, $
				set_value = ' '+strtrim(string(!y.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_y_max_label, $
				set_value = ' '+strtrim(string(!y.range[1],format='(f10.3)'),2)
	END
	zstack_plot_par.plot_x_max_label : BEGIN
		on_ioerror, plot_x_max_label_oops
		init_value = [plot_x_min,plot_x_max]
		temp_string = ''
		widget_control, zstack_plot_par.plot_x_max_label,$
				get_value = temp_string
		reads,temp_string(0),plot_x_max
		plot_x_max_label_oops:
		IF (plot_x_max NE init_value[1]) THEN BEGIN
			zstack_plot_desensitive
			x_autoscale = 0
			temp_new_display[15] = x_autoscale
			widget_control, zstack_plot_par.x_autoscale_label, set_button = 0
			CASE zstack_subroutine OF
				'align' : BEGIN
					zstack_plot_plotshifts, displayed_file_index
				END
				'tune' : BEGIN
					zstack_plot_plotshifts, displayed_file_index
				END
				'spectra' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				'profile' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				'save' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				ELSE : BEGIN
					print,'Unrecognized zstack_subroutine : zstack_plot_par.plot_x_max_label'
				END
			ENDCASE
;			zstack_plot_plotspectrum, displayed_file_index
			IF (!x.range[0] NE init_value[0]) THEN BEGIN
				plot_x_min = !x.range[0]
				temp_new_display[16] = !x.range[0]
				widget_control, zstack_plot_par.plot_x_min_label, $
						set_value = ' '+strtrim(string(!x.range[0],format='(f10.3)'),2)
			ENDIF
			IF (!x.range[1] NE init_value[1]) THEN BEGIN
				plot_x_max = !x.range[1]
				temp_new_display[17] = !x.range[1]
				widget_control, zstack_plot_par.plot_x_max_label, $
						set_value = ' '+strtrim(string(!x.range[1],format='(f10.3)'),2)
			ENDIF
			zstack_plot_sensitive
		ENDIF
;		dummy = where(temp_new_display NE temp_old_display, count)
;		IF (count EQ 0) THEN BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 0
;		ENDIF ELSE BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 1
;		ENDELSE
		widget_control, zstack_plot_par.plot_y_min_label, $
				set_value = ' '+strtrim(string(!y.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_y_max_label, $
				set_value = ' '+strtrim(string(!y.range[1],format='(f10.3)'),2)
	END
	zstack_plot_par.y_autoscale_label : BEGIN
		zstack_plot_desensitive
		IF (y_autoscale EQ 0) THEN BEGIN
			y_autoscale = 1
		ENDIF ELSE BEGIN
			y_autoscale = 0
		ENDELSE
		temp_new_display[18] = y_autoscale
;		dummy = where(temp_new_display NE temp_old_display, count)
;		IF (count EQ 0) THEN BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 0
;		ENDIF ELSE BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 1
;		ENDELSE
		zstack_plot_desensitive
		CASE zstack_subroutine OF
			'align' : BEGIN
				zstack_plot_plotshifts, displayed_file_index
			END
			'tune' : BEGIN
				zstack_plot_plotshifts, displayed_file_index
			END
			'spectra' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			'profile' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			'save' : BEGIN
				zstack_plot_plotspectrum, displayed_file_index
			END
			ELSE : BEGIN
				print,'Unrecognized zstack_subroutine : zstack_plot_par.y_autoscale_label'
			END
		ENDCASE
;		zstack_plot_plotspectrum, displayed_file_index
		widget_control, zstack_plot_par.plot_y_min_label, $
				set_value = ' '+strtrim(string(!y.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_y_max_label, $
				set_value = ' '+strtrim(string(!y.range[1],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_min_label, $
				set_value = ' '+strtrim(string(!x.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_max_label, $
				set_value = ' '+strtrim(string(!x.range[1],format='(f10.3)'),2)
		zstack_plot_sensitive
	END
	zstack_plot_par.plot_y_min_label : BEGIN
		on_ioerror, plot_y_min_label_oops
		init_value = [plot_y_min,plot_y_max]
		temp_string = ''
		widget_control, zstack_plot_par.plot_y_min_label,$
				get_value = temp_string
		reads,temp_string(0),plot_y_min
		plot_y_min_label_oops:
		IF (plot_y_min NE init_value[0]) THEN BEGIN
			zstack_plot_desensitive
			y_autoscale = 0
			temp_new_display[18] = y_autoscale
			widget_control, zstack_plot_par.y_autoscale_label, set_button = 0
			CASE zstack_subroutine OF
				'align' : BEGIN
					zstack_plot_plotshifts, displayed_file_index
				END
				'tune' : BEGIN
					zstack_plot_plotshifts, displayed_file_index
				END
				'spectra' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				'profile' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				'save' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				ELSE : BEGIN
					print,'Unrecognized zstack_subroutine : zstack_plot_par.plot_y_min_label'
				END
			ENDCASE
;			zstack_plot_plotspectrum, displayed_file_index
			IF (!y.range[0] NE init_value[0]) THEN BEGIN
				plot_y_min = !y.range[0]
				temp_new_display[19] = !y.range[0]
				widget_control, zstack_plot_par.plot_y_min_label, $
						set_value = ' '+strtrim(string(!y.range[0],format='(f10.3)'),2)
			ENDIF
			IF (!y.range[1] NE init_value[1]) THEN BEGIN
				plot_y_max = !y.range[1]
				temp_new_display[20] = !y.range[1]
				widget_control, zstack_plot_par.plot_y_max_label, $
						set_value = ' '+strtrim(string(!y.range[1],format='(f10.3)'),2)
			ENDIF
			zstack_plot_sensitive
		ENDIF
;		dummy = where(temp_new_display NE temp_old_display, count)
;		IF (count EQ 0) THEN BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 0
;		ENDIF ELSE BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 1
;		ENDELSE
	END
	zstack_plot_par.plot_y_max_label : BEGIN
		on_ioerror, plot_y_max_label_oops
		init_value = [plot_y_min,plot_y_max]
		temp_string = ''
		widget_control, zstack_plot_par.plot_y_max_label,$
				get_value = temp_string
		reads,temp_string(0),plot_y_max
		plot_y_max_label_oops:
		IF (plot_y_max NE init_value[1]) THEN BEGIN
			zstack_plot_sensitive
			y_autoscale = 0
			temp_new_display[18] = y_autoscale
			widget_control, zstack_plot_par.y_autoscale_label, set_button = 0
			CASE zstack_subroutine OF
				'align' : BEGIN
					zstack_plot_plotshifts, displayed_file_index
				END
				'tune' : BEGIN
					zstack_plot_plotshifts, displayed_file_index
				END
				'spectra' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				'profile' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				'save' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				ELSE : BEGIN
					print,'Unrecognized zstack_subroutine : zstack_plot_par.plot_y_max_label'
				END
			ENDCASE
;			zstack_plot_plotspectrum, displayed_file_index
			IF (!y.range[0] NE init_value[0]) THEN BEGIN
				plot_y_min = !y.range[0]
				temp_new_display[19] = !y.range[0]
				widget_control, zstack_plot_par.plot_y_min_label, $
						set_value = ' '+strtrim(string(!y.range[0],format='(f10.3)'),2)
			ENDIF
			IF (!y.range[1] NE init_value[1]) THEN BEGIN
				plot_y_max = !y.range[1]
				temp_new_display[20] = !y.range[1]
				widget_control, zstack_plot_par.plot_y_max_label, $
						set_value = ' '+strtrim(string(!y.range[1],format='(f10.3)'),2)
			ENDIF
			zstack_plot_sensitive
		ENDIF
;		dummy = where(temp_new_display NE temp_old_display, count)
;		IF (count EQ 0) THEN BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 0
;		ENDIF ELSE BEGIN
;			widget_control, zstack_plot_par.update_label, sensitive = 1
;		ENDELSE
	END
	zstack_plot_par.filename_display_list_label : BEGIN
		zstack_plot_desensitive
		displayed_file_index = event.index
		zstack_plot_imgdisp,displayed_file_index
		zstack_plot_plotspectrum, displayed_file_index
		zstack_plot_sensitive
	END
	zstack_plot_par.next_image_label : BEGIN
		zstack_plot_desensitive
		IF (displayed_file_index LT (n_files-1)) THEN BEGIN
			displayed_file_index = displayed_file_index+1
		ENDIF ELSE BEGIN
			displayed_file_index = 0
		ENDELSE
		zstack_plot_imgdisp,displayed_file_index
			CASE zstack_subroutine OF
				'align' : BEGIN
					zstack_plot_plotshifts, displayed_file_index
				END
				'tune' : BEGIN
					zstack_plot_plotshifts, displayed_file_index
				END
				'spectra' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				'profile' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				'save' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				ELSE : BEGIN
					print,'Unrecognized zstack_subroutine : zstack_plot_par.plot_y_max_label'
				END
			ENDCASE
;			zstack_plot_plotspectrum, displayed_file_index
		zstack_plot_sensitive
	END

	zstack_plot_par.prev_image_label : BEGIN
		zstack_plot_desensitive
		IF (displayed_file_index GT 0) THEN BEGIN
			displayed_file_index = displayed_file_index-1
		ENDIF ELSE BEGIN
			displayed_file_index = n_files-1
		ENDELSE
		zstack_plot_imgdisp,displayed_file_index
			CASE zstack_subroutine OF
				'align' : BEGIN
					zstack_plot_plotshifts, displayed_file_index
				END
				'tune' : BEGIN
					zstack_plot_plotshifts, displayed_file_index
				END
				'spectra' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				'profile' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				'save' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				ELSE : BEGIN
					print,'Unrecognized zstack_subroutine : zstack_plot_par.plot_y_max_label'
				END
			ENDCASE
;			zstack_plot_plotspectrum, displayed_file_index
		zstack_plot_sensitive
	END

	zstack_plot_par.play_movie_label : BEGIN
		zstack_plot_desensitive
		FOR i=0,(n_files-1) DO BEGIN
			IF (displayed_file_index LT (n_files-1)) THEN BEGIN
				displayed_file_index = displayed_file_index+1
			ENDIF ELSE BEGIN
				displayed_file_index = 0
			ENDELSE
			zstack_plot_imgdisp, displayed_file_index
			CASE zstack_subroutine OF
				'align' : BEGIN
					zstack_plot_plotshifts, displayed_file_index
				END
				'tune' : BEGIN
					zstack_plot_plotshifts, displayed_file_index
				END
				'spectra' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				'profile' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				'save' : BEGIN
					zstack_plot_plotspectrum, displayed_file_index
				END
				ELSE : BEGIN
					print,'Unrecognized zstack_subroutine : zstack_plot_par.plot_y_max_label'
				END
			ENDCASE
;			zstack_plot_plotspectrum, displayed_file_index
			wait,movie_delay
		ENDFOR
		zstack_plot_sensitive
	END
	zstack_plot_par.display_parameters_label : BEGIN
;		widget_control,zstack_plot_par.main_base,/destroy
;		zstack_display, zstack_subroutine, new_zoom=new_zoom
		IF (image_zoom NE temp_new_display[0]) THEN BEGIN
			temp_new_display[0] = image_zoom
			widget_control, zstack_display_par.main_base, /destroy
			zstack_display, zstack_subroutine, /new_zoom
		ENDIF
	END

	zstack_plot_par.plot_label : BEGIN
		zstack_plot_desensitive
		CASE zstack_subroutine OF

			'align' : BEGIN
				;; only act on a down click of the mouse
				IF (event.type EQ 0) THEN BEGIN
					wset,zstack_plot_par.plot_win
					this_file_shift = convert_coord(event.x,event.y,/device,/to_data)
					this_file = this_file_shift(0)
					dummy = min(abs(this_file-file_num),this_i_file)
					print,'You clicked at File # '+$
						strtrim(string(file_num(this_i_file),format='(i3)'),2)+$
						', '+filename_display_list(this_i_file)	;+$
;						', dx = '+strtrim(string(x_shift(this_i_file),format='(f10.2)'),2)+$
;						', dy = '+strtrim(string(y_shift(this_i_file),format='(f10.2)'),2)
					displayed_file_index = (this_i_file>0)<(n_files-1)
					widget_control, zstack_plot_par.filename_display_list_label,$
						set_list_select=displayed_file_index
					zstack_plot_imgdisp, displayed_file_index
					zstack_plot_plotshifts, displayed_file_index
				ENDIF
			END
			'tune' : BEGIN
				;; only act on a down click of the mouse
				IF (event.type EQ 0) THEN BEGIN
					wset,zstack_plot_par.plot_win
					this_file_shift = convert_coord(event.x,event.y,/device,/to_data)
					this_file = this_file_shift(0)
					dummy = min(abs(this_file-file_num),this_i_file)
					print,'You clicked at File # '+$
						strtrim(string(file_num(this_i_file),format='(i3)'),2)+$
						', '+filename_display_list(this_i_file)	;+$
;						', old dx = '+strtrim(string(x_shift(this_i_file),format='(f10.2)'),2)+$
;						', old dy = '+strtrim(string(y_shift(this_i_file),format='(f10.2)'),2)
;						', new dx = '+strtrim(string(x_shift(this_i_file),format='(f10.2)'),2)+$
;						', new dy = '+strtrim(string(y_shift(this_i_file),format='(f10.2)'),2)
					displayed_file_index = (this_i_file>0)<(n_files-1)
					widget_control, zstack_plot_par.filename_display_list_label,$
						set_list_select=displayed_file_index
					zstack_plot_imgdisp, displayed_file_index
					zstack_plot_plotshifts, displayed_file_index
				ENDIF
			END
			'spectra' : BEGIN
				;; only act on a down click of the mouse
				IF (event.type eq 0) THEN BEGIN
					wset,zstack_plot_par.plot_win
					this_ev_abs = convert_coord(event.x,event.y,/device,/to_data)
					this_ev = this_ev_abs(0)
					dummy = min(abs(this_ev-ev_list),this_i_ev)
					print,'You clicked at '+$
						strtrim(string(ev_list(this_i_ev),format='(f10.2)'),2)+' eV, '+$
						strtrim(string(12398.52/ev_list(this_i_ev),format='(f10.3)'),2)+' A'
					displayed_file_index = (this_i_ev>0)<(n_files-1)
					widget_control, zstack_plot_par.filename_display_list_label,$
						set_list_select=displayed_file_index
					zstack_plot_imgdisp, displayed_file_index
					zstack_plot_plotspectrum, displayed_file_index
				ENDIF
			END
			'profile' : BEGIN
				;; only act on a down click of the mouse
				IF (event.type eq 0) THEN BEGIN
					wset,zstack_plot_par.plot_win
					this_ev_abs = convert_coord(event.x,event.y,/device,/to_data)
					this_ev = this_ev_abs(0)
					dummy = min(abs(this_ev-ev_list),this_i_ev)
					print,'You clicked at '+$
						strtrim(string(ev_list(this_i_ev),format='(f10.2)'),2)+' eV, '+$
						strtrim(string(12398.52/ev_list(this_i_ev),format='(f10.3)'),2)+' A'
					displayed_file_index = (this_i_ev>0)<(n_files-1)
					widget_control, zstack_plot_par.filename_display_list_label,$
						set_list_select=displayed_file_index
					zstack_plot_imgdisp, displayed_file_index
					zstack_plot_plotspectrum, displayed_file_index
				ENDIF
			END
			'save' : BEGIN
				;; only act on a down click of the mouse
				IF (event.type eq 0) THEN BEGIN
					wset,zstack_plot_par.plot_win
					this_ev_abs = convert_coord(event.x,event.y,/device,/to_data)
					this_ev = this_ev_abs(0)
					dummy = min(abs(this_ev-ev_list),this_i_ev)
					print,'You clicked at '+$
						strtrim(string(ev_list(this_i_ev),format='(f10.2)'),2)+' eV, '+$
						strtrim(string(12398.52/ev_list(this_i_ev),format='(f10.3)'),2)+' A'
					displayed_file_index = (this_i_ev>0)<(n_files-1)
					widget_control, zstack_plot_par.filename_display_list_label,$
						set_list_select=displayed_file_index
					zstack_plot_imgdisp, displayed_file_index
					zstack_plot_plotspectrum, displayed_file_index
				ENDIF
			END
			ELSE : BEGIN
				print,'Unrecognized zstack_subroutine : zstack_plot.plot_win'
			END
		ENDCASE
		zstack_plot_sensitive
    END
	zstack_plot_par.update_label : BEGIN
		widget_control,zstack_plot_par.main_base,/destroy
		CASE zstack_subroutine OF
			'buildlist' : BEGIN
				zstack_buildlist_desensitive
				wshow, zstack_build_list_par.main_base_win
				IF (n_elements(filename_list) EQ 1) THEN BEGIN
 					zstack_buildlist_readfile
				ENDIF ELSE BEGIN
					zstack_buildlist_imgdisp,displayed_file_index
					zstack_analyze_colorbar, zstack_build_list_par.colorbar_win
				ENDELSE
				zstack_buildlist_sensitive
			END
			'align' : BEGIN
				IF (image_zoom EQ temp_old_display[0]) THEN BEGIN
					wshow, zstack_align_par.main_base_win
					zstack_align_desensitive
					zstack_align_imgdisp,displayed_file_index
					zstack_align_corrdisp,displayed_file_index
					zstack_analyze_colorbar, zstack_align_par.colorbar_win
					zstack_align_plotshifts, displayed_file_index
					zstack_align_sensitive
				ENDIF ELSE BEGIN
					widget_control, zstack_align_par.main_base, /destroy
					zstack_align,/new_zoom	; new zoom factor
				ENDELSE
			END
			'tune' : BEGIN
				IF (image_zoom EQ temp_old_display[0]) THEN BEGIN
					wshow, zstack_tune_par.main_base_win
					zstack_tune_desensitive
					IF (subimage_zoom NE temp_old_display[5]) THEN BEGIN
						wset,zstack_tune_par.closeup_this_image_win
						erase,0
						IF (doalign_complete NE 2) THEN BEGIN		; don't plot if using stored shift file
							wset,zstack_tune_par.closeup_corr_image_win
							erase,0
						ENDIF
						wset,zstack_tune_par.closeup_shift_image_win
						erase,0
					ENDIF
					zstack_tune_imgdisp,displayed_file_index
					IF (abs(doalign_complete) EQ 1) THEN BEGIN		; will only display if noy using stored shift file
						zstack_tune_corrdisp,displayed_file_index
					ENDIF
					zstack_analyze_colorbar, zstack_tune_par.colorbar_win
					zstack_tune_subimgdisp,displayed_file_index
					zstack_tune_plotshifts, displayed_file_index
					zstack_tune_sensitive
			ENDIF ELSE BEGIN
					widget_control, zstack_tune_par.main_base, /destroy
					zstack_tune,/new_zoom
				ENDELSE
			END
			'spectra' : BEGIN
				IF (image_zoom EQ temp_old_display[0]) THEN BEGIN
					wshow, zstack_spectra_par.main_base_win
					zstack_spectra_desensitive
					zstack_spectra_imgdisp,displayed_file_index
					zstack_analyze_colorbar, zstack_spectra_par.colorbar_win
					zstack_spectra_plotspectrum, displayed_file_index
					zstack_spectra_sensitive
				ENDIF ELSE BEGIN
					widget_control, zstack_spectra_par.main_base, /destroy
					zstack_spectra,/new_zoom
				ENDELSE
			END
			'profile' : BEGIN
				; need to modify zstack_profile to use new_zoom option
				IF (image_zoom EQ temp_old_display[0]) THEN BEGIN
					wshow, zstack_profile_par.main_base_win
					zstack_profile_desensitive
					wset,zstack_profile_par.profile_image_win
					erase
					IF ( (spectrum_display NE temp_old_display[13]) OR $
							(spectrum_offset NE temp_old_display[7]) ) THEN BEGIN
						profile_y_min = 0
						profile_y_max = 0
					ENDIF
					zstack_profile_imgdisp,displayed_file_index
					zstack_profile_profprep,profile_byte_image
					zstack_profile_profdisp,profile_byte_image
					zstack_analyze_colorbar, zstack_profile_par.colorbar_win
					zstack_profile_plotprofile, displayed_file_index
					zstack_profile_plotspectrum, displayed_file_index
					zstack_profile_sensitive
				ENDIF ELSE BEGIN
					IF ( (spectrum_display NE temp_old_display[13]) OR $
							(spectrum_offset NE temp_old_display[7]) ) THEN BEGIN
						profile_y_min = 0
						profile_y_max = 0
					ENDIF
					widget_control, zstack_profile_par.main_base, /destroy
					zstack_profile,/new_zoom,/passed_data
				ENDELSE
			END
			'save' : BEGIN
				IF (image_zoom EQ temp_old_display[0]) THEN BEGIN
					wshow, zstack_save_par.main_base_win
					zstack_save_desensitive
					zstack_save_imgdisp,displayed_file_index
					zstack_analyze_colorbar, zstack_save_par.colorbar_win
					zstack_save_plotspectrum, displayed_file_index
					zstack_save_sensitive
				ENDIF ELSE BEGIN
					widget_control, zstack_save_par.main_base, /destroy
					zstack_save,/new_zoom	; new zoom factor
				ENDELSE
			END
			ELSE : BEGIN
				print,'Unrecognized zstack_subroutine : zstack_display_par.update_label'
			END
		ENDCASE
	END

	zstack_plot_par.close_label : BEGIN
		widget_control,zstack_plot_par.main_base,/destroy
		image_zoom 		= temp_old_display[ 0]
		disp_min 			= temp_old_display[ 1]
		disp_max 			= temp_old_display[ 2]
		disp_gamma 		= temp_old_display[ 3]
		movie_delay 		= temp_old_display[ 4]
		subimage_zoom 		= temp_old_display[ 5]
		profile_zoom 		= temp_old_display[ 6]
		spectrum_offset 	= temp_old_display[ 7]
		new_color_table 	= temp_old_display[ 8]
		image_invert 		= temp_old_display[ 9]
		new_plot_colors 	= temp_old_display[10]
		image_ratio 		= temp_old_display[11]
		image_scale 		= temp_old_display[12]
		spectrum_display 	= temp_old_display[13]
		image_range 		= temp_old_display[14]
		x_autoscale 		= temp_old_display[15]
		plot_x_min 		= temp_old_display[16]
		plot_x_max 		= temp_old_display[17]
		y_autoscale 		= temp_old_display[18]
		plot_y_min 		= temp_old_display[19]
		plot_y_max 		= temp_old_display[20]

		CASE zstack_subroutine OF
			'buildlist' : BEGIN
				wshow,zstack_build_list_par.main_base_win
				zstack_buildlist_imgdisp,displayed_file_index
				zstack_buildlist_sensitive
			END
			'align' : BEGIN
				wshow, zstack_align_par.main_base_win
				zstack_align_imgdisp,displayed_file_index
				IF (doalign_complete GT 0) THEN BEGIN
					zstack_align_plotshifts, displayed_file_index
				ENDIF
				zstack_align_sensitive
			END
			'tune' : BEGIN
				wshow, zstack_tune_par.main_base_win
				zstack_tune_imgdisp,displayed_file_index
				zstack_tune_subimgdisp,displayed_file_index
;				IF (doalign_complete GT 0) THEN BEGIN
					zstack_tune_plotshifts, displayed_file_index
;				ENDIF
				zstack_tune_sensitive
			END
			'spectra' : BEGIN
				wshow, zstack_spectra_par.main_base_win
				zstack_spectra_imgdisp,displayed_file_index
				zstack_spectra_plotspectrum, displayed_file_index
				zstack_spectra_sensitive
			END
			'profile' : BEGIN
				wshow, zstack_profile_par.main_base_win
				zstack_profile_imgdisp,displayed_file_index
				zstack_profile_plotspectrum, displayed_file_index
				zstack_profile_sensitive
			END
			'save' : BEGIN
				wshow, zstack_save_par.main_base_win
				zstack_save_imgdisp,displayed_file_index
				zstack_save_plotspectrum, displayed_file_index
				zstack_save_sensitive
			END
			ELSE : BEGIN
				print,'Unrecognized zstack_subroutine : zstack_plot_par.close_label'
			END
		ENDCASE
	END
	ELSE:BEGIN
		; would prefer CASE structure, but it does seem to work.
		; it goes through every case, executing the code
		; This IF-THEN structure solves the problem
		IF (event.id EQ zstack_plot_par.raw_image_label) THEN BEGIN
			; if original data is being displayed
			image_ratio = 0
			temp_new_display[11] = image_ratio
			zstack_plot_imgdisp,displayed_file_index
			dummy = where(temp_new_display NE temp_old_display, count)
			IF (count EQ 0) THEN BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 0
			ENDIF ELSE BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 1
			ENDELSE
		ENDIF
		IF (event.id EQ zstack_plot_par.ratio_image_label) THEN BEGIN
			; if (image/current image) is being displayed
			image_ratio = 1
			temp_new_display[11] = image_ratio
			ratio_image_index = displayed_file_index
			zstack_plot_imgdisp,displayed_file_index
			dummy = where(temp_new_display NE temp_old_display, count)
			IF (count EQ 0) THEN BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 0
			ENDIF ELSE BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 1
			ENDELSE
		ENDIF

		IF (	event.id EQ zstack_plot_par.log_ratio_image_label) THEN BEGIN
			; if (image - current image) is being displayed
			image_ratio = 2
			temp_new_display[11] = image_ratio
			ratio_image_index = displayed_file_index
			zstack_plot_imgdisp,displayed_file_index
			dummy = where(temp_new_display NE temp_old_display, count)
			IF (count EQ 0) THEN BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 0
			ENDIF ELSE BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 1
			ENDELSE
		ENDIF

		IF (event.id EQ zstack_plot_par.subtract_image_label) THEN BEGIN
			; if (image - current image) is being displayed
			image_ratio = 3
			temp_new_display[11] = image_ratio
			ratio_image_index = displayed_file_index
			zstack_plot_imgdisp,displayed_file_index
			dummy = where(temp_new_display NE temp_old_display, count)
			IF (count EQ 0) THEN BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 0
			ENDIF ELSE BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 1
			ENDELSE
		ENDIF

		IF (event.id EQ zstack_plot_par.ratio_spectrum_label) THEN BEGIN
			; if (image/i0 spectrum) is being displayed
			image_ratio = 4
			temp_new_display[11] = image_ratio
			zstack_plot_imgdisp,displayed_file_index
			dummy = where(temp_new_display NE temp_old_display, count)
			IF (count EQ 0) THEN BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 0
			ENDIF ELSE BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 1
			ENDELSE
		ENDIF
		IF (event.id EQ zstack_plot_par.log_ratio_spectrum_label) THEN BEGIN
			; if -log(image/i0 spectrum) is being displayed
			image_ratio = 5
			temp_new_display[11] = image_ratio
			zstack_plot_imgdisp,displayed_file_index
			dummy = where(temp_new_display NE temp_old_display, count)
			IF (count EQ 0) THEN BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 0
			ENDIF ELSE BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 1
			ENDELSE
		ENDIF
		IF (event.id EQ zstack_plot_par.subtract_spectrum_label) THEN BEGIN
			; if (image - i0 spectrum) is being displayed
			image_ratio = 6
			temp_new_display[11] = image_ratio
			zstack_plot_imgdisp,displayed_file_index
			dummy = where(temp_new_display NE temp_old_display, count)
			IF (count EQ 0) THEN BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 0
			ENDIF ELSE BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 1
			ENDELSE
		ENDIF
		IF (event.id EQ zstack_plot_par.subtract_stack_label) THEN BEGIN
			; if (stack - reference stack) is being displayed
			image_ratio = 7
			temp_new_display[11] = image_ratio
			zstack_plot_imgdisp,displayed_file_index
			dummy = where(temp_new_display NE temp_old_display, count)
			IF (count EQ 0) THEN BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 0
			ENDIF ELSE BEGIN
				widget_control, zstack_plot_par.update_label, sensitive = 1
			ENDELSE
		ENDIF
	END
ENDCASE
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_plot_dialog, group_leader_ID=groupLeader
;print,'zstack_plot_dialog'
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

; Initialize variables if they have no value
;	needed when running zstack_align_dialog as stand-alone prior to running zstack_align
;	i.e., when programming dialog window
IF (n_elements(n_cols) EQ 0) THEN n_cols = 100
IF (n_elements(n_rows) EQ 0) THEN n_rows = 100
IF (n_elements(image_zoom) EQ 0) THEN image_zoom = 1
IF (n_elements(data_shifted) EQ 0) then data_shifted = 1
IF (n_elements(n_clipped_cols) EQ 0) THEN n_clipped_cols = 75
IF (n_elements(n_clipped_rows) EQ 0) THEN n_clipped_rows = 75
IF (n_elements(doalign_complete) EQ 0) THEN doalign_complete = 0
IF (n_elements(n_roi) EQ 0) THEN n_roi = ''
IF (n_elements(i0_roi) EQ 0) THEN i0_roi = ''
IF (n_elements(zstack_subroutine) EQ 0) THEN zstack_subroutine = 'spectra'
IF (n_tags(zstack_plot_par) EQ 0) THEN BEGIN
    zstack_plot_par = $
	   { zstack_plot_par				    , $
		main_base 						: 0L, $
		image_zoom_label 				: 0L, $
		disp_min_label 					: 0L, $
		disp_max_label 					: 0L, $
		disp_gamma_label 				: 0L, $
		movie_delay_label 				: 0L, $
		subimage_zoom_label 				: 0L, $
		profile_zoom_label 				: 0L, $
		spectrum_offset_label 			: 0L, $
		raw_image_label 				: 0L, $
		ratio_image_label 				: 0L, $
		log_ratio_image_label 			: 0L, $
		subtract_image_label				: 0L, $
		ratio_spectrum_label 				: 0L, $
		log_ratio_spectrum_label 			: 0L, $
		subtract_spectrum_label			: 0L, $
		subtract_stack_label				: 0L, $
		reference_spectrum_label 			: 0L, $
		reference_spectrum_menu_label 		: 0L, $
		reference_spectrum_filename_label 	: 0L, $
		reference_spectrum_factor_label		: 0L, $
		reference_image_label 			: 0L, $
		reference_image_menu_label 		: 0L, $
		reference_image_filename_label 		: 0L, $
		reference_image_factor_label		: 0L, $
		reference_stack_label 			: 0L, $
		reference_stack_filename_label 		: 0L, $
		reference_stack_factor_label		: 0L, $
		absolute_scale_label 				: 0L, $
		relative_scale_label 				: 0L, $
		image_range_label 				: 0L, $
		stack_range_label 				: 0L, $
		image_i0_scale_label 				: 0L, $
		xloadct_label 					: 0L, $
		invertct_label 					: 0L, $
		loadplotct_label 				: 0L, $
		spectra_singlebeam_label 			: 0L, $
		spectra_tranmittance_label 		: 0L, $
		spectra_absorbance_label 			: 0L, $
		plot_x_min_label 				: 0L, $
		plot_x_max_label 				: 0L, $
		x_autoscale_label 				: 0L, $
		plot_y_min_label 				: 0L, $
		plot_y_max_label 				: 0L, $
		y_autoscale_label 				: 0L, $
		image_label 					: 0L, $
		shifted_image_label 				: 0L, $
		clipped_image_label 				: 0L, $
		colorbar_label 					: 0L, $
		filename_display_list_label 		: 0L, $
		next_image_label 				: 0L, $
		prev_image_label 				: 0L, $
		play_movie_label 				: 0L, $
		display_parameters_label			: 0L, $
		plot_label 					: 0L, $
		update_label 					: 0L, $
		close_label 					: 0L, $
		image_win 						: 0L, $
		shifted_image_win 				: 0L, $
		clipped_image_win 				: 0L, $
		colorbar_win 					: 0L, $
		plot_win 						: 0L, $
		main_base_win 					: 0L  $
		}
ENDIF
screen_size = get_screen_size()
;print,screen_size
IF N_Elements(groupLeader) EQ 0 THEN BEGIN
	IF !version.os_family EQ 'Windows' then begin
		zstack_plot_par.main_base = widget_base(title='ZSTACK Plot Parameters', /row, /SCROLL)
	ENDIF else begin
		zstack_plot_par.main_base = widget_base(title='ZSTACK Plot Parameters', /row, /SCROLL, $
	              X_SCROLL_SIZE=screen_size(0)*.90,Y_SCROLL_SIZE=screen_size(1)*.90)
	ENDELSE
ENDIF ELSE BEGIN
	zstack_plot_par.main_base = widget_base(title='ZSTACK Plot Parameters', /row, $
			/Modal, group_leader_ID=groupLeader )
ENDELSE

base = widget_base(zstack_plot_par.main_base,/col);,space=0)
row1 = widget_base(base,/row,/align_center)
col1 = widget_base(row1, /col)
row = widget_base(col1, /row, /frame)
col1a = widget_base(row, /col, space=0)
row = widget_base(col1a,/row)
label = widget_label( row, value = 'Image zoom factor: ', xsize=170, /align_left)
zstack_plot_par.image_zoom_label = widget_text(row,/editable,xsize=10)
row = widget_base(col1a,/row)
label = widget_label( row, value = 'Movie delay (sec per frame) : ', xsize=170, /align_left)
zstack_plot_par.movie_delay_label = widget_text(row,/editable,xsize=10)
row = widget_base(col1a,/row)
label = widget_label( row, value = 'Spectrum Offset: ', xsize=170, /align_left)
zstack_plot_par.spectrum_offset_label = widget_text(row,/editable,xsize=10)
;IF ( ((size(i0_roi))(0) NE 0) OR (n_roi NE 0) ) THEN BEGIN
	row = widget_base(col1, /row, /frame)
	col1b = widget_base(row, /col, space=0)
	row = widget_base(col1b,/row)
	label = widget_label(row,value = 'Display spectra as:',/align_left)
	row = widget_base(col1b,/row)
	col = widget_base(row,/col,/exclusive,space=0)
	zstack_plot_par.spectra_singlebeam_label = $
			widget_button(col, value = '  Single beam')
	zstack_plot_par.spectra_tranmittance_label = $
			widget_button(col, value = '  % Transmittance')
	zstack_plot_par.spectra_absorbance_label = $
			widget_button(col, value = '  Absorbance')
;ENDIF
row = widget_base(col1, /row, /frame)
col1c = widget_base(row, /col, space=0)
row = widget_base(col1c, /row, space=0)
label = widget_label(row, xsize=100, value='Plot Scaling :', /align_left)
row = widget_base(col1c, /row, space=0)
label = widget_label(row, xsize=100, value='  X Range :', /align_left)
col = widget_base(row,/col,/nonexclusive)
zstack_plot_par.x_autoscale_label = widget_button(col, value='Autoscale')
row = widget_base(col1c,/row)
label = widget_label(row, xsize=100, value='    Minimum :', /align_left)
zstack_plot_par.plot_x_min_label = widget_text(row, xsize=10, /editable, value='')
row = widget_base(col1c,/row)
label = widget_label(row, xsize=100, value='    Maximum :', /align_left)
zstack_plot_par.plot_x_max_label = widget_text(row, xsize=10, /editable, value='')
row = widget_base(col1c, /row, space=0)
label = widget_label(row, xsize=100, value='  Y Range :', /align_left)
col = widget_base(row,/col,/nonexclusive)
zstack_plot_par.y_autoscale_label = widget_button(col, value='Autoscale')
row = widget_base(col1c,/row)
label = widget_label(row, xsize=100, value='    Minimum :', /align_left)
zstack_plot_par.plot_y_min_label = widget_text(row, xsize=10, /editable, value='')
row = widget_base(col1c,/row)
label = widget_label(row, xsize=100, value='    Maximum :', /align_left)
zstack_plot_par.plot_y_max_label = widget_text(row, xsize=10, /editable, value='')
row = widget_base(col1, /row, /frame)
col1d = widget_base(row, /col, space=0)
row = widget_base(col1d,/row)
zstack_plot_par.xloadct_label = widget_button(row,value='Load New Color Table', xsize=250)
row = widget_base(col1d,/row)
zstack_plot_par.invertct_label = widget_button(row,value='Invert Color Table', xsize=250)
row = widget_base(col1d,/row)
zstack_plot_par.loadplotct_label = widget_button(row,value='Select Plot Colors', xsize=250)
col2 = widget_base(row1, /col)
row = widget_base(col2, /row, /frame, /align_center)
CASE zstack_subroutine OF
	'buildlist' : BEGIN
		col2a = widget_base(row,/col)
		label = widget_label(col2a, value='STXM Image',/align_center)
		zstack_plot_par.image_label = widget_draw(col2a, $
				xsize=n_cols*image_zoom,ysize=n_rows*image_zoom,retain=2,/align_center,/button_events)
	END
	'align' : BEGIN
		col2a = widget_base(row,/col)
		label = widget_label(col2a, value='STXM Image',/align_center)
		zstack_plot_par.image_label = widget_draw(col2a, $
				xsize=n_cols*image_zoom,ysize=n_rows*image_zoom,retain=2,/align_center,/button_events)

		IF (doalign_complete NE 0) THEN BEGIN
			col2b = widget_base(row,/col)
			label = widget_label(col2b, value='Shifted Image',/align_center)
			zstack_plot_par.shifted_image_label = widget_draw(col2b, $
					xsize=n_cols*image_zoom,ysize=n_rows*image_zoom, retain=2,/align_center)
		ENDIF
	END
	'tune' : BEGIN
		col2a = widget_base(row,/col)
		label = widget_label(col2a, value='STXM Image',/align_center)
		zstack_plot_par.image_label = widget_draw(col2a, $
				xsize=n_cols*image_zoom,ysize=n_rows*image_zoom,retain=2,/align_center,/button_events)

		IF (doalign_complete NE 0) THEN BEGIN
			col2b = widget_base(row,/col)
			label = widget_label(col2b, value='Shifted Image',/align_center)
			zstack_plot_par.shifted_image_label = widget_draw(col2b, $
					xsize=n_cols*image_zoom,ysize=n_rows*image_zoom, retain=2,/align_center)
		ENDIF
	END
	'spectra' : BEGIN
		col2a = widget_base(row, /col)
		IF (data_shifted EQ 1) THEN BEGIN
			label = widget_label(col2a, value='Shifted Image',/align_center)
		ENDIF ELSE BEGIN
			label = widget_label(col2a, value='Unshifted Image',/align_center)
		ENDELSE
		zstack_plot_par.image_label = widget_draw(col2a,$
				xsize=n_cols*image_zoom, ysize=n_rows*image_zoom,$
				retain=2,/button_events,/motion,/align_center)
		IF (data_shifted EQ 1) THEN BEGIN
			col2b = widget_base(row,/col)
			label = widget_label(col2b, value='Clipped Image',/align_center)
			zstack_plot_par.clipped_image_label = widget_draw(col2b,$
					xsize=n_clipped_cols*image_zoom, ysize=n_clipped_rows*image_zoom,$
					retain=2,/button_events,/align_center)
		ENDIF
	END
	'profile' : BEGIN
		col2a = widget_base(row, /col)
		IF (data_shifted EQ 1) THEN BEGIN
			label = widget_label(col2a, value='Shifted Image',/align_center)
		ENDIF ELSE BEGIN
			label = widget_label(col2a, value='Unshifted Image',/align_center)
		ENDELSE
		zstack_plot_par.image_label = widget_draw(col2a,$
				xsize=n_cols*image_zoom, ysize=n_rows*image_zoom,$
				retain=2,/button_events,/motion,/align_center)
	END
	'save' : BEGIN
		col2a = widget_base(row, /col)
		IF (data_shifted EQ 1) THEN BEGIN
			label = widget_label(col2a, value='Shifted Image',/align_center)
		ENDIF ELSE BEGIN
			label = widget_label(col2a, value='Unshifted Image',/align_center)
		ENDELSE
		zstack_plot_par.image_label = widget_draw(col2a,$
				xsize=n_cols*image_zoom, ysize=n_rows*image_zoom,$
				retain=2,/button_events,/motion,/align_center)
		IF (data_shifted EQ 1) THEN BEGIN
			col2b = widget_base(row,/col)
			label = widget_label(col2b, value='Clipped Image',/align_center)
			zstack_plot_par.clipped_image_label = widget_draw(col2b,$
					xsize=n_clipped_cols*image_zoom, ysize=n_clipped_rows*image_zoom,$
					retain=2,/button_events,/align_center)
		ENDIF
	END
	ELSE : BEGIN
		print,'Unrecognized zstack_subroutine : zstack_plot'
	END
ENDCASE
col2c = widget_base(row,/col)
label = widget_label(col2c, value='I',/align_center)
zstack_plot_par.colorbar_label = widget_draw(col2c,$
              xsize=10*(image_zoom>1), ysize=n_rows*image_zoom,$
              retain=2, sensitive=0, /align_center)
zstack_plot_par.filename_display_list_label = widget_droplist( col2, $
		value = 'Select File', /align_center, scr_xsize=250)
zstack_plot_par.prev_image_label = widget_button(col2, value='Display Previous Image')
zstack_plot_par.next_image_label = widget_button(col2, value='Display Next Image')
zstack_plot_par.play_movie_label = widget_button(col2,value='Play movie')
row = widget_base( col2, /row ,/align_center)
zstack_plot_par.display_parameters_label = widget_button(row,value='Display Parameters')
row = widget_base(col2,/row,/align_center)
zstack_plot_par.plot_label = widget_draw(row, $
			xsize=((3*n_cols*image_zoom+25)>250)<500, $
			ysize=((n_rows*image_zoom)>150)<250,retain=2,/button_events)
row = widget_base(col2,/row)
zstack_plot_par.update_label = widget_button(col2,value='Update Image Display')
zstack_plot_par.close_label = widget_button(col2,value='Close without Update')
;;;;;;;;;;
Device, get_screen_size=screen_size
screen_center = [ screen_size(0) / 2 , screen_size(1) / 2 ]
geom = Widget_Info(zstack_plot_par.main_base, /Geometry)
base_size = [geom.scr_xsize,geom.scr_ysize]
x_base_size = (fix(0.90*screen_size[0]) < base_size[0])
y_base_size = (fix(0.95*screen_size[1]) < base_size[1])
halfsize = [ x_base_size / 2 , y_base_size / 2 ]
widget_control, zstack_plot_par.main_base, $
	XOffset = screen_center[0] - halfsize[0], $
	YOffset = screen_center[1] - halfsize[1], $
	scr_xsize = x_base_size, $
	scr_ysize = y_base_size
widget_control, zstack_plot_par.main_base, /realize
zstack_plot_par.main_base_win = !d.window
widget_control,zstack_plot_par.image_label, get_value = window
zstack_plot_par.image_win = window
CASE zstack_subroutine OF
	'buildlist' : BEGIN
	END
	'align' : BEGIN
		IF (doalign_complete NE 0) THEN BEGIN
			widget_control,zstack_plot_par.shifted_image_label, get_value = window
			zstack_plot_par.shifted_image_win = window
		ENDIF
	END
	'tune' : BEGIN
		IF (doalign_complete NE 0) THEN BEGIN
			widget_control,zstack_plot_par.shifted_image_label, get_value = window
			zstack_plot_par.shifted_image_win = window
		ENDIF
	END
	'spectra' : BEGIN
		IF (data_shifted EQ 1) THEN BEGIN
			widget_control,zstack_plot_par.clipped_image_label, get_value = window
			zstack_plot_par.clipped_image_win = window
		ENDIF
	END
	'profile' : BEGIN
	END
	'save' : BEGIN
		IF (data_shifted EQ 1) THEN BEGIN
			widget_control,zstack_plot_par.clipped_image_label, get_value = window
			zstack_plot_par.clipped_image_win = window
		ENDIF
	END
	ELSE : BEGIN
		print,'Unrecognized zstack_subroutine : zstack_plot'
	END
ENDCASE
widget_control,zstack_plot_par.colorbar_label, get_value = window
zstack_plot_par.colorbar_win = window
widget_control,zstack_plot_par.plot_label, get_value = window
zstack_plot_par.plot_win = window
widget_control, zstack_plot_par.update_label, sensitive = 0
return
END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO zstack_plot, subroutine, new_zoom=new_zoom, group_leader_ID=groupLeader
;print,'zstack_plot'
@bsif_common
@zstack_common
@zstack_analyze_common
@zstack_build_list_common
@zstack_align_common
@zstack_tune_common
@zstack_spectra_common
@zstack_profile_common
@zstack_save_common
@zstack_display_common
@zstack_color_common

on_error,2
zstack_subroutine = subroutine
IF NOT(keyword_set(new_zoom)) THEN BEGIN
	temp_old_display = [	image_zoom, $		;  0
					disp_min, $		;  1
					disp_max, $		;  2
					disp_gamma, $		;  3
					movie_delay, $		;  4
					subimage_zoom, $	;  5
					profile_zoom, $		;  6
					spectrum_offset, $	;  7
					0, $				;  8	new_color_table variable
					image_invert, $		;  9
					0, $				; 10	new_plot_colors variable
					image_ratio, $		; 11
					image_scale, $		; 12
					spectrum_display, $	; 13
					image_range, $		; 14
					x_autoscale, $		; 15
					plot_x_min, $		; 16
					plot_x_max, $		; 17
					y_autoscale, $		; 18
					plot_y_min, $		; 19
					plot_y_max ]		; 20
	temp_new_display = temp_old_display
ENDIF
zstack_plot_dialog, group_leader_ID=groupLeader
zstack_plot_sensitive
widget_control, zstack_plot_par.image_zoom_label, $
		set_value = ' '+strtrim(string(image_zoom,format='(f10.2)'),2)
widget_control, zstack_plot_par.movie_delay_label, $
		set_value=' '+strtrim(string(movie_delay,format='(f10.3)'),2)
widget_control, zstack_plot_par.spectrum_offset_label, $
		set_value = ' '+strtrim(string(spectrum_offset,format='(f10.2)'),2)
widget_control, zstack_plot_par.filename_display_list_label, $
		set_value = filename_display_list
IF ( ((size(i0_roi))(0) NE 0) OR (n_roi NE 0) ) THEN BEGIN
	CASE spectrum_display OF
		1 : BEGIN	; if single beam spectra are being displayed
			widget_control, zstack_plot_par.spectra_singlebeam_label, set_button = 1
		END
		2 : BEGIN	; if percent transmittance spectra are being displayed
			widget_control, zstack_plot_par.spectra_tranmittance_label, set_button = 1
		END
		3 : BEGIN	; if absorbance spectra are being displayed
			widget_control, zstack_plot_par.spectra_absorbance_label, set_button = 1
		END
	ENDCASE
ENDIF
zstack_plot_imgdisp, displayed_file_index
zstack_analyze_colorbar, zstack_plot_par.colorbar_win
CASE zstack_subroutine OF
	'buildlist' : BEGIN
	END
	'align' : BEGIN
		dummy = where([x_shift,y_shift] EQ 0,count)
		IF (count LT 2*n_files) THEN BEGIN
			zstack_plot_plotshifts, displayed_file_index
		ENDIF
	END
	'tune' : BEGIN
		dummy = where([x_shift,y_shift] EQ 0,count)
		IF (count LT 2*n_files) THEN BEGIN
			zstack_plot_plotshifts, displayed_file_index
		ENDIF
	END
	'spectra' : BEGIN
		IF (n_roi GE 1) THEN BEGIN
			zstack_plot_plotspectrum,displayed_file_index
			plot_x_min = !x.range[0]
			plot_x_max = !x.range[1]
			plot_y_min = !y.range[0]
			plot_y_max = !y.range[1]
			widget_control, zstack_plot_par.plot_y_min_label, $
					set_value = ' '+strtrim(string(!y.range[0],format='(f10.3)'),2)
			widget_control, zstack_plot_par.plot_y_max_label, $
					set_value = ' '+strtrim(string(!y.range[1],format='(f10.3)'),2)
			widget_control, zstack_plot_par.plot_x_min_label, $
					set_value = ' '+strtrim(string(!x.range[0],format='(f10.3)'),2)
			widget_control, zstack_plot_par.plot_x_max_label, $
					set_value = ' '+strtrim(string(!x.range[1],format='(f10.3)'),2)
		ENDIF
	END
	'profile' : BEGIN
		zstack_plot_plotspectrum,displayed_file_index
		plot_x_min = !x.range[0]
		plot_x_max = !x.range[1]
		plot_y_min = !y.range[0]
		plot_y_max = !y.range[1]
		widget_control, zstack_plot_par.plot_y_min_label, $
				set_value = ' '+strtrim(string(!y.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_y_max_label, $
				set_value = ' '+strtrim(string(!y.range[1],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_min_label, $
				set_value = ' '+strtrim(string(!x.range[0],format='(f10.3)'),2)
		widget_control, zstack_plot_par.plot_x_max_label, $
				set_value = ' '+strtrim(string(!x.range[1],format='(f10.3)'),2)
	END
	'save' : BEGIN
		IF (n_roi GE 1) THEN BEGIN
			zstack_plot_plotspectrum,displayed_file_index
			plot_x_min = !x.range[0]
			plot_x_max = !x.range[1]
			plot_y_min = !y.range[0]
			plot_y_max = !y.range[1]
			widget_control, zstack_plot_par.plot_y_min_label, $
					set_value = ' '+strtrim(string(!y.range[0],format='(f10.3)'),2)
			widget_control, zstack_plot_par.plot_y_max_label, $
					set_value = ' '+strtrim(string(!y.range[1],format='(f10.3)'),2)
			widget_control, zstack_plot_par.plot_x_min_label, $
					set_value = ' '+strtrim(string(!x.range[0],format='(f10.3)'),2)
			widget_control, zstack_plot_par.plot_x_max_label, $
					set_value = ' '+strtrim(string(!x.range[1],format='(f10.3)'),2)
		ENDIF
	END
	ELSE : BEGIN
		print,'Unrecognized zstack_subroutine : zstack_plot'
	END
ENDCASE
IF (x_autoscale EQ 1) THEN BEGIN
	widget_control, zstack_plot_par.x_autoscale_label, set_button = 1
ENDIF ELSE BEGIN
	widget_control, zstack_plot_par.x_autoscale_label, set_button = 0
ENDELSE
IF (y_autoscale EQ 1) THEN BEGIN
	widget_control, zstack_plot_par.y_autoscale_label, set_button = 1
ENDIF ELSE BEGIN
	widget_control, zstack_plot_par.y_autoscale_label, set_button = 0
ENDELSE
xmanager, 'zstack_plot', zstack_plot_par.main_base, $
		group_leader = zstack_plot_par.main_base
return
end
