; Copyright (c) 1998-2016 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		AX_INTERP_SQUARE
;
;LAST CHANGED: ----------------------------------- 	11 Apr 2016
;
; PURPOSE:
;	This function takes 2D data recorded with non-square pixels
; and interpolates the sampling of the larger pixel axis so that the pixels are square
;
; CATEGORY:
;	Image processing
;
; CALLING SEQUENCE:
;
;	Result = AX_INTERP_SQUARE(TMP)
;
; INPUTS:
; TMP	- axis 2D data structure
;
; KEYWORDS: none
;
; OUTPUTS:
;	returns modified axis 2D data structure with square pixels
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;   SDF.COM     common for sdf
;	NEXUS.COM	common for nexus
;
; MODIFICATION HISTORY:
; (11-Apr-16 aph) first written; taken from read_sdf
;-

function ax_interp_square, tmp
@axis.com
@sdf.com
@nexus_com

; --- check if 2d structure
if tmp.t NE '2d' then begin
	print, 'ax_interp_square: not a 2D data structure'
	return, tmp
endif
; load variables
if n_elements(pix_diff) LE 0 then pix_diff = 0.005
x = tmp.x
y = tmp.y
d = tmp.d
; help, tmp, /struct

; --- check if rectangular pixels - if so interpolate to square
if NOT keyword_set(no_interpolate) then begin
	dx = x(1) - x(0)
	dy = y(1) - y(0)
	if abs(dx - dy) GT pix_diff then begin
		if dx GT dy then begin
			axis_log, ' Rectangular pixels: x interpolated'
			nx = fix((x(n_elements(x)-1) - x(0))/dy) + 1
			x = interpol(x,nx)
			tmp.xl = 'interp-square ' + tmp.xl
		endif else begin
			axis_log, ' Rectangular pixels: y interpolated'
			ny = fix((y(n_elements(y)-1) - y(0))/dx) + 1
			y = interpol(y,ny)
			tmp.yl = 'interp-square ' + tmp.yl
		endelse
		nx = n_elements(x) & ny = n_elements(y)
		td = congrid(d,nx,ny, /interp, cubic=-0.5)
		d = td
	endif
endif

tmp = {t:'2d', x:x, y:y, d:d, e:tmp.e, xl:tmp.xl, yl:tmp.yl, dl:tmp.dl}

return, tmp
end