; Copyright (c) 1998-2016 A.P. Hitchcock, Ben Watts  All rights reserved
;+
;NAME:
;	AX_NEXUS
;
;LAST CHANGED: ----------------------------------- 05-Apr-16 (APH, Ben Watts)
;
;PURPOSE:
;	This set of procedures is a widget to manage the read in files
; in the HDF5 NeXus file format developed for STXM
; The widget uses read_nexus to read the data and convert to aXis2000 internal formats.
;
;CATEGORY:
;	AXIS: utility
;
; AX_NEXUS_EVENT ROUTINES:
; TEXT_NEXUS_FNAME			modify filename; when changed, execute BUT_NEXUS_FNAME_BROWSE to get parameters
; BUT_NEXUS_PARAMETERS  	display scan and microscope parameters
; BUT_NEXUS_FNAME_BROWSE 	browse for new filename; if found, load new parameters
; BUT_NEXUS_STACK_IMAGE    	proceed to read single image from stack
; BUT_NEXUS_OK				proceed with loading NEXUS file into AXIS
; BUT_NEXUS_CANCEL			abort read in
; BUT_NEXUS_NORMALIZE 		toggle flag to enable normalising data to ring current
; BUT_NEXUS_XY_CORRECT 		toggle flag to enable correcting per-pixel positions
; WID_DLIST_CHANNEL			drop list used to display and select the data channel
; BASE_NEXUS_UPDATE   		update widgets
; WID_DLIST_REGION    		drop list to display and select region
; WID_DLIST_IMAGE     		drop list to display and select image
;
;COMMON BLOCKS:
;	@AXIS_COM	standard set of common blocks
;	@NEXUS_COM 	structure with header info
;
;MODIFICATION HISTORY:
; (27-Apr-15 bw)  First version written using ax_sdf.pro as a template
; (13-Jul-15 aph) integrated into 02-July-15 version of aXis2000
; (15-Mar-16 BW) correct switch to display stack image number when reading in
; (27-Mar-16 aph) get multi-region working
; (05-Apr-16 aph) add viewe and parameter display
;-

pro TEXT_NEXUS_FNAME, Event
on_error,2
@axis_com
@nexus_com

; get the changed name
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_NEXUS_FNAME')
WIDGET_CONTROL,id, Sensitive=1, Get_value = fname
IF fname(0) EQ nexus_lastfile then return
; generate full name with path
pfname = nexus_path + fname

; -- check if the filename exists - if not assume this is just a trap out of
; -- the user starting to type a changed name or a mistake
t = findfile(pfname(0), count=t_count) & if t_count EQ 0 then return
;print, 'New filename is ', fname(0)
nexus_lastfile = fname(0)

; update details
 BUT_NEXUS_FNAME_BROWSE, Event, file = fname(0)

end

;-----------------------------------------------------------------
pro BUT_NEXUS_FNAME_BROWSE, Event, file=file
on_error,2
@axis_com
@nexus_com
; get new filename
if NOT keyword_set(file) then begin
	fname = pickfile2(Title='Select NEXUS file', Filter = '*.hdf5')
endif else fname = file
if strlen(fname) GT 0 then begin
	t = ax_name(fname)
	nexus_path = t(0)
	nexus_lastfile = t(1)+'.'+t(2)          ; ensure the filename is saved for future use
; update nexus_path and file_name in the text box
BASE_NEXUS_UPDATE, Event.Top, /new_file
endif
end

pro WID_DLIST_CHANNEL, Event
on_error,2
@axis_com
@nexus_com
; get the changed data channel
nexus_ch_tmp = nexus_ch
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_CHANNEL')
WIDGET_CONTROL,id, Sensitive=1, Get_value = nexus_ch
WIDGET_CONTROL,id, SET_DROPLIST_SELECT = nexus_ch
print, 'changing from ', nexus_ch_tmp , ' to ', nexus_ch
BASE_NEXUS_UPDATE, Event.Top		; update ax_nexus display

end

pro WID_DLIST_REGION, Event
on_error,2
@axis_com
@nexus_com
; get the changed region
nexus_region_tmp = nexus_region
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_REGION')
WIDGET_CONTROL,id, Sensitive=1, Get_value = nexus_region_list
nexus_region = fix(WIDGET_INFO(id, /Droplist_select))
print, 'changing from ', nexus_region_list(nexus_region_tmp) , ' to ', nexus_region_list(nexus_region)
BASE_NEXUS_UPDATE, Event.Top		; update ax_nexus display

end

pro WID_DLIST_IMAGE, Event
on_error,2
@axis_com
@nexus_com
; get the selected image
nexus_image_tmp = nexus_image
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_IMAGE')
WIDGET_CONTROL,id, Sensitive=1, Get_value = nexus_image
print, 'changing from ', nexus_image_tmp , ' to ', nexus_image
;WIDGET_CONTROL,id, SET_DROPLIST_SELECT = nexus_region

BASE_NEXUS_UPDATE, Event.Top		; update ax_nexus display

end


;-----------------------------------------------------------------
pro BUT_NEXUS_OK, Event
on_error,2
@axis_com
@nexus_com
; get the path & filename
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_NEXUS_PATH')
WIDGET_CONTROL,id, Get_value = path
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_NEXUS_FNAME')
WIDGET_CONTROL,id, Sensitive=1, Get_value = fname
fname = path + fname
print, ' Reading in NEXUS file: ', fname

; get the data channel
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_CHANNEL')
nexus_ch = WIDGET_INFO(id, /Droplist_select)
nchnls = n_elements(nexus_ch)
widget_control, /hourglass
; get the region
n_regions = nexus_head.n_regions
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_REGION')
nexus_region = WIDGET_INFO(id, /Droplist_select)
region = nexus_region(0) + 1
if  region LE n_regions then begin
; read the data file for a defined region
	nchnls=n_elements(NEXUS_Channel_List)
	print, 'Reading channel ', string(nexus_ch(0)+1,format='(i2)'), ' of ', string(nchnls,format='(i2)'), $
	' channels.  Region ', string(region,format='(i2)'), ' of ', string(n_regions,format='(i2)'), ' regions.'
	nexus = read_nexus(fname(0), channel=nexus_ch(0)+1, region=region, group=NEXUS_ID, xy_correct=xy_correct_flag)
endif else begin		; read all regions
	print, 'Reading channel ', string(nexus_ch(0)+1,format='(i2)'), ' of ', string(nchnls,format='(i2)'), $
	' channels.  All ', string(n_regions,format='(i2)'), ' regions.'
  nexus = read_nexus(fname(0), channel=nexus_ch(0)+1, region=0,xy_correct=xy_correct_flag, /all_regions, group=NEXUS_ID)
endelse

WIDGET_CONTROL, Event.Top, /DESTROY
return
end

;-----------------------------------------------------------------

pro BUT_NEXUS_NORMALIZE, Event
@axis_com
@nexus_com
on_error,2
if norm_flag EQ 1 then norm_flag = 0 else  norm_flag = 1
; print, ' normalzation status changed to ', norm_flag
 id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_NEXUS_NORMALIZE')
 WIDGET_CONTROL,id, set_button=norm_flag
end

;-----------------------------------------------------------------

pro BUT_NEXUS_XY_CORRECT, Event
@axis_com
@nexus_com
on_error,2
if xy_correct_flag EQ 1 then xy_correct_flag = 0 else  xy_correct_flag = 1
; print, ' xy_correct status changed to ', xy_correct_flag
 id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_NEXUS_XY_CORRECT')
 WIDGET_CONTROL,id, set_button=xy_correct_flag
end

;----------Display Pramters -------------------------------------------------------

pro BUT_NEXUS_PARAMETERS, Event
@axis_com
@nexus_com
on_error,2
; get the path & filename
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_NEXUS_PATH')
WIDGET_CONTROL,id, Get_value = path
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_NEXUS_FNAME')
WIDGET_CONTROL,id, Sensitive=1, Get_value = fname
fname = path + fname
print, ' Opening h5_browser on file: ', fname
tmp = h5_browser(fname)

end

;-------------------------Read in a single image from a stack ----------------------------------------
pro BUT_NEXUS_STACK_IMAGE, Event
@axis_com
@nexus_com
on_error,2

id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_IMAGE')
nexus_image_index = WIDGET_INFO(id, /Droplist_select)

; get the selected image
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_IMAGE')
WIDGET_CONTROL,id, Sensitive=1, Get_value = nexus_image

 print, 'Extracting single image at E = ', nexus_image(nexus_image_index)
nexus_image_index = fix(nexus_image_index) +1	; kluge to avoid problem that IDL uses 0 as false !
; get the path & filename
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_NEXUS_PATH')
WIDGET_CONTROL,id, Get_value = path
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_NEXUS_FNAME')
WIDGET_CONTROL,id, Sensitive=1, Get_value = fname
fname = path + fname ;+ '.hdf5'
;; get the data channel
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_CHANNEL')
this_channel = 1 + WIDGET_INFO(id, /Droplist_select)
 print, 'channel is ', this_channel
; get the region
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_REGION')
this_region = 1 + widget_info(id, /DROPLIST_SELECT)
 print, 'region is ', this_region

widget_control, /hourglass
print, 'ax_nexus calls read_nexus with channel#, region#, image# ', this_channel, this_region, nexus_image_index
nexus = read_nexus(fname(0), channel= this_channel, region = this_region, one_image=nexus_image_index, group=NEXUS_ID)

WIDGET_CONTROL, Event.Top, /DESTROY
end

;-----------------------------------------------------------------
pro BUT_NEXUS_CANCEL, Event
@axis_com
@nexus_com
on_error,2
nexus = 0

WIDGET_CONTROL, Event.Top, /DESTROY
end


;-----------------------------------------------------------------
; ****************************************************************
;-----------------------------------------------------------------
PRO BASE_NEXUS_UPDATE, Widget_id, new_file = new_file ; Update widgets with file info
on_error,2
@axis_com
@nexus_com

if keyword_Set(new_file) then begin	; only do first part if the filename was changed
	; ------------ get new filename
;	IF (size(nexus_path,/type) NE 0) AND (size(nexus_lastfile,/type) NE 0) THEN BEGIN ; if path and filename variables are defined
	IF (size(nexus_path,/type) EQ 0) OR (size(nexus_lastfile,/type) EQ 0) THEN RETURN   ; if path and filename variables are defined
	  PRINT, 'BASE_NEXUS_UPDATE: ', nexus_path, '--',  nexus_lastfile
	  t = findfile(nexus_path+nexus_lastfile, count=t_count)
	  IF t_count NE 0 THEN BEGIN ; if the file can be found
	    ; update nexus_path and file_name in the text box
	    id = widget_info(Widget_id, FIND_BY_UNAME = 'TEXT_NEXUS_PATH')
	    WIDGET_CONTROL,id, Set_value = nexus_path
	    id = widget_info(Widget_id, FIND_BY_UNAME = 'TEXT_NEXUS_FNAME')
	    WIDGET_CONTROL,id, Set_value = nexus_lastfile
	    ; read file and update type
	    widget_control, /hourglass
	    nexus_head = read_nexus(nexus_path + nexus_lastfile, /header_only)
	    ; update the type
	    id = widget_info(Widget_id, FIND_BY_UNAME = 'TEXT_NEXUS_TYPE')
	    WIDGET_CONTROL,id, Sensitive=1, Set_value = nexus_head.Type
	    ;  ------------- update data channel information
	    unpack_Channel_labels = nexus_head.Channel_labels(0)
	    nchnl = n_elements(unpack_Channel_labels)
	    NEXUS_Channel_List = strarr(nchnl)
	    for i = 0, nchnl(0)-1 do NEXUS_Channel_List(i) = unpack_Channel_labels[i]
	    id = widget_info(Widget_id, FIND_BY_UNAME = 'WID_DLIST_CHANNEL')
	    WIDGET_CONTROL,id, Set_VALUE = NEXUS_Channel_List, sensitive = 1
	    ; ------------- update region information
	    ; currently there are no names associated with regions; can only set number of elements
	    n_regions = nexus_head.n_regions
	    if n_regions GT 1 then NEXUS_region_List = strarr(n_regions+1) else NEXUS_region_List = ' '
	    for i = 0, n_regions-1 do NEXUS_Region_List(i) = 'Region '+ strtrim(string(i+1),2)
	    if n_regions GT 1 then NEXUS_Region_List(n_regions) = 'all regions'
	    id = widget_info(Widget_id, FIND_BY_UNAME = 'WID_DLIST_REGION')
	    WIDGET_CONTROL,id, Set_VALUE = NEXUS_Region_List, Sensitive=1

	    ; ---------- update image number information (for NEXAFS image sequences)
	    if TOTAL(STRLOWCASE(nexus_head.Type[0]) EQ ['sample image stack','sample']) then begin
	    ;  (BW 160310) the '[0]' on the end of 'Type' is required for the EQ comparison to be done on the entire list!
		      NEXUS_Image_List = strtrim(string(nexus_head.StackAxis.Points, format='(F7.2)'),2)
		      id = widget_info(Widget_id, FIND_BY_UNAME = 'WID_DLIST_IMAGE')
		      widget_control, id, Set_Value=NEXUS_Image_List, sensitive = 1
		      id = widget_info(Widget_id, FIND_BY_UNAME = 'BUT_NEXUS_STACK_IMAGE')
		      widget_control, id, sensitive = 1
	    endif else begin
		      id = widget_info(Widget_id, FIND_BY_UNAME = 'WID_DLIST_IMAGE')
		      widget_control, id, sensitive = 0
		      id = widget_info(Widget_id, FIND_BY_UNAME = 'BUT_NEXUS_STACK_IMAGE')
		      widget_control, id, sensitive = 0
	    endelse
	   ENDIF ELSE BEGIN
	   		PRINT, 'Last NeXus filename is not valid.'
	   		RETURN
	   	ENDELSE
	ENDIF
; ENDIF

; ---------- update display of images & other 2D data types
print, 'STARTING UPDATE OF IMAGE in AX_NEXUS '
nexus_head = read_nexus(nexus_path + nexus_lastfile, /header_only)
if TOTAL(STRLOWCASE(nexus_head.Type[0]) EQ ['sample image stack','sample']) then begin	 ; find number of stack image selected
	id = widget_info(Widget_id, FIND_BY_UNAME = 'WID_DLIST_IMAGE')
	nexus_image_index = WIDGET_INFO(id, /Droplist_select)		; get the index
	WIDGET_CONTROL,id, Sensitive=1, Get_value = nexus_image		; get the list of energies
	; get the selected image
	 print, 'Extracting single image at E = ', nexus_image(nexus_image_index)
	nexus_image_index = fix(nexus_image_index) +1	; kluge to avoid problem that IDL uses 0 as false !
endif
; get the data channel
id = widget_info(Widget_id, FIND_BY_UNAME = 'WID_DLIST_CHANNEL')
nexus_ch = fix(WIDGET_INFO(id, /Droplist_select)) + 1
print, 'nexus_channel = ', nexus_ch
; get the region
id = widget_info(Widget_id, FIND_BY_UNAME = 'WID_DLIST_REGION')
WIDGET_CONTROL,id, Sensitive=1, Get_value = nexus_region_list
n_regions = n_elements(nexus_region_list)
id = widget_info(Widget_id, FIND_BY_UNAME = 'WID_DLIST_REGION')
nexus_region = fix(WIDGET_INFO(id, /Droplist_select)) + 1
print, 'nexus region = ', nexus_region_list(nexus_region - 1)
if nexus_region_list(nexus_region - 1) EQ 'all regions' then nexus_region = 1
widget_control, /hourglass
tmp = read_nexus(nexus_path + nexus_lastfile, channel= nexus_ch, region = nexus_region, one_image = nexus_image_index, /no_save)
wset, NexusImg
tmp.dl = ''
make_square,tmp
if tmp.t EQ '2d' then splot2d, tmp
if tmp.t EQ '1d' then plot, tmp.x, tmp.d

; -------- update parameter information
t = h5_parse(nexus_path + nexus_lastfile, /read_data)
q =strarr(3)
et = t.entry1.end_time._data      ; '2016-03-25T20:02:51+01:00'  ; ;
et=strmid(et,strpos(et,'T')+1)
; print, 'et:  ', et
q(0)=strmid(et,0,2)
q(1)=strmid(et,3,2)
q(2)=strmid(et,6,2)
et = fix(q(0))*3600+fix(q(1))*60+fix(q(2))
end_time = q(0) + ':' + q(1)+ ':' +q(2)
st = t.entry1.start_time._data  ; '2016-03-25T19:48:35+01:00' ;
st=strmid(st,strpos(st,'T')+1)
; print, 'st:  ', st
q(0)=strmid(st,0,2)
q(1)=strmid(st,3,2)
q(2)=strmid(st,6,2)
st = fix(q(0))*3600+fix(q(1))*60+fix(q(2))
start_time = q(0) + ':' + q(1)+ ':' +q(2)
elapsed_time = fix(et) - fix(st)
print, 'end = ', end_time, '   start ', start_time, '  elasped ', elapsed_time
id = widget_info(Widget_id, FIND_BY_UNAME = 'TEXT_NEXUS_START')
WIDGET_CONTROL,id, Set_value = ' ' + start_time
id = widget_info(Widget_id, FIND_BY_UNAME = 'TEXT_NEXUS_ELAPSED')
WIDGET_CONTROL,id, Set_value = ' ' + strtrim(string(elapsed_time),2)

energy = t.entry1.counter0.energy._data
dwell = float(t.entry1.counter0.count_time._data(0))		; dwell time in seconds
n_pixels = float(t.entry1.counter0.data._nelements(0))	; # of pixels
data_time = dwell*n_pixels
efficiency = 100.*data_time/elapsed_time
print, 'data taking time = ', data_time, '  efficiency = ', efficiency
id = widget_info(Widget_id, FIND_BY_UNAME = 'TEXT_NEXUS_DATA_TIME')
WIDGET_CONTROL,id, Set_value = ' ' + strtrim(string(data_time, format='(I5)'),2)
id = widget_info(Widget_id, FIND_BY_UNAME = 'TEXT_NEXUS_EFFICIENCY')
WIDGET_CONTROL,id, Set_value = ' ' + strtrim(string(efficiency, format='(F4.1)'),2)
id = widget_info(Widget_id, FIND_BY_UNAME = 'TEXT_NEXUS_ENERGY')
WIDGET_CONTROL,id, Set_value = ' ' + strtrim(string(energy, format='(F6.2)'),2)
id = widget_info(Widget_id, FIND_BY_UNAME = 'TEXT_NEXUS_DWELL')
WIDGET_CONTROL,id, Set_value = ' ' + strtrim(string(1000*dwell, format='(F5.2)'),2)
polarization = t.entry1.collection.polarization.value._data
id = widget_info(Widget_id, FIND_BY_UNAME = 'TEXT_NEXUS_POLARIZATION')
WIDGET_CONTROL,id, Set_value = ' ' + strtrim(string(polarization, format='(I3)'),2)


END

pro BASE_NEXUS_event, Event
@axis_com
@nexus_com

  case Event.id of

    Widget_Info(Event.Top, FIND_BY_UNAME='TEXT_NEXUS_FNAME'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH' )then $
        TEXT_NEXUS_FNAME, Event
    end
      Widget_Info(Event.Top, FIND_BY_UNAME='WID_DLIST_CHANNEL'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then $
        WID_DLIST_CHANNEL, Event
    end
      Widget_Info(Event.Top, FIND_BY_UNAME='WID_DLIST_REGION'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then $
        WID_DLIST_REGION, Event
    end
      Widget_Info(Event.Top, FIND_BY_UNAME='WID_DLIST_IMAGE'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_STR' )then $
        WID_DLIST_IMAGE, Event
    end
    Widget_Info(Event.Top, FIND_BY_UNAME='BUT_NEXUS_FNAME_BROWSE'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_NEXUS_FNAME_BROWSE, Event
    end
    Widget_Info(Event.Top, FIND_BY_UNAME='TEXT_NEXUS_TYPE_TITLE_0'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_DEL' )then $
        BUT_NEXUS_FNAME_BROWSE, Event
    end
    Widget_Info(Event.Top, FIND_BY_UNAME='BUT_NEXUS_OK'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_NEXUS_OK, Event
    end
    Widget_Info(Event.Top, FIND_BY_UNAME='BUT_NEXUS_CANCEL'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_NEXUS_CANCEL, Event
    end
    Widget_Info(Event.Top, FIND_BY_UNAME='BUT_NEXUS_STACK_IMAGE'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_NEXUS_Stack_IMAGE, Event
    end
    Widget_Info(Event.Top, FIND_BY_UNAME='BUT_NEXUS_NORMALIZE'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_NEXUS_NORMALIZE, Event
    end
    Widget_Info(Event.Top, FIND_BY_UNAME='BUT_NEXUS_XY_CORRECT'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_NEXUS_XY_CORRECT, Event
    end
    Widget_Info(Event.Top, FIND_BY_UNAME='BUT_NEXUS_PARAMETERS'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_NEXUS_PARAMETERS, Event
    end
    else: print, 'Undefined action'
  endcase

end

;-----------------------------------------------------------------
; ****************************************************************
;-----------------------------------------------------------------


pro BASE_NEXUS, GROUP_LEADER=wGroup,  file=file, _EXTRA=_VWBExtra_
@axis_com
@nexus_com

t = size(wGroup)
IF t(1) EQ 0 THEN modal_val=0 else modal_val=1

BASE_NEXUS = Widget_Base( GROUP_LEADER=wGroup, UNAME='BASE_NEXUS', TITLE='Read NeXus HDF5 files', modal = modal_val, /column)
row1 = widget_base(base_nexus,/row)
col1 = widget_base(row1,/column)
row = widget_base(base_nexus,/row)

; --------- display window
top_row = widget_base(base_nexus,/row)
col1 = widget_base(top_row,/col)
col2 = widget_base(top_row,/col)
row = widget_base(col1,/row,/align_right)
DRAW_NEXUS = widget_draw(row, retain = 2, uvalue = 'NeXus display', xsize = 180, ysize = 180) ;, /align_centre)

; row = widget_base(col1,/row,/align_center)
; tlabel = widget_label(row,value=' ')
; BUT_NEXUS_PARAMETERS = Widget_Button(row, UNAME='BUT_NEXUS_PARAMETERS', /ALIGN_CENTER ,VALUE='Parameters')

row = widget_base(col2,/row,/align_left)
tlabel = widget_label(row,value='Path ')
TEXT_NEXUS_PATH = Widget_Text(row, UNAME='TEXT_NEXUS_PATH', /FRAME,  XSIZE=50)

row = widget_base(col2,/row)
tlabel = widget_label(row,value='File ')
TEXT_NEXUS_FNAME = Widget_Text(row, UNAME='TEXT_NEXUS_FNAME', /FRAME,/EDITABLE ,ALL_EVENTS = 1 , KBRD_FOCUS_EVENTS = 0, XSIZE=50)
tlabel = widget_label(row,value=' ')
BUT_NEXUS_FNAME_BROWSE = Widget_Button(row, UNAME='BUT_NEXUS_FNAME_BROWSE', /ALIGN_CENTER ,VALUE='Browse')

row = widget_base(col2,/row)
tlabel = widget_label(row,value='Type ')
TEXT_NEXUS_TYPE = Widget_Text(row, UNAME='TEXT_NEXUS_TYPE', EDITABLE=0 ,/ALL_EVENTS ,VALUE='Type', /ALIGN_CENTER , XSIZE=30)
tlabel = widget_label(row,value='   ')
tlabel = widget_label(row,value='   ')
BUT_NEXUS_Stack_Image = Widget_Button(row, UNAME='BUT_NEXUS_STACK_IMAGE',  VALUE='1 image')
tlabel = widget_label(row,value='   ')
BUT_NEXUS_CANCEL = Widget_Button(row, UNAME='BUT_NEXUS_CANCEL', /ALIGN_CENTER ,VALUE='Cancel')
tlabel = widget_label(row,value='       ')
BUT_NEXUS_OK = Widget_Button(row, UNAME='BUT_NEXUS_OK',  /ALIGN_CENTER, VALUE='  OK  ')

row = widget_base(col2,/row)

WID_BASE_1 = Widget_Base(row, UNAME='WID_BASE_1',COLUMN=2 ,/NONEXCLUSIVE)

BUT_NEXUS_NORMALIZE = Widget_Button(wid_base_1, UNAME='BUT_NEXUS_NORMALIZE', /ALIGN_LEFT, VALUE='I-ring norm?')

BUT_NEXUS_XY_CORRECT = Widget_Button(wid_base_1, UNAME='BUT_NEXUS_XY_CORRECT', /ALIGN_LEFT, VALUE='xy correct ?')

NEXUS_Channel_List =['PMT','OSA','Analog']
NEXUS_Ch = 0
WID_DLIST_CHANNEL = Widget_Droplist(row, /DYNAMIC_RESIZE, VALUE = NEXUS_Channel_List, UNAME='WID_DLIST_CHANNEL', TITLE='Channel')

NEXUS_Region_List =['region 1']
NEXUS_region = 0
WID_DLIST_REGION = Widget_Droplist(row, /DYNAMIC_RESIZE, VALUE = NEXUS_Region_List, $
                   event_pro='WID_DLIST_REGION', UNAME='WID_DLIST_REGION', TITLE='Region')

NEXUS_Image_List =[' ']
NEXUS_Im = 0
WID_DLIST_IMAGE = Widget_Droplist(row, /DYNAMIC_RESIZE, VALUE = NEXUS_Image_List, UNAME='WID_DLIST_IMAGE', TITLE='Image #')

; ------ parameter display
row = widget_base(col2,/row)
tlabel = widget_label(row,value='Start time')
TEXT_NEXUS_start = Widget_Text(row, UNAME='TEXT_NEXUS_START', frame=0, xsize=8)
tlabel = widget_label(row,value='  Elapsed time (s)')
TEXT_NEXUS_elapsed = Widget_Text(row, UNAME='TEXT_NEXUS_ELAPSED', xsize=6)
tlabel = widget_label(row,value='  Data time (s)')
TEXT_NEXUS_data_time = Widget_Text(row, UNAME='TEXT_NEXUS_DATA_TIME', xsize=6)
tlabel = widget_label(row,value='  Efficiency (%)')
TEXT_NEXUS_efficiency = Widget_Text(row, UNAME='TEXT_NEXUS_EFFICIENCY', xsize=6)

row = widget_base(col2,/row)
tlabel = widget_label(row,value='Energy (eV)')
TEXT_NEXUS_energy = Widget_Text(row, UNAME='TEXT_NEXUS_ENERGY', frame=0, xsize=8)
tlabel = widget_label(row,value='  Dwell time (ms)')
TEXT_NEXUS_dwell = Widget_Text(row, UNAME='TEXT_NEXUS_DWELL', frame=0, xsize=8)
tlabel = widget_label(row,value='  Polarization')
TEXT_NEXUS_dwell = Widget_Text(row, UNAME='TEXT_NEXUS_POLARIZATION', frame=0, xsize=8)


; ----------- start display without active displays / controls
widget_control, WID_DLIST_CHANNEL, sensitive = 0
widget_control, WID_DLIST_REGION, sensitive = 0
widget_control, WID_DLIST_IMAGE, sensitive = 0
widget_control, TEXT_NEXUS_TYPE, sensitive = 0
widget_control, BUT_NEXUS_Stack_Image, sensitive = 0
; GTK crash with RedHat6 on next line if NEXUS_Image_List is a zero length string
Widget_Control, /REALIZE, BASE_NEXUS

; ----------- get info needed to access the draw_window
widget_control, DRAW_NEXUS, GET_value=NexusImg

; ----------- set to current value of normalization flag
if norm_flag EQ 1 then widget_control, BUT_NEXUS_Normalize, set_button = 1 $
  else widget_control, BUT_NEXUS_Normalize, set_button = 0

; ----------- set to current value of xy_correct_flag
if xy_correct_flag EQ 1 then widget_control, BUT_NEXUS_xy_correct, set_button = 1 $
  else widget_control, BUT_NEXUS_xy_correct, set_button = 0

; ------ set last path and file name if they exist
; ------ use keyword file if provided
if keyword_set(file) then begin		; assume file contains path
	t = ax_name(file)
	nexus_path = t(0)
	nexus_lastfile = t(1)
endif

BASE_NEXUS_UPDATE, BASE_NEXUS, /new_file

XManager, 'BASE_NEXUS', BASE_NEXUS
NEXUS_ID = BASE_NEXUS

end

;-----------------------------------------------------------------
; ****************************************************************
;-----------------------------------------------------------------

function ax_nexus, file = file, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_
on_error,2
@axis_com
@nexus_com

nexus = ' '
if keyword_set(file) then begin
	BASE_NEXUS, GROUP_LEADER=wGroup, file=file, _EXTRA=_VWBExtra_
endif else begin
	BASE_NEXUS, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_
endelse

return, nexus
end
