; Copyright (c) 1998-2008 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		READ_BESSY_STXM.PRO
;
;LAST CHANGED: ----------------------------------- 	11-Oct-08
;
; PURPOSE:
;	This function reads a single image from Bessy STXM (Thieme) and applies
; positional and size information to the displayed image
;
; CATEGORY:
;	data read. can execute outside of aXis2000
;
; CALLING SEQUENCE:
;	Result = READ_BESSY_STXM(file=file, filter=filter, normalize=normalize, store=store, silent=silent)
;
; INPUTS:
;
; KEYWORDS:
;	FILE:  name of file (*.pgm)
;	FILTER: extension (default is '*.pgm')
;	NORMALIZE: if set, modify intensities to standard ring current (value of normalize parameter
;	STORE: if set, write *.axb (aXis2000 format) binary file of image
;	SILENT: if set, do not print tracking comments
;
; OUTPUTS:
;	No explicit outputs.
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;
; SIDE EFFECTS: none
;
; RESTRICTIONS: none
;
; PROCEDURE:
;adapted from crate_nc.pro supplied by Jurgen Thieme
;
;
; MODIFICATION HISTORY:
; (11-Oct-08 aph) first version
;-

function read_bessy_stxm, file=file, filter=filter, normalize=normalize, store=store, silent=silent
@axis_com
on_error,2

if n_elements(file) eq 0 then begin  ;popup file dialog box
   if not keyword_set(filter) then filter='*.pgm'
   file=PICKFILE2(/Read, FILTER=filter, /LPATH, DEFPATH=defpath)
endif

if strlen(file) GT 0 THEN BEGIN
	name_bits = ax_name(file)	; sort the name into its parts
	path = name_bits(0)
	fileshort = name_bits(1)
    ext = name_bits(2)
 endif else begin
	print, 'Filename ', file,' is not defined, or does not exist'
endelse

if not keyword_set(silent) then print, 'Reading Bessy STXM file ', file

; ====================================================
;Reading the PGM files
openr, unit, file, /get_lun
a=''
readf, unit, a
if not keyword_set(silent) then axis_log, 'header: ' + a
readf, unit, n_cols, n_rows
n_rows = fix(n_rows-2)
n_cols = fix(n_cols)
; print, 'image size (rows, cols) = ', n_rows, n_cols
readf, unit, b
; print, '3rd row ', b
b=fltarr(100)
readf, unit, b		; remove set of zeros at first line
; print, '4th row ', b
image_data=fltarr(n_cols, n_rows)
for i=0,n_rows-1 do begin		; drop last 2 lines - seems to be extra blank lines top / bottom
	readf, unit, b
	image_data(*,i) = b
;	print, i, image_data(0,i)
endfor
close,unit & free_lun, unit

;Reading the XML header file
xml_file=path+fileshort+'info.xml'
imagepar=read_bessy_xml(file=xml_file)

; -----------------  generate an axis2000 format structure from header & image
x_step = (imagepar.x_stop - imagepar.x_start)/float(n_cols)
y_step = (imagepar.y_stop - imagepar.y_start)/float(n_rows)
x = findgen(n_cols)  & y = findgen(n_rows)
x = imagepar.x_start + x*x_step
y = imagepar.y_start + y*y_step
xl = string(FORMAT='("x (um)     E = ",f8.3," eV     dwell = ",f5.2," ms")', $
      imagepar.energy,  imagepar.dwell)

; -------------- normalize to standard ring current, if selected
if keyword_Set(normalize) then begin
	image_data=image_data*normalize/imagepar.ring_current
	yl = 'Bessy STXM   N-' + string(normalize, format='(F5.1)') + ' y (um)'
endif else begin
    yl = 'Bessy STXM   i-ring= '+ string(imagepar.ring_current, format='(F5.1)')+ ' y (um)'
endelse

s={t:'2d', d:image_data, x:x, y:y, e: imagepar.energy, xl:xl, yl: yl, dl:fileshort}

if not keyword_set(silent) then axis_log, 'read Bessy stxm file ' + fileshort

if keyword_set(store) then begin
	t=axb_save(s)
endif
return, s
end