; Copyright (c) 1998-2012 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		H5_READ_PTYCHO
;
;LAST CHANGED: ----------------------------------- 	02-Dec-12
;
; PURPOSE:
;	This function reads content of *.pt  pytography results file
; using the IDL H5_browser
;
; CATEGORY:
;	data read-in; operates stand alone or in aXi2000
;
; CALLING SEQUENCE:
;	Result = H5_REA_PTYCHO(file=file,  energy = energy, dwell = dwell, pix_size = pix_size)
;
; INPUTS: keyword only
;
; KEYWORDS:
;	FILE	name of file
;	ENERGY 	energy of image
;	DWELL	dwell time of measurement
;	PIX_SIZE	pixel size of ptycograph
;
; OUTPUTS:
;	Returns an aXis format image (for the extracted STXM file)
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks

; MODIFICATION HISTORY:
; (28-nov-12 aph) first version as a script
; (02-Dec-12 aph) AXIS standard header added
;-

function h5_read_ptycho, file = file, energy = energy, dwell = dwell, pix_size = pix_size
@axis_com
on_error,2

; determine if AXIS is running (therefore may have called ax_cgo)
; either when AXIS keyword is supplied or if any widget active
if  keyword_set(axis) then axis_on = 1 else axis_on = widget_info(/active)

IF NOT keyword_set(file) then begin
	fltr='*.pt'
	file = pickfile2(/READ, FILTER=fltr, /LPATH, DEFPATH=defpath)
ENDIF
s = ''
if file EQ '' THEN RETURN, s  ; bail-out if no filename
check = findfile(file)		 ; bail-out if non-existent file
if strlen(check(0)) EQ 0 then begin
	axis_log, 'Cannot find ' + file
	return, s
endif

; ----- use h5_browser to examine file and select item
a=h5_browser(file,/dialog_read)
d=rotate(float(a._data),7)
; help, d

; --------- get label from acquisition and construct short name
tmp=a._file
t = ax_name(file) & fileshort = t(1)

; ------------ determine photon energy
; Energy = 708.2
if NOT keyword_set(energy) then begin
	if axis_on then energy = get_num(Prompt = 'Energy (eV)', Val = energy, group = AXIS_ID) $
		else  energy = get_num(Prompt = 'Energy (eV)', Val = energy)
endif
if n_elements(ptycho_E) GT 0 then energy = ptycho_E
ptycho_E = energy	; save for later use

; ------------ determine dwell
; dwell = 1000
if NOT keyword_set(dwell) then begin
	if axis_on then dwell = get_num(Prompt = 'dwell (ms)', Val = dwell, group = AXIS_ID) $
		else  dwell = get_num(Prompt = 'dwell (ms)', Val = dwell)
endif
if n_elements(ptycho_dwell) GT 0 then dwell = ptycho_dwell
ptycho_dwell = dwell	; save for later use

; ------------ determine pixel size
; pix_size = 60
if NOT keyword_set(pix_size) then begin
	if axis_on then pix_size = get_num(Prompt = 'pix_size (nm)', Val = pix_size, group = AXIS_ID) $
		else  pix_size = get_num(Prompt = 'pix_size (nm)', Val = pix_size)
endif
if n_elements(ptycho_pix) GT 0 then pix_size = ptycho_pix
ptycho_pix = pix_size	; save for later use

t = size(d)
x=findgen(t(1))*pix_size
y=findgen(t(2))*pix_size
xl = string(dwell, energy, format=  '("dwell = ",f8.1," ms.  E = ",f8.2," eV")')
yl = 'ptycography from ' + fileshort
s = {t:'2d', x:x, y:y, d:d, e: energy, xl:xl, yl: yl, dl: tmp}
t=axb_save(s)

return, s
end

