; Copyright (c) 2013 -  A.P. Hitchcock, Florian Meirer, Dieter Ingerle  All rights reserved

;+
;NAME:
;		AX_READ_TXRM
;
;LAST CHANGED: ----------------------------------- 	29-Apr-2013
;
; PURPOSE:
;	This function uses an external call to a Java routine to read
;  Xradia *.txrm images, stored in Microsoft Structured Storage (COM) format
;
; CATEGORY:
;	Image  read in
;
; CALLING SEQUENCE:
;	Result = AX_READ_TXRM(, [FILE=FILE, NOF=NOF, SILENT=SILENT])
;
; INPUTS: none
;
; KEYWORDS:
;	FILE	filename (with path)
;   NOF	    if set, do not specify *.txrm as the extension
;   SILENT	if set, do not provide printed comments
;
; OUTPUTS:
;	image is returned as a aXis2000 IDL structure
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;
; SIDE EFFECTS: none
;
; RESTRICTIONS:
;	Will only work if appropriate IDL-Java software is included in the appropriate location
;
; PROCEDURE:
; The Java class txmreader creates a java list of java objects,
; one for each image stored in the txrm file
;
; EXAMPLE:
;
; MODIFICATION HISTORY:
; (29-Apr-13 aph) first version; AXIS standard header added
;-


FUNCTION ax_read_txrm, file=file, nof=nof, silent = silent
@axis_com
@stack_process_com
@bsif_com
on_error,2


;-------- set Path and environmental variable of a Windows machine to access the Java code
java_env=getenv('IDLJAVAB_CONFIG')
;help, java_env
; print, 'present java enviroment is ', java_env

old_path=getenv('Path')
tmp_path = old_path + 'C:\aXis2000\java\idljavab.jar;C:\aXis2000\java;C:\aXis2000\java\TxmImage.jar;C:\aXis2000\java\poi-3.9-20121203.jar'

setenv, 'Path='+tmp_path
setenv,'IDLJAVAB_CONFIG=C:\aXis2000\java\idljavabrc'

;java_env=getenv('IDLJAVAB_CONFIG')
;help, java_env

; ----- create the Java object --------
oJTxmImgFactory = OBJ_NEW("IDLJavaObject$TXMREADER_TXMIMAGEFACTORY", "txmreader.TxmImageFactory")


; --- get the filename if not supplied
IF NOT keyword_set(file) then begin
	fltr='*.txrm'
	if keyword_set(nof) then fltr='*'
	file = pickfile2(/READ, FILTER=fltr, /LPATH, DEFPATH=defpath)
ENDIF
s = ''
if file EQ '' THEN RETURN, s  ; bail-out if no filename
check = findfile(file)		 ; bail-out if non-existent file
if strlen(check(0)) EQ 0 then begin
	axis_log, 'Cannot find ' + file
	return, s
endif

; ---- generate short file name
t=ax_name(file)
fileshort = t(1)

; ---- read in *.txrm file
oJFullTXMData = oJTxmImgFactory -> getData(file)
;
; this object is an IDLJAVAOBJECT$JAVA_UTIL_ARRAYLIST
; for methods see: http://docs.oracle.com/javase/6/docs/api/java/util/ArrayList.html
; now get the number of images inside

NoOfImages = oJFullTXMData -> size()
; get data for all images: NOTE indexing starts from 0
;
; for ii=0,NoOfImages-1 do begin
ii=0
;    PRINT,  '--------'
    oJTxmImg = oJFullTXMData -> get(ii)
; this is now an IDLJavaObject$TXMREADER_TXMIMAGE

pix_size = oJTxmImg -> getPixelSize()
angle    = oJTxmImg -> getAngles()
AxisNames =  oJTxmImg -> getAxisNames()
Motor_Positions =  oJTxmImg -> getMotorPositions()
Scan_Size =  oJTxmImg -> getScanSize()
	; and so on ...


; -----  extract the image and other parameters
h = oJTxmImg -> getImageHeight()
w = oJTxmImg -> getImageWidth()
img = oJTxmImg -> getPixels()
energy = oJTxmImg -> getEnergy()
dwell = 1000.*oJTxmImg -> getExpTimes()
xrf_dwell = dwell   ; store dwell in this variable in axis_com common block for future use

xstart =  oJTxmImg -> getXPosition()
ystart =  oJTxmImg -> getYPosition()

Xpix_size = oJTxmImg -> getPixelSizeX()
Ypix_size = oJTxmImg -> getPixelSizeY()

x=findgen(w) & x =xstart + x*Xpix_size		; ARE TXRM images restricted to square pixels ?
y=findgen(h) & y =ystart + y*Ypix_size
; dwell = 10.

yl = 'TXRM image     y (um)'
xl = string(energy, dwell, format= $
	  '(" x"," um.      E =",f7.2," eV.       Dwell =",f5.2," ms.")')

s = {t:'2d', x:x, y:y, d:img, e: energy, xl:xl, yl:yl, dl: fileshort}

if not keyword_set(silent) then begin
	print, 'AX_TXRM_READ: read file from ', file
	axis_log, fileshort
	text = string(w, h, energy, dwell, format= $
  '(i4," x",i4," pixels. E =",f7.2," eV. Dwell =",f5.2," ms.")')
	axis_log, text
endif


; Cleanup the object.
OBJ_DESTROY, oJTxmImgFactory
OBJ_DESTROY, oJFullTXMData
OBJ_DESTROY, oJTxmImg

; ------ restore Path and java environmental variables
setenv, 'Path='+old_path
setenv, java_env

; ----- return the image
return, s

END