; Copyright (c) 1998-2013 A.P. Hitchcock  All rights reserved

; ************************** AXIS2000.PRO ***********************************
; start-up file for AXIS2000
; IDL Program Suite for Analysis of X-ray microscopy Image and Spectral data
; @Hitchcock Enterprises unlimited
; ******************************************* AXIS 2000 (2.1j) **************
;
; AXIS2000.PRO is the front end of the AXIS program suite
;	used to determine the program directory and initialization file.
;
;
;+
;NAME:
;	AXIS2000
;
;LAST CHANGED: -----------------------------------  27-May-2013 (aph)
;
;PURPOSE:
;	This procedure starts the aXis2000 widget.
;
;CATEGORY:
;	AXIS: utility
;
;CALLING SEQUENCE:  AXIS2000, directory=directory, start_file='*.ini', , help=help, /dialog
;
;INPUTS: none
;
;KEYWORDS:
;	directory	code directory
;	start_file	name of *.ini file
;	dialog		use a pick file dialog to select code directory and start file
;	spin		activate spinning logo
;	help		print a help text
;
; OPTIONS
;	directory	:	actual directory of Axis source code
;			:	if NOT implemented - if set equal 'dialog', then dialog_pickfile will be invoked to get directory
;	start_file	:	actual filename of parameter file *.ini
;			:	if NOT implemented & set equal to 'dialog', then dialog_pickfile will be invoked to get *.ini file
;	dialog	:	use dialog_pickfile to get both directory of Axis source code and *.ini file
;			:	if /dialog option is invoked, the values of directory and start_file will be used as initial values during dialog_pickfile

;OUTPUTS: starts execution of AXIS
;
;COMMON BLOCKS: axis_com
;
; PROCEDURE
;
;
;MODIFICATION HISTORY:
; (08-dec-00 cgz) first written
; (23-jan-01 cgz) Added initial specification of ini_file if start_file is not specified
; (23-jan-01 cgz) Added /help option to give help message
; (31-jan-01 aph) get so that will auto-run on PC from start_file call (redo of 12-jan-01 changes)
; (22-aug-01 aph) add spin keyword
; (28-jul-02 aph) set directory to 'CodePath' (from axis.ini file), if not defined in call
; (20-Apr-08 aph) explicitly force CodePath to 'c:\axis2000' if not exist
; (25-Jan-12 aph) adapt CodePath to deal with Mac versus Windows
; (27-May-13 aph) try to make code compatible with an arbitrary source path in Windows
;
;-
;

PRO aXis2000, directory=directory, start_file=start_file, dialog=dialog, spin=spin, help=help

on_error, 2
@axis_com

if n_elements(CodePath) LE 1 then print,'CodePath is undefined' $
      else print, 'CodePath is ', CodePath
 if n_elements(directory) LE 1 then print, 'Default directory is undefined' $
      else print, 'Default directory is ', directory

If n_elements(CodePath) EQ 0 then begin
    if !version.os_family EQ 'Windows' then begin
    	tmp = file_Search('axis_c.pro', /fully_qualify_path)
    	t = ax_name(tmp)
    	CodePath=t(0)
    	print, 'CodePath is ', CodePath
    	IF (n_elements(directory) EQ 0) THEN directory = CodePath
;    	CodePath='C:\aXis2000'			: changed this 27-May-2013 trying to get aXis2000 to work from any folder
    endif
    if !version.os_family EQ 'unix' then  CodePath='/Users/aph/aXis2000/'
endif

IF keyword_set(help) THEN BEGIN
    print,'axis2000,[directory=directory,start_file=start_file]'
    print,'  where [directory] is the directory of aXis2000 source code'
    print,'        [start_file] is the name of initialization file (*.ini) within [directory]'
    print,'  Options include:'
	print, '   /spin   : activate spinning logo'
    print,'    /dialog : to use dialog windows to specify directory of aXis2000 source code'
    print,'              and *.ini file'
    print,'    /help   : to print this help text'
    return
ENDIF

IF (n_elements(directory) EQ 0) THEN begin
	directory = CodePath
	print, ' setting default directory to ', CodePath
	cd, directory
ENDIF

IF (n_elements(start_file) EQ 0) THEN begin
	start_file = 'axis.ini'
	ini_file = 'axis.ini'
ENDIF
IF keyword_set(dialog) THEN do_dialog = 1 ELSE do_dialog = 0

ini_file = start_file	; APH's solution to bug of dec-00

IF (do_dialog EQ 1) THEN BEGIN
	; Get source code directory and *.ini file through dialog_pickfile interfaces
	IF (strtrim(directory,2) EQ 'dialog') THEN BEGIN
		cd,current=directory
	ENDIF ELSE BEGIN
		cd, directory
	ENDELSE
	codepath = directory

	directory__dialog_start :
;print,'1 codepath : ',codepath
	; Get directory of AXIS source code
	on_ioerror, dialog_directory_label_oops
	; print message to output log
	print,string(7B)+'Please select the directory with the aXis2000 source code'+string(7B)
	dummy = dialog_pickfile( /read, /must_exist, $
		path=codepath, get_path=codepath, /directory, $
		title='Select Directory of AXIS source code :')
        	; /directory option will display only directories, not individual files
	IF (dummy EQ '') THEN codepath = directory
		; sets data_directory to initial directory if CANCEL is chosen during dialog
		; without this, data_directory becomes null string

	dialog_directory_label_oops :
	cd,codepath	; change directory to codepath
	last_char = strmid(codepath,strlen(codepath)-1,1)

	CASE strupcase(!version.os_family) OF
		'VMS' : BEGIN
			IF (last_char NE ']') THEN BEGIN
				codepath = codepath+']'
			ENDIF
		END
		'WINDOWS' : BEGIN
			IF (last_char NE '\') THEN BEGIN
				codepath = codepath+'\'
			ENDIF
		END
		'MACOS' : BEGIN
			IF (last_char NE ':') THEN BEGIN
				codepath = codepath+':'
			ENDIF
		END
		'UNIX' : BEGIN
			IF (last_char NE '/') THEN BEGIN
				codepath = codepath+'/'
			ENDIF
		END
		ELSE : BEGIN
			print,'Unrecognized operating system'
		END
	ENDCASE
;print,'2 codepath : ',codepath

	; test to see if aXis2000 is in the directory
	dummy = findfile('axis2000.pro',count=count)
	IF (count EQ 0) THEN BEGIN
		dummy = dialog_message('This directory does not contain aXis2000 :'+ $
				string(13b)+string(13b)+codepath+string(13b)+string(13b)+ $
				'Please select the directory with the aXis200 source code'+ $
				string(13b)+string(13b)+string(13b)+string(13b)+'001', $
				/error, /cancel)
		IF (dummy EQ 'OK') THEN BEGIN
			GOTO, directory__dialog_start	; open dialog to get aXis2000 directory
		ENDIF ELSE BEGIN
			GOTO, axis_2000_abort			; abort aXis2000
		ENDELSE
	ENDIF

	directory__ini_start :
;print,'3 codepath : ',codepath

	on_ioerror, dialog_ini_file_label_oops
	dummy = dialog_pickfile(filter='*.ini', /read, /fix_filter, /must_exist, $
		path=codepath, title='Select *.ini file for AXIS :')
	ini_file = dummy

	; verify presence of ini file
	IF (dummy EQ '') THEN BEGIN
		dummy = dialog_message('No initialization file given.  '+ $
				string(13b)+string(13b)+ $
				'Please select an appropriate *.ini file.  '+ $
				string(13b)+string(13b)+string(13b)+string(13b)+'002', $
				/error, /cancel)
		IF (dummy EQ 'OK') THEN BEGIN
			GOTO, directory__ini_start		; open dialog to get ini file
		ENDIF ELSE BEGIN
			GOTO, axis_2000_abort			; abort aXis2000
		ENDELSE
	ENDIF

ENDIF ELSE BEGIN

	; Get source code directory and *.ini file through optional specs
	IF keyword_set(directory) THEN BEGIN
		print,'setting directory to '+directory
		cd,directory
		codepath = directory
	ENDIF ELSE BEGIN
		; this case structure is impractical since the location of AXIS
		; is so variable from installation to installation
		; It would be better to try to find AXIS and set the directory that way.
		CASE strupcase(!version.os_family) OF
			'VMS' : BEGIN
				directory = ''
			END
			'WINDOWS' : BEGIN
		print,'setting directory to c:/axis'
				directory = 'c:\axis'
			END
			'MACOS' : BEGIN
		print,'setting directory to Mac HD:AXIS:'
				directory = 'Mac HD:AXIS:'
			END
			'UNIX' : BEGIN
				directory = ''
			END
			ELSE : BEGIN
				print,'Unrecognized operating system'
				directory = ''
			END
		ENDCASE
		cd,directory
		codepath = directory
		; should develop this to operate on multiple platforms
		; may develop ability to search for presence of axis and then set directory
	ENDELSE

	; test to see if aXis2000 is in the directory
	dummy = findfile('aXis2000.pro',count=count)
	IF (count EQ 0) THEN BEGIN
		GOTO, directory__dialog_start	; open dialog to get aXis2000 directory
;		dummy = dialog_message('This directory does not contain aXis2000.pro :'+ $
;				string(13b)+string(13b)+codepath+string(13b)+string(13b)+ $
;				'Please select the directory with the aXis200 source code'+ $
;				string(13b)+string(13b)+string(13b)+string(13b)+'003', $
;				/error, /cancel)
;		IF (dummy EQ 'OK') THEN BEGIN
;			GOTO, directory__dialog_start	; open dialog to get aXis2000 directory
;		ENDIF ELSE BEGIN
;			GOTO, axis_2000_abort			; abort aXis2000
;		ENDELSE
	ENDIF

	; test to see if aXis2000 is in the directory
; bug here when axis2000 is invoked without options
	dummy = findfile(ini_file,count=count)
	IF (count EQ 0) THEN BEGIN
		dummy = dialog_message('The initialization file was not found :'+ $
				string(13b)+string(13b)+'     '+ini_file+string(13b)+string(13b)+ $
				'Please select an appropriate *.ini file.  '+ $
				string(13b)+string(13b)+string(13b)+string(13b)+'004', $
				/error, /cancel)
		IF (dummy EQ 'OK') THEN BEGIN
			GOTO, directory__ini_start		; open dialog to get ini file
		ENDIF ELSE BEGIN
			GOTO, axis_2000_abort			; abort aXis2000
		ENDELSE
	ENDIF


ENDELSE

;extract ini filename from current ini_file which presently includes the path
CASE strupcase(!version.os_family) OF
	'VMS' : BEGIN
		firstpos = strpos(ini_file, ']')+1
	END
	'WINDOWS' : BEGIN
		firstpos = rstrpos(ini_file, '\')+1
	END
	'MACOS' : BEGIN
		firstpos = rstrpos(ini_file, ':')+1
	END
	'UNIX' : BEGIN
		firstpos = rstrpos(ini_file, '/')+1
	END
	ELSE : BEGIN
		print,'Unrecognized operating system'
	END
ENDCASE

IF (firstpos EQ -1) THEN BEGIN
	ini_file = ini_file
ENDIF ELSE BEGIN
	ini_file = strmid(ini_file, firstpos, (strlen(ini_file)-firstpos))
ENDELSE

IF (strupcase(!version.os_family) EQ 'VMS') THEN BEGIN
	semipos = rstrpos(ini_file, ';')
	IF (semipos GT 1) THEN BEGIN
		ini_file = strmid(ini_file,0,semipos)
	ENDIF
ENDIF
ini_file = strtrim(ini_file,2)
; end of extraction

dialog_ini_file_label_oops :

; print,'codepath : ',codepath
; print,'ini_file : ',codepath+ini_file

if keyword_set(spin) then axis_c, start_file = ini_file, /spin $
  else axis_c, start_file = ini_file

axis_2000_abort :

return
END