; Copyright (c) 1998-2007 Andreas Scholl (LBNL); A.P. Hitchcock  All rights reserved
;+
;NAME:
;		AX_CIMAGE.PRO
;
;LAST CHANGED: ----------------------------------- 	12-Sep-07 (aph)
;
; PURPOSE:
;	This procedure cleans up images by removing very low and very high values.
; These appear in some CCD camera outputs due to dead or easily saturated pixels
;
; CATEGORY:
;	Image processing
;
; CALLING SEQUENCE:
; 	RESULT = AX_CIMAGE(IMG_ARR)
;
; INPUTS:
;	IMG_ARR  - 2-d array (integer, double or real)
;
; KEYWORDS: none
;
; OUTPUTS:
;	Modified Image
;
; COMMON BLOCKS: none
;
; SIDE EFFECTS: none
;
; RESTRICTIONS: does not work if low contrast, Error trapped if flat image
;
; PROCEDURE:
;	This procedure calculates a histogram and determine two levels (Top and Bottom)
; such that 2 rows worth of pixels are above and two rows worth of
; pixels are below these levels. The image is then clipped between Top and Bottom
;
; EXAMPLE:
;	a=RD_PEEM()		; read in test ALS PEEM2 file (pass_test_data\ALS_PEEM2_full_0609_#020.tif)
;	splot2d, a		; display to show out-of-limits aspects of image
;	q=AX_CIMAGE(a)	; process - works for structures and images
;	splot2d, q		; display to show correction of out-of-limits aspects
;
; MODIFICATION HISTORY:
; (22-Jan-07 as) Andreas Scholl provided routine as a solution to problem of arbitary removal
;         of first 5 columns (causes dimensional problems); NB his code is object oriented IDL5.5;
;		  translated back to IDL4.2 compatible code
; (12-Sep-07 aph) incorporated into aXis2000; AXIS standard header added
;-


FUNCTION ax_CIMAGE, img_arr

on_error,2

; check if img_arr is a structure
; if so, assume it is axis format so image is imag_arr.d and process that quantity

structure_flag=0
if size(img_arr,/type) EQ 8 then begin
	structure_flag=1
	tmp_struct=img_arr
	img_arr = tmp_struct.d
	axis_log, 'AX_CIMAGE: structure being clipped '
endif

CATCH, ErrorStatus
if ErrorStatus ne 0 then begin
        axis_log, !ERROR_STATE.MSG
		axis_log, 'Min = ' + string(strtrim(arr_Minimum,2))
        axis_log, 'Max = ' + string(strtrim(arr_Maximum,2))
        return, img_arr
endif

arr_Maximum = MAX(img_arr, MIN = mini, /NAN)
arr_Minimum = mini
nbins = UINT(arr_Maximum-arr_Minimum<1024)
t=size(img_arr)
arr_PixType = t(3)
arr_nx= t(1)
arr_ny=t(2)
binsize = FIX((arr_Maximum-arr_Minimum)/nbins, TYPE = arr_PixType)
hist = HISTOGRAM(img_arr, LOCATIONS = locs, MAX = arr_Maximum, MIN = arr_Minimum, BINSIZE = binsize)
lim = ULONG(SQRT(FLOAT(arr_nx)*FLOAT(arr_ny)))*2
if lim eq 0L then lim = 1L
s = 0L
ind = 0
while (s lt lim) and (ind lt N_ELEMENTS(locs)) do begin
        s = s + hist[ind]
        ind = ind+1
endwhile
if ind ge N_ELEMENTS(locs) then ind = ind-1
arr_Bottom = locs[ind]
s = 0L
ind = (nbins>1)-1
while (s lt lim) and (ind ge 0) do begin
        s = s + hist[ind]
        ind = ind-1
endwhile
arr_Top = locs[ind+1]
if arr_Bottom ge arr_Top then arr_Top += 1

; replace all values below arr_bottom with arr_bottom & all values above arr_top with arr_top
img_arr(where(img_arr GT arr_Top))=arr_Top
img_arr(where(img_arr LT arr_Bottom))=arr_Bottom

if structure_flag EQ 1 then begin
	tmp_struct.d = img_arr
	img_arr = tmp_struct
endif

return, img_arr
end