;+ 
;NAME: 
;	CRYO_TO_STXM 
; 
;LAST CHANGED: ----------------------------------- 23-feb-00 
; 
;PURPOSE: 
;	This procedure converts one or more files in cryo-stxm format (*.sxm) 
; to the old X1A STXM format (*.nc). 
; 
;CATEGORY: 
;	STAND ALONE: file utility 
; 
;CALLING SEQUENCE: 
;	CRYO_TO_STXM, find_file_string 
; 
;CALLED FROM AXIS: 
;	->Read->Images->NSLS->cryo 
; 
;ROUTINES 
;	if the file contains multiple PROs 
; list these here, along with a brief description of their function 
; 
;INPUTS: 
;	FINDFILE_STRING = list of files to convert, from findfile routine 
; 
;KEYWORDS: none 
; 
;OUTPUTS: 
;	The result is written to file with same main name and '.nc' extension 
; 
;COMMON BLOCKS: 
;	@bsif_common - commons for old stxm format 
;	@sxm_common - part of sxm common 
;	@sxm_acommon - remainder of sxm common 
; 
; 
;MODIFICATION HISTORY: 
; (xx-mmm-99 cjj) first written as part of sxm package 
; (xx-sep-99 aph) adapted from X1A new SXM code to provide ability to read cryo files into AXIS 
; (23-feb-00 aph) AXIS standard documentation 
;- 
 
PRO cryo_to_stxm,findfile_string 
 
@bsif_common
@sxm_common
@sxm_acommon
 
filelist = findfile(findfile_string) 
n_files = n_elements(filelist) 
 
init_stxm_sd,sd 
sxm_ipar 
 
for i_file=0,(n_files-1) do begin 
  sxm_filename = filelist(i_file) 
  i_end = strpos(sxm_filename,'.sxm') 
  stxm_filename = strmid(sxm_filename,0,i_end)+'.nc' 
  print,'Converting "'+sxm_filename+'" to "'+stxm_filename+'"' 
  read_sxm,sxm_filename 
 
  ;; This translates the scan mode 
  ;; sxm_par.scanmode: 'PZT, 'STEP', 'XFOC', 'YFOC', 'SPEC' 
  ;; sd.scan_driver_mode: PZT=1, Stepper=2, Focus=3, SGM=4 
  char1 = strupcase(strmid(string(sxm_par.scanmode),0,1)) 
  char2 = strupcase(strmid(string(sxm_par.scanmode),1,1)) 
  IF (char1 EQ 'P') THEN BEGIN 
    sd.scan_driver_mode = 1 
  ENDIF ELSE IF ((char1 EQ 'X') OR (char1 EQ 'Y')) THEN BEGIN 
    sd.scan_driver_mode = 3 
  ENDIF ELSE IF ((char1 EQ 'S') AND (char2 eq 'T')) THEN  BEGIN 
    sd.scan_driver_mode = 2 
  ENDIF ELSE IF ((char1 EQ 'S') AND (char2 EQ 'P')) THEN BEGIN 
    sd.scan_driver_mode = 4 
  ENDIF ELSE BEGIN 
    print,'Unknown scan mode!' 
    return 
  ENDELSE 
 
  ;; This translates most (but not all) of the scan parameters 
  sd.scan_width = abs(x_stop-x_start) 
  sd.scan_height = abs(y_stop-y_start) 
  sd.i0_mode = 0 
  sd.clock_frequency = 100.e3 
  sd.x_start = x_start 
  sd.x_stop = x_stop 
  sd.y_start = y_start 
  sd.y_stop = y_stop 
  sd.x_step_size = abs(x_stop-x_start)/float(n_cols-1) 
  IF (n_rows GT 1) THEN BEGIN 
    sd.y_step_size = abs(y_stop-y_start)/float(n_rows-1) 
  ENDIF ELSE BEGIN 
    sd.y_step_size = 0. 
  ENDELSE 
  sd.focus_position = sxm_par.z_center_um 
  sd.dwell_time = sxm_par.msec_dwell 
  sd.entrance_slit = sxm_par.entrance_slit_um 
  sd.exit_slit = sxm_par.exit_slit_um 
  sd.wavelength = 10.*sxm_par.now_nm 
  sd.date_time = byte(strmid(string(sxm_par.systime),0,24)) 
  sd.operator = byte(strmid(string(sxm_par.operator),0,80)) 
  sd.sample = byte(strmid(string(sxm_par.sample),0,80)) 
  sd.zone_plate = byte(strmid(string(sxm_par.zp_name),0,80)) 
  sd.comment1 = byte(strmid(string(sxm_par.comments1),0,80)) 
  sd.comment2 = byte(strmid(string(sxm_par.comments2),0,80)) 
  sd.file_name = byte(strmid(string(sxm_par.filename),0,256)) 
  sd.ring_current = sxm_par.ring_ma 
  sd.undulator_gap = sxm_par.undulator_mm 
  sd.x_center = sxm_par.x_center_um 
  sd.y_center = sxm_par.y_center_um 
  sd.z_center = sxm_par.z_center_um 
  sd.initial_lambda = 10.*sxm_par.start_nm 
  sd.final_lambda = 10.*sxm_par.end_nm 
 
  write_stxm,stxm_filename,sd 
endfor 
 
return 
end 
