; Copyright (c) 1998-2005 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	AX_LOX
;
;LAST CHANGED: ----------------------------------- 18-mar-06
;
;PURPOSE:
;	This set of procedures is a widget to read in files
; in the LOX format (*.lox) developed for CaPeRS PEEM.
; The header information uses XML syntax conventions.
; The widget uses read_lox to perform the file input.
;
;CATEGORY:
;	AXIS: utility
;
; AX_LOX_EVENT ROUTINES:
; TEXT_LOX_FNAME		modify filename; when changed, execute BUT_LOX_FNAME_BROWSE to get parameters
; BUT_LOX_FNAME_BROWSE 	browse for new filename; if found, load new parameters
; BUT_LOX_DISPLAY_HDR   show parameters
; BUT_LOX_OK			proceed with loading LOX file into AXIS
; BUT_LOX_CANCEL		abort read in
; WID_DLIST_CHANNEL		drop list used to display and select the data channel
; WID_DLIST_REGION		drop list to display and select region
;
;COMMON BLOCKS:
;	@AXIS_COM	standard set of common blocks
;	@LOX_COM 	structure with header info
;
;MODIFICATION HISTORY:
; (16-Oct-05 aph) first generated by adapting AX_SDF
; (13-nov-05 aph) stacks working, improve aspects
; (18-mar-06 aph) data channel selection fixed; all widget drop lists were coded wrong
;-

pro TEXT_LOX_FNAME, Event
on_error,2
@axis_com
@lox_com

; get the changed name
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_FNAME')
WIDGET_CONTROL,id, Sensitive=1, Get_value = fname
IF fname(0) EQ lox_lastfile then return
; generate full name with path
pfname = lox_path + fname

; -- check if the filename exists - if not assume this is just a trap out of
; -- the user starting to type a changed name or a mistake
t = findfile(pfname(0), count=t_count) & if t_count EQ 0 then return
;print, 'New filename is ', fname(0)
lox_lastfile = fname(0)

; update details
 BUT_LOX_FNAME_BROWSE, Event, file = fname(0)

end

;-----------------------------------------------------------------
pro BUT_LOX_FNAME_BROWSE, Event, file=file
on_error,2
@axis_com
@lox_com
; get new filename
if NOT keyword_set(file) then begin
	fname = pickfile2(Title='Select LOX file', Filter = '*.lox')
endif else fname = file
if strlen(fname) GT 0 then begin
	t = ax_name(fname)
	lox_path = t(0)
	lox_lastfile = t(1)          ; ensure the filename is saved for future use
; update lox_path and file_name in the text box
	id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_PATH')
	WIDGET_CONTROL,id, Set_value = lox_path
	id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_FNAME')
	WIDGET_CONTROL,id, Set_value = lox_lastfile
; read file and update type
	fname = lox_path + lox_lastfile + '.lox'
;	print, ' Reading in LOX file: ', fname
	widget_control, /hourglass
	lox = read_lox(file=fname, /type, group = lox_ID)
; update the type
	id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_TYPE')
	WIDGET_CONTROL,id, Sensitive=1, Set_value = lox.Type


;  ------------- update data channel information
;	nchnl = n_elements(lox.Channel_labels)
;	LOX_Channel_List = strarr(nchnl)
;	for i = 0, nchnl(0)-1 do LOX_Channel_List(i) = lox.Channel_labels[i].Name
;	id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_CHANNEL')
;	WIDGET_CONTROL,id, Set_VALUE = LOX_Channel_List, sensitive = 1

; ------------- update region information
; currently there are no names associated with regions; can only set number of elements
if lox.Type EQ 'stack' or lox.Type EQ 'spectra' then begin
	lox_data = read_lox_spectra(file = lox.filename, /multi, /silent, group = lox_ID)
	nt = size(lox_Data)
	lox.n_regions = nt(1)-1
	if keyword_set(verbose) then axis_log, '# of regions ', lox.n_regions
endif
n_regions = lox.n_regions
if n_regions GT 1 then LOX_region_List = strarr(n_regions+1) else LOX_region_List = ' '
for i = 0, n_regions-1 do LOX_Region_List(i) = 'Region '+ strtrim(string(i+1),2)
if n_regions GT 1 then LOX_Region_List(n_regions) = 'all regions'
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_REGION')
WIDGET_CONTROL,id, Set_VALUE = LOX_Region_List, Sensitive=1

; ---------- update image number information (for NEXAFS image sequences)
if lox.Type EQ 'stack' then begin
	tmp = read_lox_spectra(file = lox.filename, /silent, group = lox_ID)
	LOX_Image_List = strtrim(string(tmp.x, format='(F7.2)'),2)
	id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_IMAGE')
	widget_control, id, Set_Value=LOX_Image_List, sensitive = 1
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_LOX_STACK_MAP')
	widget_control, id, sensitive = 1
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_LOX_STACK_IMAGE')
	widget_control, id, sensitive = 1
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_LOX_STACK_SPECTRUM')
	widget_control, id, sensitive = 1
	id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_DARK')
	WIDGET_CONTROL,id, Sensitive=1, Set_value = string(lox.dark, format='(f5.1)')
	id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_ENERGY')
	widget_control, id, sensitive = 0
endif else begin
	id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_IMAGE')
	widget_control, id, sensitive = 0
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_LOX_STACK_MAP')
	widget_control, id, sensitive = 0
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_LOX_STACK_IMAGE')
	widget_control, id, sensitive = 0
	id = widget_info(Event.top, FIND_BY_UNAME = 'BUT_LOX_STACK_SPECTRUM')
	widget_control, id, sensitive = 0
		id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_ENERGY')
	WIDGET_CONTROL,id, Sensitive=0, Set_value = string(0., format='(g9.3)')
	id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_DARK')
	WIDGET_CONTROL,id, Sensitive=0, Set_value = string(0., format='(f5.1)')
endelse


if lox.Type EQ 'image' then begin
	id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_ENERGY')
	WIDGET_CONTROL,id, Sensitive=1, Set_value = string(lox.energy, format='(g9.3)')
	id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_DARK')
	WIDGET_CONTROL,id, Sensitive=1, Set_value = string(lox.dark, format='(f5.1)')
	id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_CHANNEL')
	widget_control, id, sensitive = 0
endif else begin
	id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_CHANNEL')
	widget_control, id, sensitive = 1
endelse

if lox.Type EQ 'spectra' OR lox.Type EQ 'stack' then begin
;	print, Lox.labels
	LOX_Channel_List = Lox.labels
	LOX_Channel_List = [LOX_Channel_List, 'all channels']
	id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_CHANNEL')
	widget_control, id, Set_Value=LOX_Channel_List, sensitive = 1
endif

; ----------- activate 'Show parameters' button
	id = widget_info(Event.top, FIND_BY_UNAME ='BUT_LOX_DISPLAY_HDR')
  widget_control, id, sensitive = 1

endif
end

pro WID_DLIST_CHANNEL, Event
on_error,2
@axis_com
@lox_com
; get the changed data channel
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_CHANNEL')
lox_ch = 1 + WIDGET_INFO(id, /Droplist_select)
if lox_ch GT lox.Nsignals then lox.channel = 0 else lox.channel = lox_ch(0)
; print, 'New channel selected: # = ', strtrim(string(lox_ch),2), '  name = ', LOX_Channel_List(lox_ch-1)
end

pro WID_DLIST_REGION, Event
on_error,2
@axis_com
@lox_com
; get the changed region
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_REGION')
lox_region = 1 + WIDGET_INFO(id, /Droplist_select)
;print, 'New region selected: # = ', strtrim(string(lox_region),2), $
;             '  name = ', LOX_Region_List(lox_region-1)
end

pro WID_DLIST_IMAGE, Event
on_error,2
@axis_com
@lox_com
; get the selected image
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_IMAGE')
lox_image = 1 + WIDGET_INFO(id, /Droplist_select)
;print, 'New image selected: # = ', strtrim(string(lox_image),2), $
;             '  energy = ', LOX_Image_List(lox_image-1)
end

;-----------------------------------------------------------------
pro BUT_LOX_DISPLAY_HDR, Event, group=wGroup
on_error,2
@axis_com
@lox_com
; get the path & filename
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_PATH')
WIDGET_CONTROL,id, Get_value = path
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_FNAME')
WIDGET_CONTROL,id, Get_value = fname
if strlen(fname(0)) GT 0 then begin
	fname = lox_path + fname(0) + '.lox'
	XDisplayfile,fname, done_button = 'dismiss', title = fname, group = wGroup, width=132
endif
end

;-----------------------------------------------------------------
pro BUT_LOX_STACK_MAP, Event
@axis_com
@lox_com
on_error,2

axis_log, 'Analysing stack as a 2-image map'
; ----------- get the path & filename ----------
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_PATH')
WIDGET_CONTROL,id, Get_value = path
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_FNAME')
WIDGET_CONTROL,id, Sensitive=1, Get_value = fname
fname = path + fname + '.lox'
if keyword_set(verbose) then axis_log, ' Reading in LOX file: ' + string(fname)
; -------------- get the data channel ------------------
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_CHANNEL')
this_channel = 1+ WIDGET_INFO(id, /Droplist_select)
; print, 'channel is ', this_channel
; ----------- get the region -------
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_REGION')
this_region = 1 + widget_info(id, /DROPLIST_SELECT)
; print, 'region is ', this_region

widget_control, /hourglass
lox = read_lox(file=fname(0), channel= this_channel, region = this_region, /map, group = lox_ID)

WIDGET_CONTROL, Event.Top, /DESTROY
end

;-----------------------------------------------------------------
pro BUT_LOX_STACK_IMAGE, Event
@axis_com
@lox_com
on_error,2

id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_IMAGE')
lox_image_index = WIDGET_INFO(id, /Droplist_select)
t = tag_names(lox)
test = 0
for i=0, n_elements(t) - 1 do begin
 	 if t(i) EQ 'energy' then test = 1
endfor
if test EQ 0 then  lox = create_struct(lox, 'energy',0.0)
; help, lox.image_list, lox_image_index
lox.energy = lox.image_list(lox_image_index)
axis_log, 'Extracting single stack image at E = ' + string(lox.energy, format='(f9.3)')
lox_image_index = fix(lox_image_index) +1	; kluge to avoid problem that IDL uses 0 as false !
; get the path & filename
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_PATH')
WIDGET_CONTROL,id, Get_value = path
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_FNAME')
WIDGET_CONTROL,id, Sensitive=1, Get_value = fname
fname = path + fname
; get the region
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_REGION')
this_region = 1 + widget_info(id, /DROPLIST_SELECT)
; if keyword_set(verbose) then print, 'region is ', this_region
widget_control, /hourglass
; prepare filename for this image
fname = fname + '.lox'
lox = read_lox(file=fname, channel= 1,region = this_region, one_image = lox_image_index, group = lox_ID)
WIDGET_CONTROL, Event.Top, /DESTROY
end

;-----------------------------------------------------------------
pro BUT_LOX_STACK_SPECTRUM, Event
@axis_com
@lox_com
on_error,2

; get the path & filename
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_PATH')
WIDGET_CONTROL,id, Get_value = path
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_FNAME')
WIDGET_CONTROL,id, Sensitive=1, Get_value = fname
fname = path + fname + '.lox'
; ----- get the data channel --------
id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_CHANNEL')
lox_ch = 1 + WIDGET_INFO(id, /Droplist_select)
if lox_ch GT lox.Nsignals then 	lox.channel = 0 else lox.channel = lox_ch(0)
;	print, 	'lox.channel, lox,Nsignals', lox.channel, lox.Nsignals
; get the region
; id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_REGION')
; this_region = 1 + widget_info(id, /DROPLIST_SELECT)
; print, 'region is ', this_region

lox.type = 'spectra'		; read in as a spectrum
if lox.channel EQ 0 then begin
; read the data for all channels
;		if keyword_set(verbose) then
		axis_log, 'Reading all channels '
		lox = read_lox(file=fname, channel=0, /multi, region = 1, group = lox_ID)
	endif else begin
; read the data file for a defined channel of a single region
;		if keyword_set(verbose) then
		axis_log, 'Reading channel ' + lox.labels(lox_ch)
		lox = read_lox(file=fname, channel=lox.channel, region = 1, group = lox_ID)
	endelse

WIDGET_CONTROL, Event.Top, /DESTROY
end


; ----------------------------------------------------------------
pro TEXT_LOX_ENERGY, Event
on_error,2
@axis_com
@lox_com
; get the changed value of the energy
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_ENERGY')
WIDGET_CONTROL,id, Sensitive=1, Get_value = dummy
Lox.energy = float(dummy)
; print, 'New Energy = ', lox.energy
WIDGET_CONTROL,id, Sensitive=1, Set_value = string(lox.energy, format='(f8.3)')
end

; ----------------------------------------------------------------
pro TEXT_LOX_DARK, Event
on_error,2
@axis_com
@lox_com
; get the changed value of the dark current (CaPeRS is typically 54.)
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_DARK')
WIDGET_CONTROL,id, Sensitive=1, Get_value = dummy
Lox.dark = Float(dummy)
; print, 'New Dark current value= ', lox.dark
WIDGET_CONTROL,id, Sensitive=1, Set_value = string(lox.dark, format='(f5.1)')
end

; ----------------------------------------------------------------
pro TEXT_LOX_FoV, Event
on_error,2
@axis_com
@lox_com
; get the changed value of the field of view
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_FoV')
WIDGET_CONTROL,id, Sensitive=1, Get_value = dummy
Lox.FoV = Float(dummy)
print, 'New Field of View = ', lox.fov
WIDGET_CONTROL,id, Sensitive=1, Set_value = string(lox.FoV, format='(f5.1)')
end

;-----------------------------------------------------------------
pro BUT_LOX_OK,  Event
on_error,2
@axis_com
@lox_com

; get the path & filename
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_PATH')
WIDGET_CONTROL,id, Get_value = path
id = widget_info(Event.top, FIND_BY_UNAME = 'TEXT_LOX_FNAME')
WIDGET_CONTROL,id, Sensitive=1, Get_value = fname
fname = path + fname + '.lox'
;if keyword_set(verbose) then
axis_log, ' Reading in LOX file: ' + string(fname) + ' of type ' + lox.type

if lox.type NE 'image' then begin
; -------- if not an image get the data channel
	id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_CHANNEL')
	lox_ch = 1 + WIDGET_INFO(id, /Droplist_select)
	if lox_ch GT lox.Nsignals then 	lox.channel = 0 else lox.channel = lox_ch(0)
;	print, 	'lox.channel, lox.Nsignals', lox.channel, lox.Nsignals
endif

if lox.type EQ 'spectra' then begin
	if lox.channel EQ 0 then begin
; read the data for all channels
;		if keyword_set(verbose) then
		axis_log, 'Reading all channels '
		lox = read_lox(file=fname(0), channel=0, /multi, region = 1, group = lox_ID)
	endif else begin
; read the data file for a defined channel of a single region
;		if keyword_set(verbose) then
		axis_log, 'Reading channel ' + lox_Channel_list(lox.channel-1)
		lox = read_lox(file=fname(0), channel=lox.channel, region = 1, group = lox_ID)
	endelse
endif else begin
; ----- stack or image  =>  get the region -----------------------
	n_regions = lox.n_regions
	; print, 'lox.n_regions at OK ', n_regions
	id = widget_info(Event.top, FIND_BY_UNAME = 'WID_DLIST_REGION')
	lox_region = WIDGET_INFO(id, /Droplist_select)
	region = lox_region(0) + 1
	lox_channel = 1
	if  region LE n_regions then begin
		if keyword_set(verbose) then axis_log, 'Reading region ' +  string(region,format='(i2)')
		lox = read_lox(file=fname(0), channel=lox.channel, region = region, group = lox_ID)
	endif else begin
		if keyword_set(verbose) then  axis_log,  'Reading all regions'
		lox = read_lox(file=fname(0), channel=lox.channel, /all_regions, group = lox_ID)
	endelse
endelse
; ************ seems like this is set to read only channel 1 !! **************************

WIDGET_CONTROL, Event.Top, /DESTROY
return
end

;-----------------------------------------------------------------

pro BUT_LOX_CANCEL, Event
@axis_com
@lox_com
on_error,2
lox = 0
WIDGET_CONTROL, Event.Top, /DESTROY
end


;-----------------------------------------------------------------
; ****************************************************************
;-----------------------------------------------------------------

pro BASE_LOX_event, Event
@axis_com
@lox_com

  wWidget =  Event.top
;  help, Event, /structure
;  print, Tag_Names(Event, /STRUCTURE_NAME)

  case Event.id of

    Widget_Info(wWidget, FIND_BY_UNAME='TEXT_LOX_FNAME'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH' )then $
        TEXT_LOX_FNAME, Event
    end
      Widget_Info(wWidget, FIND_BY_UNAME='WID_DLIST_CHANNEL'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_DROPLIST' )then $
        WID_DLIST_CHANNEL, Event
    end
      Widget_Info(wWidget, FIND_BY_UNAME='WID_DLIST_REGION'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_DROPLIST' )then $
        WID_DLIST_REGION, Event
    end
      Widget_Info(wWidget, FIND_BY_UNAME='WID_DLIST_IMAGE'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_DROPLIST' )then $
        WID_DLIST_IMAGE, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_LOX_FNAME_BROWSE'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_LOX_FNAME_BROWSE, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='TEXT_LOX_TYPE_TITLE_0'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_DEL' )then $
        BUT_LOX_FNAME_BROWSE, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_LOX_DISPLAY_HDR'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_LOX_DISPLAY_HDR, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_LOX_OK'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_LOX_OK, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_LOX_CANCEL'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_LOX_CANCEL, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_LOX_STACK_MAP'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_LOX_STACK_MAP, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_LOX_STACK_IMAGE'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_LOX_Stack_IMAGE, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='BUT_LOX_STACK_SPECTRUM'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        BUT_LOX_Stack_SPECTRUM, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='TEXT_LOX_ENERGY'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH' )then $
        TEXT_LOX_ENERGY, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='TEXT_LOX_DARK'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH' )then $
        TEXT_LOX_DARK, Event
    end
    Widget_Info(wWidget, FIND_BY_UNAME='TEXT_LOX_FoV'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH' )then $
        TEXT_LOX_FoV, Event
    end
    else: axis_log, 'Undefined action'
  endcase

end

;-----------------------------------------------------------------
; ****************************************************************
;-----------------------------------------------------------------


pro BASE_LOX, GROUP_LEADER=wGroup,  file=file, verbose=verbose,  _EXTRA=_VWBExtra_
@lox_com

;  Resolve_Routine, 'ax_lox_eventcb'     ; Load event callback routines
t = size(wGroup)
IF t(1) EQ 0 THEN modal_val=0 else modal_val=1

;print, wgroup, 'modal_val = ' , modal_val

BASE_LOX = Widget_Base( GROUP_LEADER=wGroup, UNAME='BASE_LOX',  $
      TITLE='Read Lox Format files', $
       modal = modal_val, /column)
row1 = widget_base(base_lox,/row)
col1 = widget_base(row1,/column)
row = widget_base(base_lox,/row)


LOX_ID = BASE_LOX
; print, 'Lox ID is ', lox_id

tlabel = widget_label(row,value='Path ')
TEXT_LOX_PATH = Widget_Text(row, UNAME='TEXT_LOX_PATH',  $
     /FRAME,  XSIZE=60)

row = widget_base(base_lox,/row)
tlabel = widget_label(row,value='File   ')
TEXT_LOX_FNAME = Widget_Text(row, UNAME='TEXT_LOX_FNAME',  $
      /FRAME,/EDITABLE ,ALL_EVENTS = 1 , KBRD_FOCUS_EVENTS = 0, XSIZE=30)
BUT_LOX_FNAME_BROWSE = Widget_Button(row,  $
      UNAME='BUT_LOX_FNAME_BROWSE', /ALIGN_CENTER ,VALUE='Browse')
tlabel = widget_label(row,value='')
BUT_LOX_DISPLAY_HDR = Widget_Button(row,  $
      UNAME='BUT_LOX_DISPLAY_HDR' ,  $    ;XOFFSET=320,
      /ALIGN_CENTER ,VALUE='Parameters')
tlabel = widget_label(row,value='  ')
BUT_LOX_CANCEL = Widget_Button(row, UNAME='BUT_LOX_CANCEL',  $
      /ALIGN_CENTER ,VALUE='Cancel')


row = widget_base(base_lox,/row)
tlabel = widget_label(row,value='Type ')
TEXT_LOX_TYPE = Widget_Text(row, UNAME='TEXT_LOX_TYPE',  $
      EDITABLE=0 ,/ALL_EVENTS ,VALUE=[ 'Type', '', '', '' ], $
      /ALIGN_CENTER , SCR_YSIZE=20, XSIZE=20)
tlabel = widget_label(row,value='')
BUT_LOX_Stack_Map = Widget_Button(row, $
       UNAME='BUT_LOX_STACK_MAP', VALUE='Map')
BUT_LOX_Stack_Image = Widget_Button(row, $
       UNAME='BUT_LOX_STACK_IMAGE',  VALUE='1 image')
BUT_LOX_Stack_Spectrum = Widget_Button(row, $
       UNAME='BUT_LOX_STACK_SPECTRUM',  VALUE='spectrum')
tlabel = widget_label(row,value='         ')
BUT_LOX_OK = Widget_Button(row, UNAME='BUT_LOX_OK',  $
      /ALIGN_CENTER, VALUE='    OK    ')

row = widget_base(base_lox,/row)
tlabel = widget_label(row,value='Energy (eV) ')
TEXT_LOX_ENERGY = Widget_Text(row, UNAME='TEXT_LOX_ENERGY',  $
      /FRAME, /EDITABLE , XSIZE=10)
tlabel = widget_label(row,value='Dark level ')
TEXT_LOX_DARK = Widget_Text(row, UNAME='TEXT_LOX_DARK',  $
      /FRAME, /EDITABLE , XSIZE=8)
tlabel = widget_label(row,value='Field of View ')
TEXT_LOX_FoV = Widget_Text(row, UNAME='TEXT_LOX_FoV',  $
      /FRAME, /EDITABLE , XSIZE=8)


row = widget_base(base_lox,/row)
  LOX_Channel_List =['mesh','PEEM']
  LOX_Ch = 0
  WID_DLIST_CHANNEL = Widget_Droplist(row,  $
      /DYNAMIC_RESIZE, VALUE = LOX_Channel_List, $
      UNAME='WID_DLIST_CHANNEL' ,XOFFSET=21, TITLE='Data Channel ')
  LOX_Region_List =['region 1']
  LOX_region = 0
  WID_DLIST_REGION = Widget_Droplist(row,  $
      /DYNAMIC_RESIZE, VALUE = LOX_Region_List, $
      UNAME='WID_DLIST_REGION' ,XOFFSET=170, TITLE='Region ')
; row = widget_base(base_lox,/row)
  LOX_Image_List =['']
  LOX_Im = 0
  WID_DLIST_IMAGE = Widget_Droplist(row,  $
      /DYNAMIC_RESIZE, VALUE = LOX_Image_List, $
      UNAME='WID_DLIST_IMAGE' ,XOFFSET=290,  TITLE='Image Energy ')

; ----------- start display without active displays / controls
  widget_control, WID_DLIST_CHANNEL, sensitive = 0
  widget_control, WID_DLIST_REGION, sensitive = 0
  widget_control, WID_DLIST_IMAGE, sensitive = 0
  widget_control, TEXT_LOX_TYPE, sensitive = 0
  widget_control, BUT_LOX_DISPLAY_HDR, sensitive = 0
  widget_control, BUT_LOX_Stack_Map, sensitive = 0
  widget_control, BUT_LOX_Stack_Image, sensitive = 0
  Widget_Control, /REALIZE, BASE_LOX

; ------ set last path and file name if they exist
; ------ use keyowrd file if provided
if keyword_set(file) then begin		; assume file contains path
	t = ax_name(file)
	lox_path = t(0)
  	lox_lastfile = t(1)
endif
; ----------- load last path information --------------------------------
t1 = size(lox_path,/type)
if t1(0) NE 0 then begin
	widget_control, TEXT_LOX_PATH, set_value = lox_path
; load last file information
	t2 = size(lox_lastfile,/type)
	if t2(0) NE 0 then begin
		widget_control, TEXT_LOX_FNAME, set_value = lox_lastfile

;------------- get details of this file if it is valid-------------
		fname = lox_path + lox_lastfile + '.lox'
		t = findfile(fname(0), count=t_count)
		if t_count NE 0 then begin
; read file and update type
			axis_log, 'Reading last Lox file ' + fname
			widget_control, /hourglass
			lox = read_lox(file=fname, /type, group = lox_ID)  ;, /verbose)
	; Assume the file is actually a LOX type file and read its header
	;update the type
			WIDGET_CONTROL,TEXT_LOX_TYPE, Sensitive=1, Set_value = lox.Type
	; ------------- update region information
	; currently there are no names associated with regions; can only set number of elements
			if lox.Type EQ 'stack' or lox.Type EQ 'spectra' then begin
				lox_data = read_lox_spectra(file = lox.filename, $
				       /multi, /silent, group = lox_ID)  ;, /verbose)
				nt = size(lox_Data)
				lox.n_regions = nt(1)-1
	;			print, '# of regions ', lox.n_regions
	;			print, Lox.labels
				LOX_Channel_List = Lox.labels
				LOX_Channel_List = [LOX_Channel_List, 'all channels']
				widget_control, WID_DLIST_CHANNEL, Set_Value=LOX_Channel_List, sensitive = 1
			endif
			n_regions = lox.n_regions
			if n_regions GT 1 then LOX_region_List = strarr(n_regions+1) else LOX_region_List = ' '
			for i = 0, n_regions-1 do LOX_Region_List(i) = 'Region '+ strtrim(string(i+1),2)
			if n_regions GT 1 then LOX_Region_List(n_regions) = 'all regions'
			WIDGET_CONTROL,WID_DLIST_REGION, Set_VALUE = LOX_Region_List, Sensitive=1
			widget_control, WID_DLIST_REGION, sensitive = 1
	; ---------- update image number information (for NEXAFS image sequences)
			if lox.Type EQ 'stack' then begin
				LOX_Image_List = strtrim(string(lox.image_list, format='(F7.2)'),2)
				widget_control, WID_DLIST_IMAGE, Set_Value=LOX_Image_List, sensitive = 1
				WIDGET_CONTROL,TEXT_LOX_DARK, Sensitive=1, Set_value = string(lox.dark, format='(f5.1)')
				WIDGET_CONTROL,TEXT_LOX_FoV, Sensitive=1, Set_value = string(lox.foV, format='(f5.1)')
				widget_control, BUT_LOX_Stack_Map, sensitive = 1
				widget_control, BUT_LOX_Stack_Image, sensitive = 1
				widget_control, BUT_LOX_Stack_Spectrum, sensitive = 1
				WIDGET_CONTROL,TEXT_LOX_ENERGY, Sensitive=0
			endif else begin
				widget_control, WID_DLIST_IMAGE, sensitive = 0
				widget_control, BUT_LOX_Stack_Map, sensitive = 0
				widget_control, BUT_LOX_Stack_Image, sensitive = 0
				widget_control, BUT_LOX_Stack_Spectrum, sensitive = 0
			endelse

	; --------- update energy, background, FoV  if it is an image
			if lox.Type EQ 'image' then begin
				WIDGET_CONTROL,TEXT_LOX_ENERGY, Sensitive=1, Set_value = string(lox.energy, format='(f8.3)')
				WIDGET_CONTROL,TEXT_LOX_DARK, Sensitive=1, Set_value = string(lox.dark, format='(f5.1)')
				WIDGET_CONTROL,TEXT_LOX_FoV, Sensitive=1, Set_value = string(lox.foV, format='(f5.1)')
	  			widget_control, WID_DLIST_CHANNEL, sensitive = 0
			endif

	; -------- deactivate regions for spectra ------------
			if lox.Type EQ 'spectra' then WIDGET_CONTROL,WID_DLIST_REGION,  Sensitive=0

	; ----------- activate 'Show parameters' button
			 widget_control, BUT_LOX_DISPLAY_HDR, sensitive = 1
		endif else axis_log, 'last LOX_Filename not defined'
	endif else  axis_log, 'last LOX_Path or LOX_Filename not valid'
endif else axis_log, 'last LOX_Path not defined'


XManager, 'BASE_LOX', BASE_LOX

end

;-----------------------------------------------------------------
; ****************************************************************
;-----------------------------------------------------------------

function ax_lox, file = file, verbose=verbose, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_
on_error,2
@axis_com
@lox_com

lox = ' '
if keyword_set(file) then begin
	BASE_LOX, GROUP_LEADER=wGroup, file=file, _EXTRA=_VWBExtra_
endif else begin
	BASE_LOX, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_
endelse

return, lox
end
